/*
 * MIT License
 *
 * Copyright (c) 2017 Yuriy Budiyev [yuriy.budiyev@yandex.ru]
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.budiyev.android.imageloader;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.drawable.BitmapDrawable;
import android.support.annotation.MainThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import android.view.View;

import java.lang.ref.WeakReference;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;

/**
 * Load image action for {@link ImageLoader}
 */
final class LoadImageAction<T> {
    private final ImageSource<T> mImageSource;
    private final WeakReference<View> mViewReference;
    private final ImageLoader<T> mImageLoader;
    private final Lock mPauseLoadingLock;
    private final Condition mPauseLoadingCondition;
    private final Callbacks<T> mCallbacks;
    private volatile Future<?> mFuture;
    private volatile boolean mCancelled;

    public LoadImageAction(@NonNull ImageSource<T> source, @NonNull View view,
            @Nullable Callbacks<T> callback, @NonNull ImageLoader<T> loader, @NonNull Lock lock,
            @NonNull Condition condition) {
        mImageSource = source;
        mViewReference = new WeakReference<>(view);
        mImageLoader = loader;
        mPauseLoadingLock = lock;
        mPauseLoadingCondition = condition;
        mCallbacks = callback;
    }

    public void execute() {
        if (mCancelled) {
            return;
        }
        mFuture = InternalUtils.getImageLoaderExecutor().submit(new LoadImageTask());
    }

    public void cancel() {
        mCancelled = true;
        Future<?> future = mFuture;
        if (future != null) {
            future.cancel(false);
        }
    }

    @NonNull
    public ImageSource<T> getImageSource() {
        return mImageSource;
    }

    @Nullable
    @MainThread
    public View getAttachedView() {
        View imageView = mViewReference.get();
        LoadImageAction<?> loadImageAction = mImageLoader.getLoadImageAction(imageView);
        if (loadImageAction == this) {
            return imageView;
        } else {
            return null;
        }
    }

    public boolean isCancelled() {
        return mCancelled;
    }

    @WorkerThread
    private void loadImage() {
        while (!mCancelled && mImageLoader.isLoadingPaused() && !mImageLoader.isExitTasksEarly()) {
            mPauseLoadingLock.lock();
            try {
                mPauseLoadingCondition.await();
            } catch (InterruptedException e) {
                return;
            } finally {
                mPauseLoadingLock.unlock();
            }
        }
        if (mCancelled || mImageLoader.isExitTasksEarly()) {
            return;
        }
        Bitmap image = null;
        StorageImageCache storageImageCache = mImageLoader.getStorageImageCache();
        String key = mImageSource.getKey();
        T data = mImageSource.getData();
        if (storageImageCache != null) {
            image = storageImageCache.get(key);
        }
        Context context = mImageLoader.getContext();
        if (image == null) {
            BitmapLoader<T> bitmapLoader = mImageLoader.getBitmapLoader();
            if (bitmapLoader != null) {
                try {
                    image = bitmapLoader.load(context, data);
                } catch (Throwable error) {
                    Callbacks.notifyError(mCallbacks, mImageSource, error);
                    return;
                }
                if (image == null) {
                    Callbacks.notifyError(mCallbacks, mImageSource,
                            new NullPointerException("BitmapLoader returned null"));
                    return;
                }
            }
            if (image != null) {
                Callbacks.notifyLoaded(mCallbacks, mImageSource, image);
                if (storageImageCache != null) {
                    storageImageCache.put(key, image);
                }
            }
        }
        if (image == null) {
            return;
        }
        BitmapDrawable drawable = new RecyclingBitmapDrawable(context.getResources(), image);
        MemoryImageCache memoryImageCache = mImageLoader.getMemoryImageCache();
        if (memoryImageCache != null) {
            memoryImageCache.put(key, drawable);
        }
        if (mCancelled || mImageLoader.isExitTasksEarly()) {
            return;
        }
        InternalUtils
                .runOnMainThread(new SetImageAction<>(drawable, mImageLoader, mCallbacks, this));
    }

    private final class LoadImageTask implements Callable<Void> {
        @Override
        public Void call() throws Exception {
            loadImage();
            mFuture = null;
            return null;
        }
    }
}
