/*
MIT License

Copyright (c) 2017 Yuriy Budiyev [yuriy.budiyev@yandex.ru]

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
 */
package com.budiyev.android.imageloader;

import android.content.ContentResolver;
import android.content.Context;
import android.content.res.Resources;
import android.net.Uri;
import android.support.annotation.AnyRes;
import android.support.annotation.MainThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.widget.ImageView;

import java.io.File;

public final class ImageLoadRequest {
    private final ImageLoader<Uri> mLoader;
    private volatile ImageSource<Uri> mSource;
    private volatile ImageView mView;
    private volatile ImageLoadCallback<Uri> mCallback;

    ImageLoadRequest(@NonNull Context context) {
        mLoader = ImageLoaderImpl.getInstance(context);
    }

    /**
     * Source form which image should be loaded
     */
    @NonNull
    public ImageLoadRequest from(@NonNull Uri source) {
        mSource = new ImageSourceImpl<>(source);
        return this;
    }

    /**
     * Source form which image should be loaded
     */
    @NonNull
    public ImageLoadRequest from(@NonNull File source) {
        mSource = new ImageSourceImpl<>(Uri.fromFile(source));
        return this;
    }

    /**
     * Source form which image should be loaded
     */
    @NonNull
    public ImageLoadRequest from(@AnyRes int resourceId) {
        Resources resources = mLoader.getContext().getResources();
        mSource = new ImageSourceImpl<>(Uri.parse(ContentResolver.SCHEME_ANDROID_RESOURCE + "://" +
                resources.getResourcePackageName(resourceId) + "/" +
                resources.getResourceTypeName(resourceId) + "/" +
                resources.getResourceEntryName(resourceId)));
        return this;
    }

    /**
     * View into which loaded image should be displayed
     */
    @NonNull
    public ImageLoadRequest into(@NonNull ImageView view) {
        mView = view;
        return this;
    }

    /**
     * Callback of loading process
     */
    @NonNull
    public ImageLoadRequest callback(@Nullable ImageLoadCallback<Uri> callback) {
        mCallback = callback;
        return this;
    }

    /**
     * Load image
     */
    @MainThread
    public void load() {
        mLoader.loadImage(mSource, mView, mCallback);
    }
}
