/*
MIT License

Copyright (c) 2017 Yuriy Budiyev [yuriy.budiyev@yandex.ru]

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
 */
package com.budiyev.android.imageloader;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.support.annotation.FloatRange;
import android.support.annotation.MainThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.widget.ImageView;

import java.io.File;
import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * {@link ImageLoader} is a universal tool for loading bitmaps efficiently in Android, which
 * provides automatic memory and storage caching. {@link ImageLoader} is usable without caches,
 * with one of them, or both (without caches, caching is not available). Also, {@link ImageLoader}
 * is usable without {@link BitmapLoader} (loading new bitmaps is not available in this case).
 */
public class ImageLoader<T> {
    private static final String HASH_ALGORITHM_SHA256 = "SHA-256";
    private static final String DEFAULT_STORAGE_CACHE_DIRECTORY = "image_loader_cache";
    private static final Bitmap.CompressFormat DEFAULT_COMPRESS_FORMAT = Bitmap.CompressFormat.JPEG;
    private static final double DEFAULT_STORAGE_FRACTION = 0.1d;
    private static final float DEFAULT_MEMORY_FRACTION = 0.25f;
    private static final int DEFAULT_COMPRESS_QUALITY = 85;
    private final Lock mPauseLoadingLock = new ReentrantLock();
    private final Condition mPauseLoadingCondition = mPauseLoadingLock.newCondition();
    private final Context mContext;
    private volatile BitmapLoader<T> mBitmapLoader;
    private volatile MemoryImageCache mMemoryImageCache;
    private volatile StorageImageCache mStorageImageCache;
    private volatile PlaceholderProvider<T> mPlaceholderProvider;
    private volatile boolean mImageFadeIn = true;
    private volatile boolean mExitTasksEarly;
    private volatile boolean mLoadingPaused;
    private volatile int mImageFadeInTime = 200;

    /**
     * ImageLoader without any cache or bitmap loader
     * <br>
     * Use <b>application context</b> to avoid memory leaks.
     *
     * @param context Context
     * @see Context#getApplicationContext()
     */
    public ImageLoader(@NonNull Context context) {
        this(context, null, null, null);
    }

    /**
     * ImageLoader with default memory and storage caches
     * <br>
     * Use <b>application context</b> to avoid memory leaks.
     *
     * @param context      Context
     * @param bitmapLoader Bitmap loader
     * @see Context#getApplicationContext()
     */
    public ImageLoader(@NonNull Context context, @Nullable BitmapLoader<T> bitmapLoader) {
        this(context, bitmapLoader, newMemoryImageCache(), newStorageImageCache(context));
    }

    /**
     * ImageLoader with specified bitmap loader, memory image cache and storage image cache
     * <br>
     * Use <b>application context</b> to avoid memory leaks.
     *
     * @param context           Context
     * @param bitmapLoader      Bitmap loader
     * @param memoryImageCache  Memory image cache
     * @param storageImageCache Storage image cache
     * @see Context#getApplicationContext()
     */
    public ImageLoader(@NonNull Context context, @Nullable BitmapLoader<T> bitmapLoader,
            @Nullable MemoryImageCache memoryImageCache,
            @Nullable StorageImageCache storageImageCache) {
        mContext = Utils.requireNonNull(context);
        mBitmapLoader = bitmapLoader;
        mMemoryImageCache = memoryImageCache;
        mStorageImageCache = storageImageCache;
    }

    /**
     * Context of this {@link ImageLoader} instance
     */
    @NonNull
    protected Context getContext() {
        return mContext;
    }

    /**
     * Load image to imageView from imageSource
     *
     * @param source   Image source
     * @param view     Image view
     * @param callback Optional callback
     */
    @MainThread
    public void loadImage(@NonNull ImageSource<T> source, @NonNull ImageView view,
            @Nullable ImageLoadCallback<T> callback) {
        MemoryImageCache memoryImageCache = getMemoryImageCache();
        BitmapDrawable drawable = null;
        if (memoryImageCache != null) {
            drawable = memoryImageCache.get(source.getKey());
        }
        Bitmap image;
        if (drawable != null && (image = drawable.getBitmap()) != null) {
            T data = source.getData();
            if (callback != null) {
                callback.onLoaded(data, image, true, false);
            }
            view.setImageDrawable(drawable);
            if (callback != null) {
                callback.onDisplayed(data, image, view);
            }
        } else if (cancelPotentialWork(source, view)) {
            LoadImageAction<T> loadAction =
                    new LoadImageAction<>(source, view, callback, this, mPauseLoadingLock,
                            mPauseLoadingCondition);
            AsyncBitmapDrawable asyncBitmapDrawable =
                    new AsyncBitmapDrawable(getContext().getResources(),
                            getPlaceholderImage(source), loadAction);
            view.setImageDrawable(asyncBitmapDrawable);
            loadAction.execute();
        }
    }

    /**
     * Load image to imageView from imageSource
     *
     * @param source Image source
     * @param view   Image view
     */
    @MainThread
    public void loadImage(@NonNull ImageSource<T> source, @NonNull ImageView view) {
        loadImage(source, view, null);
    }

    /**
     * Delete cached image for specified {@link ImageSource}
     */
    public void invalidate(@NonNull ImageSource<T> source) {
        String key = source.getKey();
        MemoryImageCache memoryImageCache = getMemoryImageCache();
        if (memoryImageCache != null) {
            memoryImageCache.remove(key);
        }
        StorageImageCache storageImageCache = getStorageImageCache();
        if (storageImageCache != null) {
            storageImageCache.remove(key);
        }
    }

    public void setPlaceholderProvider(@Nullable PlaceholderProvider<T> provider) {
        mPlaceholderProvider = provider;
    }

    @Nullable
    public PlaceholderProvider<T> getPlaceholderProvider() {
        return mPlaceholderProvider;
    }

    @Nullable
    public Bitmap getPlaceholderImage(@NonNull ImageSource<T> source) {
        PlaceholderProvider<T> provider = getPlaceholderProvider();
        if (provider == null) {
            return null;
        } else {
            return provider.get(getContext(), source.getData());
        }
    }

    /**
     * Placeholder image
     * <br>
     * Displayed while image is loading
     *
     * @param image Image bitmap
     */
    public void setPlaceholderImage(@Nullable Bitmap image) {
        mPlaceholderProvider = new PlaceholderProviderImpl<>(image);
    }

    /**
     * Convenience method to set placeholder image from resource with image data.
     * <br>
     * Displayed while image is loading
     *
     * @param resourceId Image resource identifier
     */
    @WorkerThread
    public void setPlaceholderImage(int resourceId) {
        mPlaceholderProvider = new PlaceholderProviderImpl<>(
                BitmapFactory.decodeResource(getContext().getResources(), resourceId));
    }

    /**
     * Current {@link BitmapLoader} implementation
     * <br>
     * {@link BitmapLoader} is used for loading new bitmaps
     * if there are no cached images with the same key
     */
    @Nullable
    public BitmapLoader<T> getBitmapLoader() {
        return mBitmapLoader;
    }

    /**
     * Current {@link BitmapLoader} implementation
     * <br>
     * {@link BitmapLoader} is used for loading new bitmaps
     * if there are no cached images with the same key
     */
    public void setBitmapLoader(@Nullable BitmapLoader<T> loader) {
        mBitmapLoader = loader;
    }

    /**
     * Current {@link MemoryImageCache} implementation
     * <br>
     * {@link MemoryImageCache} is used for caching images in memory
     */
    @Nullable
    public MemoryImageCache getMemoryImageCache() {
        return mMemoryImageCache;
    }

    /**
     * Current {@link MemoryImageCache} implementation
     * <br>
     * {@link MemoryImageCache} is used for caching images in memory
     */
    public void setMemoryImageCache(@Nullable MemoryImageCache cache) {
        mMemoryImageCache = cache;
    }

    /**
     * Current {@link StorageImageCache} implementation
     * <br>
     * {@link StorageImageCache} is used for caching images in storage
     */
    @Nullable
    public StorageImageCache getStorageImageCache() {
        return mStorageImageCache;
    }

    /**
     * Current {@link StorageImageCache} implementation
     * <br>
     * {@link StorageImageCache} is used for caching images in storage
     */
    public void setStorageImageCache(@Nullable StorageImageCache cache) {
        mStorageImageCache = cache;
    }

    /**
     * Whether to use fade effect to display images
     *
     * @see #getImageFadeInTime()
     * @see #setImageFadeInTime(int)
     */
    public boolean isImageFadeIn() {
        return mImageFadeIn;
    }

    /**
     * Whether to use fade effect to display images
     *
     * @see #getImageFadeInTime()
     * @see #setImageFadeInTime(int)
     */
    public void setImageFadeIn(boolean fadeIn) {
        mImageFadeIn = fadeIn;
    }

    /**
     * Check if image loading is paused
     *
     * @see #setPauseLoading(boolean)
     */
    public boolean isLoadingPaused() {
        return mLoadingPaused;
    }

    /**
     * Whether to pause image loading. If this method is invoked with {@code true} parameter,
     * all loading actions will be paused until it will be invoked with {@code false}.
     */
    public void setPauseLoading(boolean pause) {
        mPauseLoadingLock.lock();
        try {
            mLoadingPaused = pause;
            if (!pause) {
                mPauseLoadingCondition.signalAll();
            }
        } finally {
            mPauseLoadingLock.unlock();
        }
    }

    /**
     * Whether to exit all image loading tasks before start of image loading
     */
    public boolean isExitTasksEarly() {
        return mExitTasksEarly;
    }

    /**
     * Whether to exit all image loading tasks before start of image loading
     */
    public void setExitTasksEarly(boolean exit) {
        mExitTasksEarly = exit;
        if (exit) {
            mPauseLoadingLock.lock();
            try {
                mPauseLoadingCondition.signalAll();
            } finally {
                mPauseLoadingLock.unlock();
            }
        }
    }

    /**
     * Fade effect duration if that effect is enabled
     *
     * @see #isImageFadeIn()
     * @see #setImageFadeIn(boolean)
     */
    public int getImageFadeInTime() {
        return mImageFadeInTime;
    }

    /**
     * Fade effect duration if that effect is enabled
     *
     * @see #isImageFadeIn()
     * @see #setImageFadeIn(boolean)
     */
    public void setImageFadeInTime(int time) {
        mImageFadeInTime = time;
    }

    /**
     * Clear all caches
     *
     * @see #getMemoryImageCache()
     * @see #setMemoryImageCache(MemoryImageCache)
     * @see #getStorageImageCache()
     * @see #setStorageImageCache(StorageImageCache)
     */
    public void clearCache() {
        MemoryImageCache memoryImageCache = getMemoryImageCache();
        if (memoryImageCache != null) {
            memoryImageCache.clear();
        }
        StorageImageCache storageImageCache = getStorageImageCache();
        if (storageImageCache != null) {
            storageImageCache.clear();
        }
    }

    @Nullable
    @MainThread
    public static LoadImageAction<?> getLoadImageAction(@Nullable ImageView view) {
        if (view != null) {
            Drawable drawable = view.getDrawable();
            if (drawable instanceof AsyncBitmapDrawable) {
                return ((AsyncBitmapDrawable) drawable).getLoadImageAction();
            }
        }
        return null;
    }

    @MainThread
    public static void cancelWork(@Nullable ImageView view) {
        LoadImageAction<?> loadImageAction = getLoadImageAction(view);
        if (loadImageAction != null) {
            loadImageAction.cancel();
        }
    }

    @MainThread
    public static boolean cancelPotentialWork(@NonNull ImageSource<?> source,
            @Nullable ImageView view) {
        LoadImageAction<?> loadImageAction = getLoadImageAction(view);
        if (loadImageAction != null) {
            if (!Utils.equals(loadImageAction.getImageSource().getKey(), source.getKey())) {
                loadImageAction.cancel();
            } else {
                return false;
            }
        }
        return true;
    }

    /**
     * Fraction of maximum number of bytes heap can expand to
     *
     * @param fraction Fraction
     * @return Number of bytes
     */
    public static int getMaxMemoryFraction(@FloatRange(from = 0.1, to = 0.8) float fraction) {
        if (fraction < 0.1F || fraction > 0.8F) {
            throw new IllegalArgumentException(
                    "Argument \"fraction\" must be between 0.1 and 0.8 (inclusive)");
        }
        return Math.round(fraction * Runtime.getRuntime().maxMemory());
    }

    /**
     * Fraction of available storage space in specified path
     *
     * @param path     Path
     * @param fraction Fraction
     * @return Number of bytes
     */
    public static long getAvailableStorageFraction(@NonNull File path,
            @FloatRange(from = 0.01, to = 1.0) double fraction) {
        if (fraction < 0.01D || fraction > 1.0D) {
            throw new IllegalArgumentException(
                    "Argument \"fraction\" must be between 0.01 and 1.0 (inclusive)");
        }
        return Math.round(Utils.getAvailableBytes(path) * fraction);
    }

    /**
     * Fraction of total storage space in specified path
     *
     * @param path     Path
     * @param fraction Fraction
     * @return Number of bytes
     */
    public static long getTotalStorageFraction(@NonNull File path,
            @FloatRange(from = 0.01, to = 1.0) double fraction) {
        if (fraction < 0.01D || fraction > 1.0D) {
            throw new IllegalArgumentException(
                    "Argument \"fraction\" must be between 0.01 and 1.0 (inclusive)");
        }
        return Math.round(Utils.getTotalBytes(path) * fraction);
    }

    /**
     * Calculate sample size for required size from source size
     * Sample size is the number of pixels in either dimension that
     * correspond to a single pixel
     *
     * @param sourceWidth               Source width
     * @param sourceHeight              Source height
     * @param requiredWidth             Required width
     * @param requiredHeight            Required height
     * @param ignoreTotalNumberOfPixels Ignore total number of pixels
     *                                  (requiredWidth * requiredHeight)
     * @return Sample size
     */
    public static int calculateSampleSize(int sourceWidth, int sourceHeight, int requiredWidth,
            int requiredHeight, boolean ignoreTotalNumberOfPixels) {
        int sampleSize = 1;
        if (sourceWidth > requiredWidth || sourceHeight > requiredHeight) {
            int halfWidth = sourceWidth / 2;
            int halfHeight = sourceHeight / 2;
            while ((halfWidth / sampleSize) > requiredWidth &&
                    (halfHeight / sampleSize) > requiredHeight) {
                sampleSize *= 2;
            }
            if (ignoreTotalNumberOfPixels) {
                return sampleSize;
            }
            long totalPixels = (sourceWidth * sourceHeight) / (sampleSize * sampleSize);
            long totalRequiredPixels = requiredWidth * requiredHeight;
            while (totalPixels > totalRequiredPixels) {
                sampleSize *= 2;
                totalPixels /= 4L;
            }
        }
        return sampleSize;
    }

    /**
     * Load sampled bitmap from uri
     *
     * @param context                   Context
     * @param uri                       Uri
     * @param requiredWidth             Required width
     * @param requiredHeight            Required height
     * @param ignoreTotalNumberOfPixels Ignore total number of pixels
     *                                  (requiredWidth * requiredHeight)
     * @return Loaded bitmap or {@code null}
     */
    @Nullable
    @WorkerThread
    public static Bitmap loadSampledBitmapFromUri(@NonNull Context context, @NonNull Uri uri,
            int requiredWidth, int requiredHeight, boolean ignoreTotalNumberOfPixels) {
        BitmapFactory.Options options = null;
        if (requiredWidth != Integer.MAX_VALUE || requiredHeight != Integer.MAX_VALUE) {
            options = new BitmapFactory.Options();
            options.inJustDecodeBounds = true;
            InputStream inputStream = null;
            try {
                inputStream = Utils.getDataStreamFromUri(context, uri);
                BitmapFactory.decodeStream(inputStream, null, options);
            } catch (IOException e) {
                return null;
            } finally {
                Utils.close(inputStream);
            }
            options.inJustDecodeBounds = false;
            options.inSampleSize =
                    calculateSampleSize(options.outWidth, options.outHeight, requiredWidth,
                            requiredHeight, ignoreTotalNumberOfPixels);
        }
        InputStream inputStream = null;
        try {
            inputStream = Utils.getDataStreamFromUri(context, uri);
            return BitmapFactory.decodeStream(inputStream, null, options);
        } catch (IOException e) {
            return null;
        } finally {
            Utils.close(inputStream);
        }
    }

    /**
     * Loading sampled bitmap from file
     *
     * @param file                      File
     * @param requiredWidth             Required width
     * @param requiredHeight            Required height
     * @param ignoreTotalNumberOfPixels Ignore total number of pixels
     *                                  (requiredWidth * requiredHeight)
     * @return Loaded bitmap or {@code null}
     */
    @Nullable
    @WorkerThread
    public static Bitmap loadSampledBitmapFromFile(@NonNull File file, int requiredWidth,
            int requiredHeight, boolean ignoreTotalNumberOfPixels) {
        BitmapFactory.Options options = null;
        if (requiredWidth != Integer.MAX_VALUE || requiredHeight != Integer.MAX_VALUE) {
            options = new BitmapFactory.Options();
            options.inJustDecodeBounds = true;
            InputStream inputStream = null;
            try {
                inputStream = new FileInputStream(file);
                BitmapFactory.decodeStream(inputStream, null, options);
            } catch (IOException e) {
                return null;
            } finally {
                Utils.close(inputStream);
            }
            options.inJustDecodeBounds = false;
            options.inSampleSize =
                    calculateSampleSize(options.outWidth, options.outHeight, requiredWidth,
                            requiredHeight, ignoreTotalNumberOfPixels);
        }
        InputStream inputStream = null;
        try {
            inputStream = new FileInputStream(file);
            return BitmapFactory.decodeStream(inputStream, null, options);
        } catch (IOException e) {
            return null;
        } finally {
            Utils.close(inputStream);
        }
    }

    /**
     * Load sampled bitmap from file descriptor
     *
     * @param fileDescriptor            File descriptor
     * @param requiredWidth             Required width
     * @param requiredHeight            Required height
     * @param ignoreTotalNumberOfPixels Ignore total number of pixels
     *                                  (requiredWidth * requiredHeight)
     * @return Loaded bitmap or {@code null}
     */
    @Nullable
    @WorkerThread
    public static Bitmap loadSampledBitmapFromFileDescriptor(@NonNull FileDescriptor fileDescriptor,
            int requiredWidth, int requiredHeight, boolean ignoreTotalNumberOfPixels) {
        BitmapFactory.Options options = null;
        if (requiredWidth != Integer.MAX_VALUE || requiredHeight != Integer.MAX_VALUE) {
            options = new BitmapFactory.Options();
            options.inJustDecodeBounds = true;
            InputStream inputStream = new FileInputStream(fileDescriptor);
            BitmapFactory.decodeStream(inputStream, null, options);
            Utils.close(inputStream);
            options.inJustDecodeBounds = false;
            options.inSampleSize =
                    calculateSampleSize(options.outWidth, options.outHeight, requiredWidth,
                            requiredHeight, ignoreTotalNumberOfPixels);
        }
        InputStream inputStream = new FileInputStream(fileDescriptor);
        Bitmap bitmap = BitmapFactory.decodeStream(inputStream, null, options);
        Utils.close(inputStream);
        return bitmap;
    }

    /**
     * Load sampled bitmap from resource
     *
     * @param resources                 Resources
     * @param resourceId                Resource id
     * @param requiredWidth             Required width
     * @param requiredHeight            Required height
     * @param ignoreTotalNumberOfPixels Ignore total number of pixels
     *                                  (requiredWidth * requiredHeight)
     * @return Loaded bitmap or {@code null}
     */
    @Nullable
    @WorkerThread
    public static Bitmap loadSampledBitmapFromResource(@NonNull Resources resources, int resourceId,
            int requiredWidth, int requiredHeight, boolean ignoreTotalNumberOfPixels) {
        BitmapFactory.Options options = new BitmapFactory.Options();
        TypedValue typedValue = new TypedValue();
        options.inJustDecodeBounds = true;
        options.inTargetDensity = resources.getDisplayMetrics().densityDpi;
        InputStream inputStream = resources.openRawResource(resourceId, typedValue);
        if (typedValue.density == TypedValue.DENSITY_DEFAULT) {
            options.inDensity = DisplayMetrics.DENSITY_DEFAULT;
        } else if (typedValue.density != TypedValue.DENSITY_NONE) {
            options.inDensity = typedValue.density;
        }
        BitmapFactory.decodeStream(inputStream, null, options);
        Utils.close(inputStream);
        options.inJustDecodeBounds = false;
        if (requiredWidth != Integer.MAX_VALUE || requiredHeight != Integer.MAX_VALUE) {
            options.inSampleSize =
                    calculateSampleSize(options.outWidth, options.outHeight, requiredWidth,
                            requiredHeight, ignoreTotalNumberOfPixels);
        }
        inputStream = resources.openRawResource(resourceId, typedValue);
        Bitmap bitmap = BitmapFactory.decodeStream(inputStream, null, options);
        Utils.close(inputStream);
        return bitmap;

    }

    /**
     * Load sampled bitmap from byte array
     *
     * @param byteArray                 Byte array
     * @param requiredWidth             Required width
     * @param requiredHeight            Required height
     * @param ignoreTotalNumberOfPixels Ignore total number of pixels
     *                                  (requiredWidth * requiredHeight)
     * @return Loaded bitmap or {@code null}
     */
    @Nullable
    @WorkerThread
    public static Bitmap loadSampledBitmapFromByteArray(@NonNull byte[] byteArray,
            int requiredWidth, int requiredHeight, boolean ignoreTotalNumberOfPixels) {
        BitmapFactory.Options options = null;
        if (requiredWidth != Integer.MAX_VALUE || requiredHeight != Integer.MAX_VALUE) {
            options = new BitmapFactory.Options();
            options.inJustDecodeBounds = true;
            BitmapFactory.decodeByteArray(byteArray, 0, byteArray.length, options);
            options.inSampleSize =
                    calculateSampleSize(options.outWidth, options.outHeight, requiredWidth,
                            requiredHeight, ignoreTotalNumberOfPixels);
            options.inJustDecodeBounds = false;
        }
        return BitmapFactory.decodeByteArray(byteArray, 0, byteArray.length, options);
    }

    /**
     * Create new common bitmap loader for uris
     *
     * @return bitmap loader
     */
    @NonNull
    public static BitmapLoader<Uri> newUriBitmapLoader() {
        return new BitmapLoader<Uri>() {
            @Nullable
            @Override
            public Bitmap load(@NonNull Context context, Uri data) throws Exception {
                return loadSampledBitmapFromUri(context, data, Integer.MAX_VALUE, Integer.MAX_VALUE,
                        true);
            }
        };
    }

    /**
     * Create new common bitmap loader for files
     *
     * @return bitmap loader
     */
    @NonNull
    public static BitmapLoader<File> newFileBitmapLoader() {
        return new BitmapLoader<File>() {
            @Nullable
            @Override
            public Bitmap load(@NonNull Context context, File data) throws Exception {
                return loadSampledBitmapFromFile(data, Integer.MAX_VALUE, Integer.MAX_VALUE, true);
            }
        };
    }

    /**
     * Create new common bitmap loader for file descriptors
     *
     * @return bitmap loader
     */
    @NonNull
    public static BitmapLoader<FileDescriptor> newFileDescriptorBitmapLoader() {
        return new BitmapLoader<FileDescriptor>() {
            @Nullable
            @Override
            public Bitmap load(@NonNull Context context, FileDescriptor data) throws Exception {
                return loadSampledBitmapFromFileDescriptor(data, Integer.MAX_VALUE,
                        Integer.MAX_VALUE, true);
            }
        };
    }

    /**
     * Create new common bitmap loader for resources
     *
     * @return bitmap loader
     */
    @NonNull
    public static BitmapLoader<Integer> newResourceBitmapLoader() {
        return new BitmapLoader<Integer>() {
            @Nullable
            @Override
            public Bitmap load(@NonNull Context context, Integer data) throws Exception {
                return loadSampledBitmapFromResource(context.getResources(), data,
                        Integer.MAX_VALUE, Integer.MAX_VALUE, true);
            }
        };
    }

    /**
     * Create new common bitmap loader for byte arrays
     *
     * @return bitmap loader
     */
    @NonNull
    public static BitmapLoader<byte[]> newByteArrayBitmapLoader() {
        return new BitmapLoader<byte[]>() {
            @Nullable
            @Override
            public Bitmap load(@NonNull Context context, byte[] data) throws Exception {
                return loadSampledBitmapFromByteArray(data, Integer.MAX_VALUE, Integer.MAX_VALUE,
                        true);
            }
        };
    }

    /**
     * Create new common image source that is usable in most cases
     * <br>
     * SHA-256 hash of {@link String#valueOf(Object)} of {@code data} will be used as a key.
     *
     * @param data Source data
     * @return Image source
     */
    @NonNull
    public static <T> ImageSource<T> newImageSource(@NonNull T data) {
        return new ImageSourceImpl<>(data);
    }

    /**
     * Generate SHA-256 hash string with {@link Character#MAX_RADIX} radix
     * for specified {@link String}; usable for keys of {@link ImageSource} implementations
     *
     * @param string Source string
     * @return SHA-256 hash string
     * @see ImageSource#getKey()
     */
    @NonNull
    public static String generateSHA256(@NonNull String string) {
        try {
            MessageDigest messageDigest = MessageDigest.getInstance(HASH_ALGORITHM_SHA256);
            messageDigest.update(string.getBytes());
            return new BigInteger(1, messageDigest.digest()).toString(Character.MAX_RADIX);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Create memory image cache with specified maximum size
     *
     * @param maxSize Maximum size in bytes
     * @return Memory image cache
     */
    @NonNull
    public static MemoryImageCache newMemoryImageCache(int maxSize) {
        return new MemoryImageCacheImpl(maxSize);
    }

    /**
     * Create memory image cache with maximum size 25% of
     * total available memory fraction
     *
     * @return Memory image cache
     */
    @NonNull
    public static MemoryImageCache newMemoryImageCache() {
        return newMemoryImageCache(getMaxMemoryFraction(DEFAULT_MEMORY_FRACTION));
    }

    /**
     * Create storage image cache with specified parameters
     *
     * @param directory       Directory
     * @param maxSize         Maximum size
     * @param compressFormat  Compress format
     * @param compressQuality Compress quality
     * @return Storage image cache
     */
    @NonNull
    @SuppressWarnings("ResultOfMethodCallIgnored")
    public static StorageImageCache newStorageImageCache(@NonNull File directory, long maxSize,
            @NonNull Bitmap.CompressFormat compressFormat, int compressQuality) {
        if (!directory.exists()) {
            directory.mkdirs();
        }
        return new StorageImageCacheImpl(directory, maxSize, compressFormat, compressQuality);
    }

    /**
     * Create storage image cache in specified directory with maximum size 10%
     * of total storage size
     *
     * @param directory Cache directory
     * @return Storage image cache
     */
    @NonNull
    @SuppressWarnings("ResultOfMethodCallIgnored")
    public static StorageImageCache newStorageImageCache(@NonNull File directory) {
        if (!directory.exists()) {
            directory.mkdirs();
        }
        return newStorageImageCache(directory,
                getTotalStorageFraction(directory, DEFAULT_STORAGE_FRACTION),
                DEFAULT_COMPRESS_FORMAT, DEFAULT_COMPRESS_QUALITY);
    }

    /**
     * Create storage image cache in default application cache directory with maximum size 10%
     * of total storage size
     *
     * @param context Context
     * @return Storage image cache
     */
    @NonNull
    public static StorageImageCache newStorageImageCache(@NonNull Context context) {
        return newStorageImageCache(getDefaultStorageCacheDirectory(context));
    }

    /**
     * Default storage image cache directory
     *
     * @param context Context
     * @return Cache directory
     */
    @NonNull
    public static File getDefaultStorageCacheDirectory(@NonNull Context context) {
        File cacheDir = context.getExternalCacheDir();
        if (cacheDir == null) {
            cacheDir = context.getCacheDir();
        }
        return new File(cacheDir, DEFAULT_STORAGE_CACHE_DIRECTORY);
    }
}
