package com.nanorep.accessibility.voice

import android.Manifest
import android.content.Context
import android.support.v4.content.ContextCompat
import android.support.v4.content.PermissionChecker
import android.view.View
import android.widget.TextView
import com.nanorep.accessibility.voice.engines.SpeechToText


/**
 * Created by tehila on 29/11/2017.
 */
/**
 * optional combinations:
 * provider - registerListener - action - output
 * provider - default - none/action - output
 * default - registerListener - action - none
 * default - default - action - output
 * default - registerListener - none - none
 * provider - registerListener - none - none
 *
 * provide a bridge from/to a voice recognition engine.
 */
class VoiceRecognition private constructor(context: Context, var speechAdminister: SpeechRecognitionProvider? = null,
                                           var speechListener: SpeechedTextListener? = null,
                                           var actionView:View? = null, var outputView: TextView? = null) { // -> default - none - none - none engineOnly


    internal var permissionStatus : Int = PermissionChecker.PERMISSION_DENIED

    init{
        permissionStatus = ContextCompat.checkSelfPermission(context, Manifest.permission.RECORD_AUDIO)

        actionView?.setOnClickListener {

            speechAdminister?.start()

        }
    }

    fun release(){
        speechAdminister?.release()
        speechAdminister = null
        speechListener = null
        actionView = null
        outputView = null
    }

    companion object {
        fun create(context: Context, init: Builder.() -> Unit) = Builder(init).build(context)
    }


    class Builder private constructor() {

        internal constructor(init: Builder.() -> Unit) : this() {
            init()
        }


        private var speechRecognizer: SpeechRecognitionProvider? = null

        private var speechListener: SpeechedTextListener? = null


        private var actionView: View? = null

        private var outputView: TextView? = null


        fun speechAdminister(init: Builder.() -> SpeechRecognitionProvider) = apply {
            speechRecognizer = init()
        }

        fun speechAdminister(provider: SpeechRecognitionProvider) = apply {
            speechRecognizer = provider
        }

        fun speechListener(listener: SpeechedTextListener) = apply {
            speechListener = listener
        }

        fun speechListener(init: Builder.() -> SpeechedTextListener) = apply {
            speechListener = init()
        }

        fun bindActionView(init: Builder.() -> View) = apply {
            actionView = init()
            //self.actionView?.setOnClickListener { speechRecognizer?.start() }
        }

        fun bindOutputView(init: Builder.() -> TextView) = apply {
            outputView = init()
        }

        fun build(context: Context): VoiceRecognition {
            speechRecognizer = speechRecognizer ?: SpeechToText(context)

            speechListener = speechListener ?: let {
                //define default registerListener if not provided by user
                object : SpeechedTextListener {
                    override fun onResults(text: String) {
                        outputView?.text = text
                    }
                }
            }

            speechRecognizer?.registerListener(speechListener!!)

            return VoiceRecognition(context, speechRecognizer, speechListener, actionView, outputView)

        }
    }
}

