package com.zm.app.zxing.camera;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.regex.Pattern;

import android.app.Service;
import android.content.Context;
import android.graphics.Point;
import android.hardware.Camera;
import android.hardware.Camera.Size;
import android.os.Build;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.View;
import android.view.WindowManager;

final class CameraConfigurationManager {

	private static final String TAG = CameraConfigurationManager.class
			.getSimpleName();

	private static final int TEN_DESIRED_ZOOM = 27;
	private static final int DESIRED_SHARPNESS = 30;

	private static final Pattern COMMA_PATTERN = Pattern.compile(",");

	private final Context context;
	private Point screenResolution;
	private Point cameraResolution;
	private int previewFormat;
	private String previewFormatString;
	private View mView;

	CameraConfigurationManager(Context context,View v) {
		this.context = context;
		this.mView = v;
	}

	void initFromCameraParameters(Camera camera) {
		Camera.Parameters parameters = camera.getParameters();
	    previewFormat = parameters.getPreviewFormat();
	    previewFormatString = parameters.get("preview-format");
		int sw = getScreenSize(context, 0);
		int sh = getScreenSize(context, 1);
		if(mView == null){
			cameraResolution = getOptimalSize(camera.getParameters().getSupportedPreviewSizes(), sw, sh);
		}else {
			cameraResolution = getLandOptimalSize(camera.getParameters().getSupportedPreviewSizes(), sw, sh);
		}
		screenResolution = new Point(sw, sh-50);
		Log.d(TAG, "Camera resolution: " + screenResolution);
	}

	void setDesiredCameraParameters(Camera camera,boolean isLand) {
		Camera.Parameters parameters = camera.getParameters();
		Log.d(TAG, "Setting preview size: " + cameraResolution);
		parameters.setPreviewSize(cameraResolution.x, cameraResolution.y);
		setFlash(parameters);
		setZoom(parameters);

		//旋转
		if(!isLand){
			setDisplayOrientation(camera, 90);
		}
		camera.setParameters(parameters);
	}

	Point getCameraResolution() {
		return cameraResolution;
	}

	Point getScreenResolution() {
		return screenResolution;
	}

	int getPreviewFormat() {
		return previewFormat;
	}

	String getPreviewFormatString() {
		return previewFormatString;
	}

	private static Point getCameraResolution(Camera.Parameters parameters,
			Point screenResolution) {

		String previewSizeValueString = parameters.get("preview-size-values");

		if (previewSizeValueString == null) {
			previewSizeValueString = parameters.get("preview-size-value");
		}

		Point cameraResolution = null;

		if (previewSizeValueString != null) {
			Log.d(TAG, "preview-size-values parameter: "
					+ previewSizeValueString);
			// cameraResolution =
			// findBestPreviewSizeValue(previewSizeValueString,
			// screenResolution);
		}

		if (cameraResolution == null) {
			cameraResolution = new Point(screenResolution.x, screenResolution.y);
			// cameraResolution = new Point(
			// (screenResolution.x >> 3) << 3,
			// (screenResolution.y >> 3) << 3);
		}

		return cameraResolution;
	}

	private static Point findBestPreviewSizeValue(
			CharSequence previewSizeValueString, Point screenResolution) {
		int bestX = 0;
		int bestY = 0;
		int diff = Integer.MAX_VALUE;
		for (String previewSize : COMMA_PATTERN.split(previewSizeValueString)) {

			previewSize = previewSize.trim();
			int dimPosition = previewSize.indexOf('x');
			if (dimPosition < 0) {
				Log.w(TAG, "Bad preview-size: " + previewSize);
				continue;
			}

			int newX;
			int newY;
			try {
				newX = Integer.parseInt(previewSize.substring(0, dimPosition));
				newY = Integer.parseInt(previewSize.substring(dimPosition + 1));
			} catch (NumberFormatException nfe) {
				Log.w(TAG, "Bad preview-size: " + previewSize);
				continue;
			}

			int newDiff = Math.abs(newX - screenResolution.x)
					+ Math.abs(newY - screenResolution.y);
			if (newDiff == 0) {
				bestX = newX;
				bestY = newY;
				break;
			} else if (newDiff < diff) {
				bestX = newX;
				bestY = newY;
				diff = newDiff;
			}

		}

		if (bestX > 0 && bestY > 0) {
			return new Point(bestX, bestY);
		}
		return null;
	}

	private static int findBestMotZoomValue(CharSequence stringValues,
			int tenDesiredZoom) {
		int tenBestValue = 0;
		for (String stringValue : COMMA_PATTERN.split(stringValues)) {
			stringValue = stringValue.trim();
			double value;
			try {
				value = Double.parseDouble(stringValue);
			} catch (NumberFormatException nfe) {
				return tenDesiredZoom;
			}
			int tenValue = (int) (10.0 * value);
			if (Math.abs(tenDesiredZoom - value) < Math.abs(tenDesiredZoom
					- tenBestValue)) {
				tenBestValue = tenValue;
			}
		}
		return tenBestValue;
	}

	private void setFlash(Camera.Parameters parameters) {
		if (Build.MODEL.contains("Behold II") && Build.VERSION.SDK_INT == 3) {
			parameters.set("flash-value", 1);
		} else {
			parameters.set("flash-value", 2);
		}

		parameters.set("flash-mode", "off");
	}

	private void setZoom(Camera.Parameters parameters) {

		String zoomSupportedString = parameters.get("zoom-supported");
		if (zoomSupportedString != null
				&& !Boolean.parseBoolean(zoomSupportedString)) {
			return;
		}

		int tenDesiredZoom = TEN_DESIRED_ZOOM;

		String maxZoomString = parameters.get("max-zoom");
		if (maxZoomString != null) {
			try {
				int tenMaxZoom = (int) (10.0 * Double
						.parseDouble(maxZoomString));
				if (tenDesiredZoom > tenMaxZoom) {
					tenDesiredZoom = tenMaxZoom;
				}
			} catch (NumberFormatException nfe) {
				Log.w(TAG, "Bad max-zoom: " + maxZoomString);
			}
		}

		String takingPictureZoomMaxString = parameters
				.get("taking-picture-zoom-max");
		if (takingPictureZoomMaxString != null) {
			try {
				int tenMaxZoom = Integer.parseInt(takingPictureZoomMaxString);
				if (tenDesiredZoom > tenMaxZoom) {
					tenDesiredZoom = tenMaxZoom;
				}
			} catch (NumberFormatException nfe) {
				Log.w(TAG, "Bad taking-picture-zoom-max: "
						+ takingPictureZoomMaxString);
			}
		}

		String motZoomValuesString = parameters.get("mot-zoom-values");
		if (motZoomValuesString != null) {
			tenDesiredZoom = findBestMotZoomValue(motZoomValuesString,
					tenDesiredZoom);
		}

		String motZoomStepString = parameters.get("mot-zoom-step");
		if (motZoomStepString != null) {
			try {
				double motZoomStep = Double.parseDouble(motZoomStepString
						.trim());
				int tenZoomStep = (int) (10.0 * motZoomStep);
				if (tenZoomStep > 1) {
					tenDesiredZoom -= tenDesiredZoom % tenZoomStep;
				}
			} catch (NumberFormatException nfe) {

			}
		}

		if (maxZoomString != null || motZoomValuesString != null) {
			parameters.set("zoom", String.valueOf(tenDesiredZoom / 10.0));
		}

		if (takingPictureZoomMaxString != null) {
			parameters.set("taking-picture-zoom", tenDesiredZoom);
		}
	}

	public static int getDesiredSharpness() {
		return DESIRED_SHARPNESS;
	}

	protected void setDisplayOrientation(Camera camera, int angle) {
		Method downPolymorphic;
		try {
			downPolymorphic = camera.getClass().getMethod(
					"setDisplayOrientation", new Class[] { int.class });
			if (downPolymorphic != null)
				downPolymorphic.invoke(camera, new Object[] { angle });
		} catch (Exception e1) {
		}
	}
	
    private int getScreenSize(Context context, int flag){
    	if(mView !=null){
    		if(flag == 0){
    			return mView.getWidth();
    		}else{
    			return mView.getHeight();
    		}
    	}else {
    		WindowManager wm = (WindowManager)context.getSystemService(Service.WINDOW_SERVICE);
        	DisplayMetrics dm = new DisplayMetrics();
        	wm.getDefaultDisplay().getMetrics(dm);
    		if(flag == 0){
        		return dm.widthPixels;
        	}else{
        		return dm.heightPixels;
        	}
		}
    }
    
    /**
     * 该方法逻辑适用于摄像头的预览界面尺寸总是全屏时（或除状态栏部分）
     * @param sizes
     * @param w
     * @param h
     * @return
     */
    private Point getOptimalSize(List<Size> sizes, int w, int h){
    	Collections.sort(sizes, sizeComparator);
    	Size highestResolution = sizes.get(sizes.size()-1);
    	if(highestResolution.width >= h && highestResolution.height >= w){
    		//返回屏幕尺寸
    		return new Point(getScreenSize(context, 1), getScreenSize(context, 0));
    	}else{
    		float rw = highestResolution.width;
    		float rh = highestResolution.height;
    		float ratio = rw/rh;
    		for(int i=sizes.size()-1; i>=0; i--){
    			Size s = sizes.get(i);
    			float tw = s.width;
    			float th = s.height;
    			float tRatio = tw/th;
    			if(Math.abs(tRatio - ratio) < 0.2){
    				return new Point(s.width, s.height);
    			}
    		}
    		return null;
    	}
    }
    /**
     * 该方法逻辑适用于摄像头的预览界面尺寸不是全屏
     * @param sizes
     * @param w
     * @param h
     * @return
     */
    private Point getLandOptimalSize(List<Size> sizes, int w, int h){
    	Collections.sort(sizes, sizeComparator);
    	Size highestResolution = sizes.get(sizes.size()-1);
    	float rw = highestResolution.width;
		float rh = highestResolution.height;
		float ratio = rw/rh;
		for(int i=sizes.size()-2; i>=0; i--){
			Size s = sizes.get(i);
			float tw = s.width;
			float th = s.height;
			float tRatio = tw/th;
			if(Math.abs(tRatio - ratio) < 0.2){
				return new Point(s.width, s.height);
			}
		}
		return new Point(highestResolution.width, highestResolution.height);
    }
    
    private Comparator<Size> sizeComparator = new Comparator<Camera.Size>() {
		@Override
		public int compare(Size lhs, Size rhs) {
			int firResolution = lhs.width*lhs.height;
			int secResolution = rhs.width*rhs.height;
			if(firResolution == secResolution){
				return 0;
			}else if(firResolution <secResolution){
				return -1;
			}else{
				return 1;
			}
		}
	};

}
