package com.boanda.android.sync;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONObject;
import org.xmlpull.v1.XmlPullParser;

import com.szboanda.android.platform.db.TableDao;
import com.szboanda.android.platform.util.FileUtils;
import com.szboanda.android.platform.util.JsonUtils;

import android.text.TextUtils;
import android.util.Xml;

/**
 * <p>Description: 数据同步文件解析器</p>
 * @author    Siyi Lu    Create Date: 2015-7-9
 */
public class SyncFileParser {
	
	/**表标签*/
	private final static String TAG_TABLE = "table";
	/**响应时间标签*/
	private final static String TAG_REP_TIME = "response-time";
	/**字段标签*/
	private final static String TAG_FIELD = "field";
	/**数据条目标签*/
	private final static String TAG_DATA_ITEM = "data-item";
	/**数据字段内容标签*/
	private final static String TAG_DATA_FIELD = "data-field";
	
	private String encode = "UTF-8";
	/**表名*/
	private String tableName = "";
	/**响应时间*/
	private long responseTime ;
	/**表字段*/
	private List<DataSyncField> fields = null;
	
	public String getTableName() {
		return tableName;
	}

	public long getResponseTime() {
		return responseTime;
	}

	public List<DataSyncField> getFields() {
		return fields;
	}

	public void setTableName(String tableName) {
		this.tableName = tableName;
	}

	public void setResponseTime(long responseTime) {
		this.responseTime = responseTime;
	}

	public void setFields(List<DataSyncField> fields) {
		this.fields = fields;
	}

	public boolean parse(File file){
		boolean isSuccess = false;
		try{
			if(file != null && file.exists()){
				XmlPullParser parser = Xml.newPullParser();
				InputStream input = null;
				try{
					input = new BufferedInputStream(new FileInputStream(file));
					parser.setInput(input, encode);
					int eventType = parser.getEventType();
					while (eventType != XmlPullParser.END_DOCUMENT) {
						switch (eventType) {
						case XmlPullParser.START_DOCUMENT:
							fields = new ArrayList<DataSyncField>();
							break;
						case XmlPullParser.START_TAG:
							parseTag(parser);
							break;
						default:
							break;
						}
						eventType = parser.next();
					}
					//解析结束
					if(dao != null){
						dao.endTransaction();
					}
					isSuccess = true;
				}catch (Exception e) {
					e.printStackTrace();
				}finally{
					try{
						input.close();
						input = null;
					}catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}catch(Exception e){
			e.printStackTrace();
		}
		return isSuccess;
	}
	
	public boolean parseJsonFile(File file){
		boolean isSuccess = false;
		try{
			if(file != null && file.exists()){
				String content = FileUtils.readContent(file.getAbsolutePath());
				if(!TextUtils.isEmpty(content)){
					JSONObject data = JsonUtils.parseJsonObject(content);
					if(data != null){
						tableName = data.optString("table");
						JSONArray dataArray = data.optJSONArray("data_array");
						if(dataArray != null){
							for(int i = 0; i < dataArray.length(); i++){
								JSONObject item = dataArray.optJSONObject(i);
								saveData(item);
							}
						}
						//解析结束
						if(dao != null){
							dao.endTransaction();
						}
						isSuccess = true;
					}
				}
			}
		}catch(Exception e){
			e.printStackTrace();
		}
		return isSuccess;
	}
	
	/**
	 * 解析标签
	 * @param parser
	 */
	private void parseTag(XmlPullParser parser){
		try{
			
			String tagName = parser.getName();
			if(TAG_TABLE.equals(tagName)){
				tableName = parser.nextText();
			}else if(TAG_REP_TIME.equals(tagName)){
				responseTime = Long.parseLong(parser.nextText());
			}else if(TAG_FIELD.equals(tagName)){
				parseTableFields(parser);
			}else if(TAG_DATA_ITEM.equals(tagName)){
				JSONObject itemData = parseDataItem(parser);
				saveData(itemData);
			}
		}catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * 解析表字段
	 * @param parser
	 */
	private void parseTableFields(XmlPullParser parser){
		String name = parser.getAttributeValue("", "name");
		String type = parser.getAttributeValue("", "type");
		boolean isPrimaryKey = Boolean.parseBoolean(parser.getAttributeValue("", "primary_key"));
		DataSyncField field = new DataSyncField();
		field.setName(name);
		field.setType(type);
		field.setPrimaryKey(isPrimaryKey);
		fields.add(field);
	}
	
	/**
	 * 解析数据条目
	 * @param parser
	 * @return
	 */
	private JSONObject parseDataItem(XmlPullParser parser){
		JSONObject result = null;
		try{
			int event = parser.nextTag();
			String name = parser.getName();
			result = new JSONObject();
			//直到</data-item>解析完一条数据
			while (TAG_DATA_FIELD.equals(name) && event == XmlPullParser.START_TAG) {
				String fieldName = parser.getAttributeValue("", "field");
				String fieldValue = parser.nextText();
				result.put(fieldName, fieldValue);
				parser.nextTag();
				name = parser.getName();
			}
		}catch (Exception e) {
			e.printStackTrace();
		}
		return result;
	}
	
	TableDao dao = null;
	public void saveData(JSONObject itemData){
		if(dao == null){
			dao = new TableDao(tableName);
			dao.beginTransaction();
		}
		dao.replaceData(itemData);
	}
	
	/**以下是数据同步文件示例：
	<?xml version="1.0" encoding="utf-8"?>
	<data-sync>
		<table>T_COMN_WRWDM</table>
		<response-time>20150708143539</response-time>
		<fields>
			<field name="WRWLB" type="VARCHAR" primary_key="true"/>
			<field name="WRWDM" type="VARCHAR" primary_key="true"/>
			<field name="WRW" type="VARCHAR"/>
			<field name="PXM" type="VARCHAR"/>
			<field name="CJSJ" type="VARCHAR"/>
			<field name="XGSJ" type="VARCHAR"/>
			<field name="ORGID" type="VARCHAR"/>
		</fields>
		<data-list>
			<data-item>
				<data-field field="WRWLB"><![CDATA[DQ]]></data-field>
				<data-field field="WRWDM"><![CDATA[a21004]]></data-field>
				<data-field field="WRW"><![CDATA[NO2]]></data-field>
				<data-field field="PXM"><![CDATA[03]]></data-field>
				<data-field field="CJSJ"><![CDATA[20150708143004]]></data-field>
				<data-field field="XGSJ"><![CDATA[20150708143004]]></data-field>
				<data-field field="ORGID"><![CDATA[450000000000]]></data-field>
			</data-item>
			<data-item>
				<data-field field="WRWLB"><![CDATA[DQ]]></data-field>
				<data-field field="WRWDM"><![CDATA[a21026-01]]></data-field>
				<data-field field="WRW"><![CDATA[SO2]]></data-field>
				<data-field field="PXM"><![CDATA[01]]></data-field>
				<data-field field="CJSJ"><![CDATA[20150708143004]]></data-field>
				<data-field field="XGSJ"><![CDATA[20150708143004]]></data-field>
				<data-field field="ORGID"><![CDATA[450000000000]]></data-field>
			</data-item>
		</data-list>
	</data-sync>
*/
}
