package com.szboanda.android.platform.view;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.text.TextUtils;
import android.text.method.KeyListener;
import android.util.AttributeSet;
import android.widget.EditText;
import android.widget.LinearLayout;

import com.szboanda.android.platform.R;
import com.szboanda.android.platform.util.DimensionUtils;

public class PropertyView extends EditText
	implements IBindableView{
	
	public static final String KEY_LABEL = "LABEL";
	
	public static final String KEY_VALUE = "VALUE";
	
	private static final int DEFAULT_LABEL_COLOR = 0xFF666666;
	
	/**是否绘制文本的背景框*/
	private boolean drawTextBg = false;
	
	/**文本的背景框的圆角半径*/
	private float mTextBgRadius;
	
	/**文本的背景框颜色*/
	private int mTextBgColor;
	
	/**是否可编辑*/
	private boolean isEditable = true;
	
	/**{@link EditText}已经废除editable属性，当设置为不可编辑时，使用setKeyListener(null)
	 *来实现，mKeyListener只是做为一个缓存 
	 */
	private KeyListener mKeyListener;
	
	protected BindableViewHandler mBindHandler;
	
	private LabelDrawer mLabelDrawer;
	
	private boolean isLabelAlignRight = false;

    private boolean hasColon = false;
	
	public PropertyView(Context context, AttributeSet attrs, int defStyleAttr) {
		super(context, attrs, defStyleAttr);
	}

	public PropertyView(Context context, AttributeSet attrs) {
		super(context, attrs);
		init();
		initWithAttributeSet(attrs);
	}

	public PropertyView(Context context) {
		super(context);
		init();
	}
	
	/**文本的边界范围*/
	RectF mTextBgBound = null;
	/**绘制文本的画笔*/
	Paint mTextBgPaint = null;
	private void init(){
		setVerticalScrollBarEnabled(false);
		mKeyListener = getKeyListener();
		
		mTextBgRadius = DimensionUtils.dip2Px(getContext(), 3);
		mTextBgColor = 0xFFDDDDDD;
		mTextBgBound = new RectF();
		mTextBgPaint = new Paint();
		mTextBgPaint.setAntiAlias(true);
		mTextBgPaint.setStyle(Paint.Style.FILL);
		mTextBgPaint.setColor(mTextBgColor);
		int defaultPadding = DimensionUtils.dip2Px(getContext(), 5);
		setPadding(defaultPadding, getPaddingTop(), defaultPadding, getPaddingTop());
		
		mBindHandler = new BindableViewHandler(this);
		mLabelDrawer = new LabelDrawer();
		mLabelDrawer.setLabelSize(getPaint().getTextSize());
	}
	
	private void initWithAttributeSet(AttributeSet attrs){
		mBindHandler.initWithTypedArray(getResources(), attrs);
		TypedArray ta = getResources().obtainAttributes(attrs, R.styleable.PropertyView);
		isEditable 								= ta.getBoolean(R.styleable.PropertyView_editable, true);
		String label 							= ta.getString(R.styleable.PropertyView_labelText);
		int labelTextColor 				= ta.getColor(R.styleable.PropertyView_labelTextColor, DEFAULT_LABEL_COLOR);
		int labelWeight 					= ta.getInt(R.styleable.PropertyView_labelWeight, -1);
		int labelPaddingLeft 			= (int)ta.getDimension(R.styleable.PropertyView_labelPaddingLeft, 0);
		int labelMarginRight 			= (int)ta.getDimension(R.styleable.PropertyView_labelMarginRight, 0);
		mTextBgRadius 					= ta.getDimension(R.styleable.PropertyView_radius, mTextBgRadius);
		mTextBgColor 						= ta.getColor(R.styleable.PropertyView_textBgColor, mTextBgColor);
		labelMarginRight 				= (int)ta.getDimension(R.styleable.PropertyView_marginBetweenLabelText, labelMarginRight);
		drawTextBg 							= ta.getBoolean(R.styleable.PropertyView_drawTextBg, false);
		isLabelAlignRight 				= ta.getBoolean(R.styleable.PropertyView_labelAlignRight, false);
		
		mTextBgPaint.setColor(mTextBgColor);
		ta.recycle();
		if(!isEditable){
			setKeyListener(null);
		}
		if(!TextUtils.isEmpty(label)){
			label +=  hasColon ? "：" : "";
		}
		//检测是否配置了android视图本身的background属性，如果没有，设置视图背景为透明
		String namespace = "http://schemas.android.com/apk/res/android";
		String layoutBackground = attrs.getAttributeValue(namespace, "background");
		if(TextUtils.isEmpty(layoutBackground)){
			int plr = getPaddingLeft();
			int ptb = getPaddingTop();
			setBackgroundColor(Color.TRANSPARENT);
			setPadding(plr, ptb, plr, ptb);
		}
		mLabelDrawer = new LabelDrawer(label);
		mLabelDrawer.setLabelSize(getPaint().getTextSize());
		mLabelDrawer.setLableTextColor(labelTextColor);
		mLabelDrawer.setLabelWeight(labelWeight);
		mLabelDrawer.setLabelPaddingLeft(labelPaddingLeft);
		mLabelDrawer.setLabelMarginRight(labelMarginRight);
		mLabelDrawer.setHorizontalAlignRight(isLabelAlignRight);
		
	}
	
	public void setLabel(String label) {
		mLabelDrawer.setLabel(label + (hasColon ? "：" : ""));
	}
	
	public void setLableTextColor(int color){
		mLabelDrawer.setLableTextColor(color);
	}
	
	public void setLabelWeight(int labelWeight) {
		mLabelDrawer.setLabelWeight(labelWeight > 50 ? 50 : labelWeight);
	}
	
	public void setLabelPaddingLeft(int padding){
		mLabelDrawer.setLabelPaddingLeft(padding);
	}
	
	public void setLabelMarginRight(int margin){
		mLabelDrawer.setLabelMarginRight(margin);
	}

	public void setTextBgColor(int color){
		mTextBgColor = color;
		mTextBgPaint.setColor(color);
	}
	
	public void setTextBgRadius(int radius){
		mTextBgRadius = radius;
	}

    public void setEditable(boolean editable){
		isEditable = editable;
		if(editable){
			setKeyListener(mKeyListener);
		}else{
			setKeyListener(null);
		}
	}
	
	public void setDrawTextBg(boolean drawTextBg) {
		this.drawTextBg = drawTextBg;
	}

	public void setHorizontalAlignRight(boolean isHorizontalAlignRight) {
		mLabelDrawer.setHorizontalAlignRight(isHorizontalAlignRight);
	}

	public void setColonAuto(boolean auto){
        hasColon = auto;
    }

	public void setLabelAlignVerticalCenter(boolean isLabelAlignVerticalCenter) {
		mLabelDrawer.setLabelAlignVerticalCenter(isLabelAlignVerticalCenter);
	}

    /**
     * 绘制的背景时，为了有paddding效果，比实际的padding左移了一个Padding的距离,
     * 见{@link #drawTextBackground(Canvas)}，所以需要多设置一个原始padding的值
     * @param left
     * @param top
     * @param right
     * @param bottom
     */
    @Override
    public void setPadding(int left, int top, int right, int bottom) {
        if(mLabelDrawer != null){
            int orientation = mLabelDrawer.getLabelOrientation();
            if(LinearLayout.HORIZONTAL == orientation) {
                super.setPadding(left + mLabelDrawer.getOriginalPaddingLeft(), top, right, bottom);
            }else{
                super.setPadding(left, top + mLabelDrawer.getOriginalPaddingTop(), right, bottom);
            }
        }else {
            super.setPadding(left, top, right, bottom);
        }
    }

    public LabelDrawer getLabelDrawer(){
		return mLabelDrawer;
	}
	
	@Override
	protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
		super.onMeasure(widthMeasureSpec, heightMeasureSpec);
		int height = mLabelDrawer.measureLabel(this, getMeasuredWidth(), getMeasuredHeight());
		setMeasuredDimension(getMeasuredWidth(), height);
	}
	
	@Override
	protected void onDraw(Canvas canvas) {
		if(drawTextBg){
			drawTextBackground(canvas);
		}
		super.onDraw(canvas);
		mLabelDrawer.drawLabel(canvas);
	}
	
	/** 绘制文本的背景*/
	private void drawTextBackground(Canvas canvas){
		canvas.save();
		canvas.restore();
        int labelOrientation = mLabelDrawer.getLabelOrientation();
        if(LinearLayout.HORIZONTAL == labelOrientation) {
            mTextBgBound.set(
                    mLabelDrawer.getAdjustedPadding() - mLabelDrawer.getOriginalPaddingLeft()
                    , 0, getMeasuredWidth(), getMeasuredHeight());
        }else{
            int top = mLabelDrawer.getAdjustedPadding() - mLabelDrawer.getOriginalPaddingTop();
            mTextBgBound.set(0, top, getMeasuredWidth(), getMeasuredHeight());
        }
		canvas.drawRoundRect(mTextBgBound, mTextBgRadius, mTextBgRadius, mTextBgPaint);
	}

	@Override
	public void setField(String field) {
		mBindHandler.setField(field);
	}

	@Override
	public void setEmptyMsg(String notEmptyTipMsg) {
		mBindHandler.setNotEmptyMsg(notEmptyTipMsg);
	}

	@Override
	public void setBindValue(String value) {
		setText(value);
	}

	@Override
	public String getBindValue() {
		return getText().toString();
	}

	@Override
	public BindableViewHandler getBindHandler() {
		return mBindHandler;
	}
	
}
