package com.szboanda.android.platform.util.tree;

import android.text.TextUtils;

import com.szboanda.android.platform.db.DbHelper;
import com.szboanda.android.platform.db.SQLiteDao;

import org.xutils.db.Selector;
import org.xutils.db.sqlite.WhereBuilder;
import org.xutils.db.table.ColumnEntity;
import org.xutils.db.table.TableEntity;

import java.util.ArrayList;
import java.util.List;

/**
 * <p>Description: 树结构构建器</p>
 * @author    Siyi Lu    Create Date: 2015-7-14
 * @param <T>
 */
public class TreeBuilder<T extends ITreeNode> {
	
	/**自动递归形成完成整树,数据量不大的情况下可设置为true*/
	private boolean autoRecursive;
	
	/**创建的树*/
	private T mTree;
	
	/**父ID的字段名*/
	private String pidColName;
	
	/**根节点ID的值*/
	private Object rootNodeIdValue;
	
	private WhereBuilder whereBuilder;
	
	private TreeNodeAdapter mAdapter;
	
	/**所有节点集合，当{@link #autoRecursive} 为true时，为防止自动递归反复查询数据库
	 * 在getRootNode方法一次性查询所有节点数据
	 * */
	private List<T> allNodes = null;
	
	/**
	 * 构造器
	 * @param rootNodeIdValue	根节点的主键值
	 * @param pidColName				存储父节点主键值的字段名
	 */
	public TreeBuilder(String rootNodeIdValue, String pidColName){
		this.rootNodeIdValue = rootNodeIdValue;
		this.pidColName = pidColName;
	}
	
	public TreeBuilder(TreeNodeAdapter adapter){
		mAdapter = adapter;
		autoRecursive = true;
		adapter.setAutoRecursive(autoRecursive);
	}
	
	/**
	 * 指定查询父节点时使用的主键列名，通常在联合主键的场合下使用，即表有两个键是unique，
	 * @param explicitId
	 */
	public void setExplicitId(ColumnEntity explicitId){
		mId = explicitId;
	}
	
	public boolean isAutoRecursive() {
		return autoRecursive;
	}

	public void setAutoRecursive(boolean autoRecursive) {
		this.autoRecursive = autoRecursive;
		if(mAdapter != null){
			mAdapter.setAutoRecursive(autoRecursive);
		}
	}
	
	public String getPidColName() {
		return pidColName;
	}

	public void setPidColName(String pidColName) {
		this.pidColName = pidColName;
	}

	public Object getRootNodeIdValue() {
		return rootNodeIdValue;
	}

	public void setRootNodeIdValue(Object rootNodeIdValue) {
		this.rootNodeIdValue = rootNodeIdValue;
	}
	
	public WhereBuilder getWhereBuilder() {
		return whereBuilder;
	}

	public void setWhereBuilder(WhereBuilder whereBuilder) {
		this.whereBuilder = whereBuilder;
	}

	public T getTree(){
		return mTree;
	}


	SQLiteDao dao = null;
	TableEntity table = null;
	ColumnEntity mId = null;
	Class<T> classOfT;
	@SuppressWarnings("unchecked")
	public void buildTree(Class<T> cls){
		classOfT = cls;
		initDaoAndTableId(cls);
		if(!TextUtils.isEmpty(mId.getName())){
			try{
				mTree = mAdapter == null ? getRootNode(dao, cls)
						: (T)mAdapter.getRootNode(dao, rootNodeIdValue);
				if(mTree == null){
					return;
				}
				recursiveNode(mTree, cls);
			}catch (Exception e) {
				e.printStackTrace();
			}
		}
		//如果是自动递归，则关闭数据
		if(autoRecursive){
//			dao.close();//对于一个页面存在多下拉列表项取值时，关闭之后其他不能获取，debug报错
		}
	}
	
	public void initDaoAndTableId(Class<T> cls){
        try{
            if(classOfT == null){
                classOfT = cls;
            }
            dao = DbHelper.getDao();
            table = dao.getTable(cls);
            if(mId == null){
                mId = table.getId();
            }
        }catch(Exception e){
            e.printStackTrace();
        }
	}

	private T getRootNode(SQLiteDao dbUtils, Class<T> cls){
		try{
			T result = null;
			Selector s = dbUtils.selector(cls);
			s.where(mId.getName(), "=", rootNodeIdValue);
			if(whereBuilder != null){
				s.and(whereBuilder);
			}
			result = (T) s.findFirst();
			if(autoRecursive){
				Selector si = dbUtils.selector(cls);
				if(whereBuilder != null){
					si.where(whereBuilder);
				}
				allNodes = si.findAll();
			}
			return result;
		}catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}
	
	/**
	 * 递归构建树节点
	 * @param parent
	 * @param cls
	 */
	@SuppressWarnings("unchecked")
	private void recursiveNode(ITreeNode parent, Class<T> cls){
		if(!parent.childEnable()){
			return;
		}
		List<T> childNodes = null;
		try{
			childNodes = mAdapter == null ? getChildNodes(parent, cls)
					: (List<T>)mAdapter.getChildNodes(dao, parent);
			if(childNodes != null && childNodes.size()>0){
				for(ITreeNode node : childNodes){
					parent.addChild(node);
					if(autoRecursive){//开启自动递归时才自动构建下一级节点
						recursiveNode(node, cls);
					}
				}
			}
		}catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	private List<T> getChildNodes(ITreeNode parent, Class<T> cls){
		List<T> childNodes = null;
		try{
			if(TextUtils.isEmpty(pidColName)){
				throw new Exception("必须设置父ID的字段名");
			}
			Object idValue = mId.getColumnValue(parent);
			if(autoRecursive && allNodes != null){
				childNodes = new ArrayList<T>();
				ColumnEntity parentCol = (ColumnEntity) table.getColumnMap().get(pidColName);
				for(T n : allNodes){
					if(String.valueOf(idValue).equals(String.valueOf(parentCol.getColumnValue(n)))){
						childNodes.add(n);
					}
				}
			}else{
                Selector select = dao.selector(cls);
                select.where(pidColName, "=", idValue);
				if(whereBuilder != null){
					select.and(whereBuilder);
				}
				childNodes = select.findAll();
			}
		}catch (Exception e) {
			e.printStackTrace();
		}
		return childNodes;
	}
	
	/**
	 * 当{@link #autoRecursive} 为false时，即时创建当前节点的子节点
	 * @param parent
	 */
	public void buildChildNodes(ITreeNode parent){
		recursiveNode(parent, classOfT);
	}
	
	/**
	 * 关闭树构建。在非自动递归自节点的情况下，不会自动关闭数据库
	 */
	public void closeBuild(){
		if(!autoRecursive){
            try{
                dao.close();
            }catch(Exception e){
                e.printStackTrace();
            }
		}
	}
	
	/**
	 * 示例代码
	 * @param args
	 */
	public static void main(String[] args) {
		
		//示例一：组织机构树
//		OrganizationAdapter orgiAdapter = new OrganizationAdapter();
//		TreeBuilder<Department> orgiBuilder = new TreeBuilder<Department>(orgiAdapter);
//		自动递归，构建时只要节点有子节点都自动递归
//		orgiBuilder.setAutoRecursive(true);
		//指定根节点的ID
//		orgiBuilder.setRootNodeIdValue("ROOT");
		//开始构建树
//		orgiBuilder.buildTree(Department.class);
//		Department department = orgiBuilder.getTree();
		//遍历树
//		department.iterator();
		
		
		//示例二：构建行业类型树
//		ComnCodeTreeAdapter codeAdapter = new ComnCodeTreeAdapter("44", "ROOT");
//		TreeBuilder<CommonCode> codeBuilder = new TreeBuilder<CommonCode>(codeAdapter);
//		codeBuilder.buildTree(CommonCode.class);
//		CommonCode codeTree = codeBuilder.getTree();
//		codeTree.iterator();
		
		//示例三：构建树结构单个表的树，行政区划树
//		TreeBuilder<RegionCode> rb = new TreeBuilder<RegionCode>("450000000000", "FDM");
		//不自动递归，但是默认构建到树的根节点下一级
//		rb.setAutoRecursive(false);
//		rb.buildTree(RegionCode.class);
//		RegionCode rc = rb.getTree();
		//获取到一个节点
//		ITreeNode child = rc.getChilds().get(0);
		//主动调用构建子节点
//		rb.buildChildNodes(child);
//		rb.getTree().iterator();
		//非自动递归方式需要主动关闭构建连接
//		rb.closeBuild();
		
		//示例四：菜单树,因为菜单查询菜单时，除ID和父ID外，还需要加额外固定条件
//		TreeBuilder<Menu> builder = new TreeBuilder<Menu>("ROOT", "FCDXH");
//		builder.setAutoRecursive(true);
//		固定的额外条件
//		WhereBuilder wb = WhereBuilder.b("YHID", "=", "system");
//		builder.setWhereBuilder(wb);
//		builder.buildTree(Menu.class);
//		Menu userMenu = builder.getTree();
//		userMenu.iterator();
	}
	
	public void clear(){
		mTree = null;
		if(allNodes != null){
			allNodes.clear();
		}
	}
	
}
