package com.szboanda.android.platform.widget;


import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Typeface;
import android.support.annotation.ColorInt;
import android.text.format.DateFormat;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewConfiguration;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.NumberPicker;
import android.widget.NumberPicker.OnValueChangeListener;

import com.szboanda.android.platform.R;
import com.szboanda.android.platform.util.DimensionUtils;

import java.text.SimpleDateFormat;
import java.util.Calendar;
 
/**
 * 日期时间选择控件，不建议单独使用，改用dialog，单独使用请注意修改主题
 * @author SuHao
 * @since 2016-05-19 22:55:39
 * @see DateTimePickerDialog
 */
public class DateTimePicker extends FrameLayout {
	
	public static final int SELECT_TYPE_DATE_TIME = 0;
	public static final int SELECT_TYPE_DATE = 1;
	public static final int SELECT_TYPE_TIME = 2;
	public final String[] DATE_FORMAT = {"yyyy-MM-dd HH:mm","yyyy-MM-dd","HH:mm"}; 
	public final String[] FAMILY_NAME = {"normal","sans","serif","monospace"};
	
	private int dividerColor = 0xFF72C8E6;//必须带透明度
	private int dividerDistance = 0;
	private int maxHeight = 0;
	private int textSize = 0;
	private int typeface = 0;
	private int textStyle = 0;
	private int textColor = 0xFF000000;
	private int friction = 8;
	
    private NumberPickerReflect mNumberPicker1;
    private NumberPickerReflect mNumberPicker2;
    private NumberPickerReflect mNumberPicker3;
    private Calendar mDate;
    private int mSelectType = SELECT_TYPE_DATE_TIME;
    private OnDateTimeChangedListener mOnDateTimeChangedListener;
    
    public DateTimePicker(Context context) {
    	this(context, new PickerOptions());
    }

    public DateTimePicker(Context context, int selectType) {
    	this(context, new PickerOptions(selectType));
    }
    
    public DateTimePicker(Context context, AttributeSet attrs){
    	super(context, attrs);
    	int selectType = SELECT_TYPE_DATE_TIME;
    	long date = System.currentTimeMillis();
		if(attrs != null){
			TypedArray array = context.obtainStyledAttributes(attrs, R.styleable.DateTimePicker);
			selectType = array.getInt(R.styleable.DateTimePicker_select_type, selectType);
			String specificDate = array.getString(R.styleable.DateTimePicker_specific_date);
			dividerColor = array.getColor(R.styleable.DateTimePicker_dtp_dividerColor, dividerColor);
			textColor = array.getColor(R.styleable.DateTimePicker_dtp_textColor, textColor);
			maxHeight = DimensionUtils.dip2Px(context, 180);
			maxHeight = array.getDimensionPixelSize(R.styleable.DateTimePicker_dtp_maxHeight, maxHeight);
			dividerDistance = DimensionUtils.dip2Px(context, 48);
			dividerDistance = array.getDimensionPixelSize(R.styleable.DateTimePicker_dtp_dividerDistance, dividerDistance);
			textSize = array.getDimensionPixelSize(R.styleable.DateTimePicker_dtp_textSize, textSize);
			typeface = array.getInt(R.styleable.DateTimePicker_dtp_typeface, typeface);
			textStyle = array.getInt(R.styleable.DateTimePicker_dtp_typeface, textStyle);
			array.recycle();
			try {
				date = new SimpleDateFormat(DATE_FORMAT[selectType]).parse(specificDate).getTime();
			} catch (Exception e) {
			}
		}
		initViews(context, selectType);
		setDate(date);
    }
    
    public DateTimePicker(Context context, PickerOptions opts) {
    	super(context);
    	long date = evaluateOptions(opts);
    	initViews(context, mSelectType);
    	setDate(date);
    }
    
    public void setOptions(PickerOptions opts){
    	if(opts == null)
    		return;
    	long date = evaluateOptions(opts);
    	//更新视图
    	float firstNumberPickerWeight = mSelectType == SELECT_TYPE_DATE_TIME ? 2 : 1;
    	((LinearLayout.LayoutParams)mNumberPicker1.getLayoutParams()).weight = firstNumberPickerWeight;
    	setCustom(mNumberPicker1);
    	setCustom(mNumberPicker2);
    	setCustom(mNumberPicker3);
    	//更新时间
    	setDate(date);
    }
    
    public void setDate(long date){
    	/*设置时间*/
        mDate = Calendar.getInstance();
        mDate.setTimeInMillis(date);
        
        if(mSelectType == SELECT_TYPE_DATE_TIME){
        	updateDateControl();
        	updateHourControl();
        	updateMinuteControl();
        	mNumberPicker1.setOnValueChangedListener(mOnDateChangedListener);
        	mNumberPicker2.setOnValueChangedListener(mOnHourChangedListener);
        	mNumberPicker3.setOnValueChangedListener(mOnMinuteChangedListener);
        }else if(mSelectType == SELECT_TYPE_DATE){
        	updateYearControl();
        	updateMonthControl();
        	updateDayControl();
        	mNumberPicker1.setOnValueChangedListener(mOnYearChangedListener);
        	mNumberPicker2.setOnValueChangedListener(mOnMonthChangedListener);
        	mNumberPicker3.setOnValueChangedListener(mOnDayChangedListener);
        }else if(mSelectType == SELECT_TYPE_TIME){
        	mNumberPicker1.setVisibility(View.GONE);
        	updateHourControl();
        	updateMinuteControl();
        	mNumberPicker2.setOnValueChangedListener(mOnHourChangedListener);
        	mNumberPicker3.setOnValueChangedListener(mOnMinuteChangedListener);
        }
    }
    
    public void setOnDateTimeChangedListener(OnDateTimeChangedListener callback) {
        mOnDateTimeChangedListener = callback;
    }
   
    public Calendar getDate(){
    	return mDate;
    }
    
    private long evaluateOptions(PickerOptions opts){
    	long date = System.currentTimeMillis();
    	if(opts != null){
    		this.dividerColor = opts.dividerColor;
        	this.dividerDistance = opts.dividerDistance;
        	this.maxHeight = opts.maxHeight;
        	this.textColor = opts.textColor;
        	this.textSize = opts.textSize;
        	this.typeface = opts.typeface;
        	this.textStyle = opts.textStyle;
        	if(opts.friction > 0){
        		this.friction = opts.friction;
        	}
        	if(opts.selectType >= SELECT_TYPE_DATE_TIME && opts.selectType <= SELECT_TYPE_TIME){
        		mSelectType = opts.selectType;
        	}
        	date = opts.date;
    	}
    	return date;
    }
    
	/**
	 * 初始化控件
	 */
    private final void initViews(Context context, int selectType){
    	if(selectType >= SELECT_TYPE_DATE_TIME && selectType <= SELECT_TYPE_TIME){
    		mSelectType = selectType;
    	}
    	 
        /* 构建布局  */
    	float firstNumberPickerWeight = mSelectType == SELECT_TYPE_DATE_TIME ? 2 : 1;
		LinearLayout layout = new LinearLayout(context);
		layout.setOrientation(LinearLayout.HORIZONTAL);
		layout.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT,
				FrameLayout.LayoutParams.WRAP_CONTENT));

		mNumberPicker1 = new NumberPickerReflect(context);
		setCustom(mNumberPicker1);
		layout.addView(mNumberPicker1,
				new LinearLayout.LayoutParams(0, LinearLayout.LayoutParams.WRAP_CONTENT, firstNumberPickerWeight));
		mNumberPicker2 = new NumberPickerReflect(context);
		setCustom(mNumberPicker2);
		layout.addView(mNumberPicker2, new LinearLayout.LayoutParams(0, LinearLayout.LayoutParams.WRAP_CONTENT, 1));
		mNumberPicker3 = new NumberPickerReflect(context);
		setCustom(mNumberPicker3);
		layout.addView(mNumberPicker3, new LinearLayout.LayoutParams(0, LinearLayout.LayoutParams.WRAP_CONTENT, 1));

		addView(layout);
    }
    
    private void setCustom(NumberPickerReflect numberPicker){
    	// set divider color
   		numberPicker.setDividerColor(dividerColor);
    	// set text color
    	numberPicker.setTextColor(textColor);
    	// set max height
    	if(maxHeight > 0)
    		numberPicker.setMaxHeight(maxHeight);
    	// set divider distance
    	if(dividerDistance > 0)
    		numberPicker.setDividerDistance(dividerDistance);
    	// set text size
    	if(textSize > 0)
    		numberPicker.setTextSize(textSize);
    	// set typeface
    	numberPicker.setTypeface(Typeface.create(FAMILY_NAME[typeface], textStyle));
    	// 增大加速度，这样滚动的快一些，默认是cfg.getScaledMaximumFlingVelocity()/8
    	ViewConfiguration cfg = ViewConfiguration.get(getContext());
    	numberPicker.setMaximumFlingVelocity(cfg.getScaledMaximumFlingVelocity()/friction);
    }
    
    /**
     * 
     * 控件监听器
     */
    private NumberPicker.OnValueChangeListener mOnDateChangedListener = new OnValueChangeListener() {
        @Override
        public void onValueChange(android.widget.NumberPicker picker, int oldVal, int newVal) {
            mDate.add(Calendar.DAY_OF_YEAR, newVal - oldVal);
            updateDateControl();//
            onDateTimeChanged();
        }
    };
    
    private NumberPicker.OnValueChangeListener mOnYearChangedListener = new OnValueChangeListener() {
        @Override
        public void onValueChange(android.widget.NumberPicker picker, int oldVal, int newVal) {
            mDate.add(Calendar.YEAR, newVal - oldVal);
            updateYearControl();//年份改变时，同步修改每月天数
            updateDayControl();
            onDateTimeChanged();
        }
    };
    
    private NumberPicker.OnValueChangeListener mOnMonthChangedListener = new OnValueChangeListener() {
        @Override
        public void onValueChange(android.widget.NumberPicker picker, int oldVal, int newVal) {
        	//预更改月份，判断日期是不是超了，超出的话改用目标月份最大值
        	Calendar cal = Calendar.getInstance();
        	cal.set(Calendar.YEAR, mDate.get(Calendar.YEAR));
        	cal.set(Calendar.MONTH, picker.getValue());
        	int maxDay = cal.getActualMaximum(Calendar.DAY_OF_MONTH);
        	int day = mDate.get(Calendar.DAY_OF_MONTH);
        	if(day > maxDay){
        		mDate.set(Calendar.DAY_OF_MONTH, maxDay);
        	}
        	//更改月份
            mDate.set(Calendar.MONTH, picker.getValue());
            updateDayControl();//月份有变化时，修改每月天数
            onDateTimeChanged();
        }
    };
    
    private NumberPicker.OnValueChangeListener mOnDayChangedListener = new OnValueChangeListener() {
        @Override
        public void onValueChange(android.widget.NumberPicker picker, int oldVal, int newVal) {
            mDate.set(Calendar.DAY_OF_MONTH, picker.getValue());
            onDateTimeChanged();
        }
    };
 
    private NumberPicker.OnValueChangeListener mOnHourChangedListener = new OnValueChangeListener() {
        @Override
        public void onValueChange(android.widget.NumberPicker picker, int oldVal, int newVal) {
            mDate.set(Calendar.HOUR_OF_DAY, picker.getValue());
            onDateTimeChanged();
        }
    };
 
    private NumberPicker.OnValueChangeListener mOnMinuteChangedListener = new OnValueChangeListener() {
        @Override
        public void onValueChange(android.widget.NumberPicker picker, int oldVal, int newVal) {
            mDate.set(Calendar.MINUTE, picker.getValue());
            onDateTimeChanged();
        }
    };
    
    /**日期设置*/
    private void updateDateControl() {//numberpicker1
    	mNumberPicker1.setDisplayedValues(null);
    	mNumberPicker1.setMinValue(0);
        mNumberPicker1.setMaxValue(6);
    	String[] displayValues = new String[7];
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(mDate.getTimeInMillis());
        cal.add(Calendar.DAY_OF_YEAR, -7 / 2);
		for (int i = 0; i < displayValues.length; ++i) {// 一周
			displayValues[i] = (String) DateFormat.format("M月dd日 EEEE", cal);
			cal.add(Calendar.DAY_OF_YEAR, 1);
		}
        mNumberPicker1.setDisplayedValues(displayValues);
        mNumberPicker1.setValue(7 / 2);
        mNumberPicker1.invalidate();
    }
    
    /**年份设置*/
    private void updateYearControl() {//numberpicker1
    	mNumberPicker1.setDisplayedValues(null);
    	mNumberPicker1.setMinValue(0);
        mNumberPicker1.setMaxValue(9);
    	String[] displayValues = new String[10];
    	Calendar cal = Calendar.getInstance();
    	cal.setTimeInMillis(mDate.getTimeInMillis());
    	cal.add(Calendar.YEAR, -5);
    	for (int i = 0; i < displayValues.length; ++i) {// 10年
    		displayValues[i] = (String) DateFormat.format("yyyy年", cal);
    		cal.add(Calendar.YEAR, 1);
    	}
    	mNumberPicker1.setDisplayedValues(displayValues);
    	mNumberPicker1.setValue(5);
    	mNumberPicker1.invalidate();
    }
    
    /**月份设置*/
    private void updateMonthControl(){//numberpicker2
    	mNumberPicker2.setDisplayedValues(null);
    	mNumberPicker2.setMaxValue(11);
        mNumberPicker2.setMinValue(0);
    	String[] displayValues = new String[12];
    	Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(mDate.getTimeInMillis());
        cal.set(Calendar.MONTH, 0);
        for (int i = 0; i < displayValues.length; ++i) {//12个月
            displayValues[i] = (String) DateFormat.format("M月",cal);
            cal.add(Calendar.MONTH, 1);
        }
        mNumberPicker2.setDisplayedValues(displayValues);
        mNumberPicker2.setValue(mDate.get(Calendar.MONTH));
        mNumberPicker2.invalidate();
    }
    
    /**天数设置*/
    private void updateDayControl(){//numberpicker3
    	Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(mDate.getTimeInMillis());
        cal.set(Calendar.DAY_OF_MONTH, 1);
        int maxDay = cal.getActualMaximum(Calendar.DAY_OF_MONTH);
        mNumberPicker3.setDisplayedValues(null);//在setMax之前调用
        mNumberPicker3.setMaxValue(maxDay);
        mNumberPicker3.setMinValue(1);
        String[] displayValues = new String[maxDay];
        for (int i = 0; i < displayValues.length; ++i) {//动态获取每个月天数
            displayValues[i] = (String) DateFormat.format("d日",cal);
            cal.add(Calendar.DAY_OF_MONTH, 1);
        }
        mNumberPicker3.setDisplayedValues(displayValues);
        mNumberPicker3.setValue(mDate.get(Calendar.DAY_OF_MONTH));
        mNumberPicker3.invalidate();
    }
    
    /**小时设置*/
    private void updateHourControl(){//numberpicker2
    	mNumberPicker2.setDisplayedValues(null);
    	mNumberPicker2.setMaxValue(23);
        mNumberPicker2.setMinValue(0);
    	String[] displayValues = new String[24];
    	Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(mDate.getTimeInMillis());
        cal.set(Calendar.HOUR_OF_DAY, 0);
        SimpleDateFormat sdf = new SimpleDateFormat("HH时");
        for (int i = 0; i < displayValues.length; ++i) {//24小时
            displayValues[i] = sdf.format(cal.getTime());
            cal.add(Calendar.HOUR_OF_DAY, 1);
        }
        mNumberPicker2.setDisplayedValues(displayValues);
        mNumberPicker2.setValue(mDate.get(Calendar.HOUR_OF_DAY));
        mNumberPicker2.invalidate();
    }
    
    /**分钟设置*/
    private void updateMinuteControl(){//numberpicker3
    	mNumberPicker3.setDisplayedValues(null);
    	mNumberPicker3.setMaxValue(59);
        mNumberPicker3.setMinValue(0);
    	String[] displayValues = new String[60];//60分钟
    	Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(mDate.getTimeInMillis());
        cal.set(Calendar.MINUTE, 0);
        for (int i = 0; i < displayValues.length; ++i) {
            displayValues[i] = (String) DateFormat.format("mm分",cal);
            cal.add(Calendar.MINUTE, 1);
        }
        mNumberPicker3.setDisplayedValues(displayValues);
        mNumberPicker3.setValue(mDate.get(Calendar.MINUTE));
        mNumberPicker3.invalidate();
    }
    
    private void onDateTimeChanged() {
		if (mOnDateTimeChangedListener != null) {
			mOnDateTimeChangedListener.onDateTimeChanged(this, mDate.get(Calendar.YEAR),
					(mDate.get(Calendar.MONTH)), mDate.get(Calendar.DAY_OF_MONTH), mDate.get(Calendar.HOUR_OF_DAY),
					mDate.get(Calendar.MINUTE));
		}
    }
    
    public static class PickerOptions{
    	/**datetime=0,date=1,time=2*/
    	public int selectType;
    	public long date = System.currentTimeMillis();
    	/**8位，必须带透明度*/@ColorInt
    	public int dividerColor = 0xFF72C8E6;
    	public int dividerDistance;
    	public int maxHeight;
    	/**unit pixel*/
    	public int textSize;
    	/**normal=0,sans=1,serif=2,monospace=3*/
    	public int typeface;
    	/**normal=0,bold=1,italic=2*/
    	public int textStyle;
    	@ColorInt
    	public int textColor = 0xFF000000;
    	/**摩擦力，值越大，滚动越慢*/
    	public int friction = 8;
    	
    	public PickerOptions(){}
    	
    	public PickerOptions(int selectType){
    		this.selectType = selectType;
    	}
    }
    
    /*
     *接口回调 参数是当前的View 年月日小时分钟
     */
    public interface OnDateTimeChangedListener {
        void onDateTimeChanged(DateTimePicker view, int year, int month,
                int day, int hour, int minute);
    }
}