package com.szboanda.android.platform.view;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.Gravity;

import java.util.ArrayList;
import java.util.List;

/**
 * 文本绘制工具，独立出来，使类的职能更清晰
 * Created by Administrator on 2017/6/9.
 */

public class TextDrawer {

    private Canvas mTextCanvas;

    private Bitmap mTextBitmap;

    private String mText;

    private List<String> mBreakLabels = new ArrayList<>();

    private Paint mPaint;

    private int mAvailableWidth;

    private int mMeasureHeight = 0;

    private float mLineSpace = 0;

    private int mHorizontalAlign = Gravity.LEFT;

    public TextDrawer(Context context){
        mPaint = new Paint();
        mPaint.setAntiAlias(true);
        float defaultTextSize = TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_SP,
                16, context.getResources().getDisplayMetrics());
        mPaint.setTextSize(defaultTextSize);
    }

    public TextDrawer text(String text){
        mText = text;
        return this;
    }

    public TextDrawer textSize(float size){
        if(size > 0) {
            mPaint.setTextSize(size);
        }
        return this;
    }

    public TextDrawer textColor(int color){
        mPaint.setColor(color);
        return this;
    }

    public TextDrawer availableWidth(int width){
        mAvailableWidth = width;
        return this;
    }

    public TextDrawer align(int gravity){
        mHorizontalAlign = gravity;
        return this;
    }

    public int getTextWidth(){
        if(mBreakLabels.size() > 0) {
            return (int) mPaint.measureText(mBreakLabels.get(0));
        }else{
            return 0;
        }
    }

    public int getMeasureHeight(){
        return mMeasureHeight;
    }

    public void measure(){
        if(!TextUtils.isEmpty(mText)) {
            breakLabel(mText);
            mTextBitmap = Bitmap.createBitmap(getTextWidth(), mMeasureHeight, Bitmap.Config.ARGB_4444);
            mTextCanvas = new Canvas(mTextBitmap);
        }
    }

    public boolean isBreaked(){
        return mPaint.measureText(mText) > mAvailableWidth;
    }

    private void breakLabel(String label){
        int index = mPaint.breakText(label, true, mAvailableWidth, null);
        if(index <= label.length()){
            String lineLabel = label.substring(0, index);
            mBreakLabels.add(lineLabel);
            mMeasureHeight += getTextHeight() + mLineSpace;
            String residualLabel = label.substring(index, label.length());
            if(!TextUtils.isEmpty(residualLabel)){
                breakLabel(residualLabel);
            }else{
                mMeasureHeight -= mLineSpace;
            }
        }
    }

    /**
     * 获取文字高度
     * @return
     */
    private int getTextHeight(){
        Paint.FontMetrics fontMetrics = mPaint.getFontMetrics();
        return (int)(fontMetrics.bottom + Math.abs(fontMetrics.top));
    }

    public Bitmap draw(){
        if(mBreakLabels.size() > 0){
            float y = getTextHeight() - mPaint.getFontMetrics().bottom;
            for(String label : mBreakLabels){
                int x = mHorizontalAlign == Gravity.LEFT ? 0 : getTextWidth() - (int) mPaint.measureText(label);
                mTextCanvas.drawText(label, x, y, mPaint);
                y += getTextHeight() + mLineSpace;
            }
        }
        return mTextBitmap;
    }
}
