package com.szboanda.android.platform.view;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Paint.Cap;
import android.graphics.Paint.Join;
import android.graphics.Paint.Style;
import android.graphics.Path;
import android.graphics.Path.Direction;
import android.graphics.PorterDuff.Mode;
import android.graphics.PorterDuffXfermode;
import android.graphics.RectF;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.MotionEvent;
import android.widget.Button;

import com.szboanda.android.platform.R;

/**
 * <p>Description: 处定义纯色背景按钮，封装了设置正常状态，
 * 点击效果，禁用效果的纯色背景、任意角的圆角</p>
 * @author    Siyi Lu    Create Date: 2015-8-28
 */
public class PureColorButton extends Button{

	int bgColor = Color.BLUE;
	
	int disableColor = Color.BLUE;
	
	int normalColor = Color.BLUE;
	
	int pressColor = Color.BLUE;
	
	private Drawable mBgDrawable;
	
	private Bitmap mBgImage;
	
	public PureColorButton(Context context, AttributeSet attrs) {
		super(context, attrs);
		initDrawMaterial(attrs);
	}

	public PureColorButton(Context context) {
		super(context);
		initDrawMaterial(null);
	}
	
	public void setBgColor(int bgColor) {
		this.bgColor = bgColor;
		this.normalColor = bgColor;
	}
	
	public void setPressColor(int pressColor) {
		this.pressColor = pressColor;
	}
	
	public void setBackgroundDrawable(Drawable drawable){
		this.mBgDrawable = drawable;
	}
	
	public void setDisableColr(int disableColr){
		this.disableColor = disableColr;
	}

	public void setRadius(float radius) {
		this.mRadius = radius;
	}
	
	public void setBgDrawable(Drawable bgDrawable){
		this.mBgDrawable = bgDrawable;
	}

	RectF mBgColorRect;
	Paint mPaint = null;
	Path mPath = null;
	float mRadius; 
	float topLeftRadius;
	float topRightRadius;
	float bottomLeftRadius;
	float bottomRightRadius;
	private void initDrawMaterial(AttributeSet attrs){
		setBackgroundColor(0x00000000);
		normalColor = 0xFF999999;
		disableColor = 0xFFCCCCCC;
		pressColor = 0xFF666666;
		float defaultRadius = 0;
		mRadius 						= TypedValue.applyDimension(
				TypedValue.COMPLEX_UNIT_DIP, 3, getResources().getDisplayMetrics());
		topLeftRadius 				= defaultRadius;
		topRightRadius 			= defaultRadius;
		bottomLeftRadius 		= defaultRadius;
		bottomRightRadius 		= defaultRadius;
		if(attrs != null){
			TypedArray ta 				= getContext().obtainStyledAttributes(attrs, R.styleable.PureColorButton);
			normalColor 					= ta.getColor(R.styleable.PureColorButton_backgroundColor, normalColor);
			pressColor 					= ta.getColor(R.styleable.PureColorButton_pressColor, pressColor);
			disableColor 				= ta.getColor(R.styleable.PureColorButton_disableColor, disableColor);
			mBgDrawable 				= ta.getDrawable(R.styleable.PureColorButton_backgroundDrawable);
			mRadius 							= ta.getDimension(R.styleable.PureColorButton_radius, mRadius);
			topLeftRadius 				= ta.getDimension(R.styleable.PureColorButton_topLeftRadius, defaultRadius);
			topRightRadius 			= ta.getDimension(R.styleable.PureColorButton_topRightRadius, defaultRadius);
			bottomLeftRadius 		= ta.getDimension(R.styleable.PureColorButton_bottomLeftRadius, defaultRadius);
			bottomRightRadius 	= ta.getDimension(R.styleable.PureColorButton_bottomRightRadius, defaultRadius);
			ta.recycle();
		}
		bgColor = normalColor;
		if(!isEnabled()){
			bgColor = disableColor;
		}
		mBgColorRect = new RectF();
		mPaint = new Paint();
		mPaint.setAntiAlias(true);
		mPaint.setStrokeCap(Cap.ROUND);
		mPaint.setStrokeJoin(Join.ROUND);
		mPaint.setStyle(Style.FILL_AND_STROKE);
		mPaint.setXfermode(new PorterDuffXfermode(Mode.SRC_ATOP));
		mPath = new Path();
	}
	
	
	public void setTopLeftRadius(float topLeftRadius) {
		this.topLeftRadius = topLeftRadius;
	}

	public void setTopRightRadius(float topRightRadius) {
		this.topRightRadius = topRightRadius;
	}

	public void setBottomLeftRadius(float bottomLeftRadius) {
		this.bottomLeftRadius = bottomLeftRadius;
	}

	public void setBottomRightRadius(float bottomRightRadius) {
		this.bottomRightRadius = bottomRightRadius;
	}
	
	public void setRadiusAll(float topLeftRadius, float topRightRadius
			,float bottomLeftRadius, float bottomRightRadius){
		this.topLeftRadius = topLeftRadius;
		this.topRightRadius = topRightRadius;
		this.bottomLeftRadius = bottomLeftRadius;
		this.bottomRightRadius = bottomRightRadius;
	}

	@Override
	protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
		super.onMeasure(widthMeasureSpec, heightMeasureSpec);
		mBgColorRect.set(1, 1, getMeasuredWidth()-1, getMeasuredHeight()-1);
	}
	
	@Override
	protected void onDraw(Canvas canvas) {
		mPaint.setColor(bgColor);
		if((topLeftRadius + topRightRadius 
					+ bottomLeftRadius + bottomRightRadius) > 0){
			joinCornersPointsLine(canvas);
			canvas.drawPath(mPath, mPaint);
			drawCorners(canvas);
		}else{
			canvas.drawRoundRect(mBgColorRect, mRadius, mRadius, mPaint);
		}
		canvas.save();
		
		if(mBgDrawable != null){
			if(mBgDrawable instanceof BitmapDrawable){
				mBgImage = ((BitmapDrawable)mBgDrawable).getBitmap();
			}
		}
		if(mBgImage != null){
			int left = getMeasuredWidth()/2 - mBgImage.getWidth()/2;
			int top = getMeasuredHeight()/2 - mBgImage.getHeight()/2;
			canvas.drawBitmap(mBgImage, left, top, mPaint);
		}
		translateDrawableToCenter(canvas);
		super.onDraw(canvas);
	}
	
	private void translateDrawableToCenter(Canvas canvas){
		canvas.save();
		Drawable[] drawables = getCompoundDrawables();
		if(drawables != null){
			Drawable leftD = drawables[0];
			Drawable topD = drawables[1];
			Drawable rightD = drawables[2];
			Drawable bottomD = drawables[3];
			translateLeftDrawable(canvas, leftD);
			translateTopDrawable(canvas, topD);
			translateRightDrawable(canvas, rightD);
			translateBottomDrawable(canvas, bottomD);
		}
	}
	
	private void translateLeftDrawable(Canvas canvas, Drawable leftDrawable){
		if(leftDrawable != null){
			float textWidth = getPaint().measureText(getText().toString());
            int drawablePadding = getCompoundDrawablePadding();
            int drawableWidth = 0;
            drawableWidth = leftDrawable.getIntrinsicWidth();
            float bodyWidth = textWidth + drawableWidth + drawablePadding;
            canvas.translate((getWidth() - bodyWidth) / 2, 0);
		}
	}
	
	private void translateTopDrawable(Canvas canvas, Drawable topDrawable){
		if(topDrawable != null){
			int lineCount = getLineCount();
			float th = getLineHeight() * lineCount;
			int drawableHeight = topDrawable.getIntrinsicHeight();
			float adjust = TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, 10, getResources().getDisplayMetrics());
			float contentHeight = th + drawableHeight + getCompoundDrawablePadding() +adjust;
			canvas.translate(0, (getHeight()-contentHeight)/2);
			setGravity(Gravity.CENTER_HORIZONTAL | Gravity.TOP);
		}
	}
	
	private void translateRightDrawable(Canvas canvas, Drawable rightDrawable){
		if(rightDrawable != null){
			
		}
	}
	
	private void translateBottomDrawable(Canvas canvas, Drawable bottomDrawable){
		if(bottomDrawable != null){
			
		}
	}
	
	/**
	 * 使用连接点的方式绘制除圆角外的区域
	 * @param canvas
	 */
	private void joinCornersPointsLine(Canvas canvas){
		float left;
		float top;
		float right;
		float bottom;
		mPath.reset();
		mPath.moveTo(0, 0);
		if(topLeftRadius != 0){
			mPath.lineTo(0, topLeftRadius);
			mPath.lineTo(topLeftRadius, topLeftRadius);
			mPath.lineTo(topLeftRadius, 0);
		}else{
			mPath.lineTo(0, 0);
		}
		if(topRightRadius != 0){
			left = getMeasuredWidth() -topRightRadius;
			top = 0;
			right = getMeasuredWidth();
			bottom = topRightRadius;
			mPath.lineTo(left, top);
			mPath.lineTo(left, bottom);
			mPath.lineTo(right, bottom);
		}else{
			mPath.lineTo(getMeasuredWidth(), 0);
		}
		
		if(bottomRightRadius != 0){
			left = getMeasuredWidth() - bottomRightRadius;
			top = getMeasuredHeight() - bottomRightRadius;
			right = getMeasuredWidth();
			bottom = getMeasuredHeight();
			mPath.lineTo(right, top);
			mPath.lineTo(left, top);
			mPath.lineTo(left, bottom);
		}else{
			mPath.lineTo(getMeasuredWidth(), getMeasuredHeight());
		}
		
		if(bottomLeftRadius != 0){
			left = 0;
			top = getMeasuredHeight() - bottomLeftRadius;
			right = bottomLeftRadius;
			bottom = getMeasuredHeight();
			mPath.lineTo(right, bottom);
			mPath.lineTo(right, top);
			mPath.lineTo(left, top);
		}else{
			mPath.lineTo(0, getMeasuredHeight());
		}
	}
	
	/**
	 * 绘制各个圆角
	 * @param canvas
	 */
	private void drawCorners(Canvas canvas){
		mPath.reset();
		if(topLeftRadius != 0){
			addTopLeftPath(canvas, mPaint, topLeftRadius);
		}
		if(topRightRadius != 0){
			addTopRight(canvas, mPaint, topRightRadius);
		}
		if(bottomLeftRadius != 0){
			addBottomLeft(canvas, mPaint, bottomLeftRadius);
		}
		if(bottomRightRadius != 0){
			addBottomRight(canvas, mPaint, bottomRightRadius);
		}
		canvas.drawPath(mPath, mPaint);
		canvas.save();
	}
	
	private void addTopLeftPath(Canvas canvas, Paint paint, float radius){
    	float x = radius;
    	float y = radius;
    	mPath.addCircle(x, y, radius, Direction.CCW);
    }
     
    private void addTopRight(Canvas canvas, Paint paint, float radius){
    	float x = getMeasuredWidth() - radius;
    	float y = radius;
    	mPath.addCircle(x, y, radius, Direction.CCW);
    }
     
    private void addBottomLeft(Canvas canvas,Paint paint, float radius){
    	float x = radius;
    	float y = getMeasuredHeight() - radius;
    	mPath.addCircle(x, y, radius, Direction.CCW);
    }
     
    private void addBottomRight(Canvas canvas, Paint paint, float radius){
    	float x = getMeasuredWidth() - radius;
    	float y = getMeasuredHeight() - radius;
    	mPath.addCircle(x, y, radius, Direction.CCW);
    }
	
	@Override
	public boolean onTouchEvent(MotionEvent event) {
		if(isEnabled() && isClickable()){
			int code = event.getAction();
			switch (code) {
			case MotionEvent.ACTION_DOWN:
				bgColor = pressColor;
				break;
			case MotionEvent.ACTION_UP:
				bgColor = normalColor;
				break;
			case MotionEvent.ACTION_CANCEL:
				bgColor = normalColor;
				break;
			default:
				break;
			}
			invalidate();
		}
		return super.onTouchEvent(event);
	}
	
	@Override
	public void setEnabled(boolean enabled) {
		super.setEnabled(enabled);
		onEnableChanged();
	}
	
	private void onEnableChanged(){
		bgColor = isEnabled() ? normalColor : disableColor;
	}
	
}
