package com.szboanda.android.platform.view;

import com.szboanda.android.platform.util.BitmapUtils;
import com.szboanda.android.platform.util.BitmapUtils.ArrowDirection;
import com.szboanda.android.platform.util.DimensionUtils;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Bitmap.Config;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Typeface;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.ImageView.ScaleType;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import java.util.Collections;

import static android.R.attr.width;

/**
 * <p>Description: 带标题的视图容器</p>
 * @author    Siyi Lu    Create Date: 2015-7-30
 */
public class LinearLayoutWithTitle extends LinearLayout
	implements View.OnClickListener{
	
	/**标题左侧图标*/
	private ImageView imageLeftIndicator;
	
	/**标题右侧图标*/
	private ImageView imageRightIndicator;
	
	/**标题*/
	private TextView txtTitle;
	
	/**图标的默认宽、高尺寸*/
	private int defaultIndicatorDimens;
	
	private LinearLayout mChildsParent;
	
	private boolean isChildsParentIndent = true;
	
	/**子节点视图是否可收缩*/
	private boolean isCollapseable = true;
	
	/**子节点视图是否已收缩*/
	private boolean isExpanded = true;
	
	public LinearLayoutWithTitle(Context context, AttributeSet attrs) {
		super(context, attrs);
		init();
	}

	public LinearLayoutWithTitle(Context context) {
		super(context);
		init();
	}
	
	private void init(){
		defaultIndicatorDimens = DimensionUtils.dip2Px(getContext(), 26);
		setWillNotDraw(false);
		setOrientation(VERTICAL);
		initDrawMaterial();
		initTitle();
		initContentParent();
	}
	
	RelativeLayout titleContainer = null;
	
	Bitmap indicatorExpanded = null;
	
	Bitmap indicatorCollapseed = null;
	
	
	/**初始化标题*/
	private void initTitle(){
		int titleHeight = DimensionUtils.dip2Px(getContext(), 42);
		titleContainer = new RelativeLayout(getContext());
		LayoutParams lp1 = new LayoutParams(LayoutParams.MATCH_PARENT, titleHeight);
		titleContainer.setLayoutParams(lp1);
		int padding = DimensionUtils.dip2Px(getContext(), 5);
		titleContainer.setPadding(0, 0, padding, 0);
		titleContainer.setOnClickListener(isCollapseable ? this : null);
		addView(titleContainer);
		
		int leftIndicatorId = 1;
		imageLeftIndicator = new ImageView(getContext());
		imageLeftIndicator.setId(leftIndicatorId);
		imageLeftIndicator.setScaleType(ScaleType.FIT_CENTER);
		RelativeLayout.LayoutParams lp2 = new RelativeLayout.LayoutParams(
				defaultIndicatorDimens, defaultIndicatorDimens);
		lp2.addRule(RelativeLayout.CENTER_VERTICAL);
		imageLeftIndicator.setLayoutParams(lp2);
		titleContainer.addView(imageLeftIndicator);
		drawDot();
		imageLeftIndicator.setImageBitmap(dot);
		
		int titleId = 2; 
		txtTitle = new TextView(getContext());
		txtTitle.setId(titleId);
		txtTitle.setTextSize(18);
		txtTitle.setTextColor(0xFF3551AE);
		txtTitle.setTypeface(Typeface.DEFAULT_BOLD);
		RelativeLayout.LayoutParams lp3 = new RelativeLayout.LayoutParams(
				LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
		lp3.addRule(RelativeLayout.RIGHT_OF, 1);
		lp3.addRule(RelativeLayout.LEFT_OF, 3);
		txtTitle.setGravity(Gravity.CENTER_VERTICAL);
		txtTitle.setLayoutParams(lp3);
		titleContainer.addView(txtTitle);
		 
		int rightIndicatorId = 3;
		imageRightIndicator = new ImageView(getContext());
		imageRightIndicator.setId(rightIndicatorId);
		RelativeLayout.LayoutParams lp4 = new RelativeLayout.LayoutParams(
				LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
		lp4.addRule(RelativeLayout.CENTER_VERTICAL);
		lp4.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
		lp4.rightMargin = DimensionUtils.dip2Px(getContext(), 8);
		imageRightIndicator.setLayoutParams(lp4);
		imageRightIndicator.setScaleType(ScaleType.FIT_CENTER);
		titleContainer.addView(imageRightIndicator);
		
		setRightIndicator(isExpanded);
	}
	
	public void setCollapseable(boolean collapseable){
		isCollapseable = collapseable;
	}
	
	/**
	 * 所有内容视图装在一下单独的LinearLayout中
	 */
	private void initContentParent(){
		mChildsParent = new LinearLayout(getContext());
		LayoutParams lp = new LayoutParams(
				LayoutParams.MATCH_PARENT,LayoutParams.WRAP_CONTENT);
		mChildsParent.setLayoutParams(lp);
		mChildsParent.setOrientation(VERTICAL);
		addView(mChildsParent);
	}
	
	/**
	 * 添加内容视图统一使用这个方法，不使用{@link ViewGroup#addView(View)}方法
	 * 因为内容视图的内容需要统一与标题对齐
	 * @param child
	 */
	public void addChild(View child){
		mChildsParent.addView(child);
	}
	
	public void removeAllChilds(){
		mChildsParent.removeAllViews();
	}
	
	public void setTitle(String title){
		txtTitle.setText(title);
	}
	
	public void setTitleColor(int color){
		txtTitle.setTextColor(color);
	}
	
	public TextView getTitleView(){
		return txtTitle;
	}
	
	public ImageView getLeftIndicator(){
		return imageLeftIndicator;
	}
	
	public ImageView getRightIndicator(){
		return imageRightIndicator;
	}
	
	public View getChildsParent(){
		return mChildsParent;
	}
	
	public RelativeLayout getTitleContainer(){
		return titleContainer;
	}
	
	/**
	 * 收缩子视图
	 */
	public void collapseChilds(){
		if(isExpanded){
			isExpanded = false;
			mChildsParent.setVisibility(
					isExpanded ? View.VISIBLE:View.GONE);
			setRightIndicator(isExpanded);
		}
	}
	
	/**
	 * 展开子视图
	 */
	public void expandChilds(){
		if(!isExpanded){
			isExpanded = true;
			mChildsParent.setVisibility(
					isExpanded ? View.VISIBLE:View.GONE);
			setRightIndicator(isExpanded);
		}
	}

	public void setChildsParentIndent(boolean indent){
		isChildsParentIndent = indent;
	}

	public void setTitleContainerHeight(int height) {
		LayoutParams layoutParams = (LayoutParams)titleContainer.getLayoutParams();
		layoutParams.height = DimensionUtils.dip2Px(getContext(), height);
		titleContainer.setLayoutParams(layoutParams);
	}

	public void setTitleLeftMargin(int margin){
		RelativeLayout.LayoutParams lp = (RelativeLayout.LayoutParams)txtTitle.getLayoutParams();
		lp.leftMargin = margin;
	}
	
	@Override
	protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
		super.onMeasure(widthMeasureSpec, heightMeasureSpec);
		if(isExpanded && mChildsParent != null){
			int indentWidth = imageLeftIndicator.getMeasuredWidth()
					/(isChildsParentIndent ? 1 : 2);
			//因为子视图需要重新布局所以需要减掉缩进的长度
			measureChild(mChildsParent
					,widthMeasureSpec - indentWidth
					,heightMeasureSpec);
			int titleHeight = txtTitle.getMeasuredHeight();
			int contentHeight = mChildsParent.getMeasuredHeight();
			setMeasuredDimension(getMeasuredWidth()
					,titleHeight + contentHeight + getPaddingTop() + getPaddingBottom());
		}
	}
	
	@Override
	protected void onLayout(boolean changed, int l, int t, int r, int b) {
		super.onLayout(changed, l, t, r, b);
		if(isExpanded && mChildsParent != null){//使子视图与标题文本左对齐
			int left = isChildsParentIndent ? (int)txtTitle.getX() : imageLeftIndicator.getMeasuredWidth()/2;
			mChildsParent.layout(
					left, txtTitle.getHeight(), r
					,txtTitle.getHeight()+mChildsParent.getMeasuredHeight());
		}
	}
	
	@Override
	protected void onDraw(Canvas canvas) {
		super.onDraw(canvas);
		if(isExpanded && mChildsParent.getChildCount() > 0){
			drawDivideLine(canvas);
		}
	}
	
	public void setRightIndicatorVisable(boolean isVisable){
		imageRightIndicator.setVisibility(
				isVisable || isCollapseable ? View.VISIBLE:View.GONE);
	}
	
	/**
	 * 初始化绘制相关的画笔，位图尺寸
	 */
	private void initDrawMaterial(){
		dot = Bitmap.createBitmap(defaultIndicatorDimens, defaultIndicatorDimens, Config.ARGB_8888);
		dot.eraseColor(Color.TRANSPARENT);
		mPaint = new Paint();
		mPaint.setStyle(Paint.Style.FILL);
		mPaint.setAntiAlias(true);
		mPaint.setColor(0x88FF0000);
	}
	
	Bitmap dot = null;
	Paint mPaint = null;
	
	/**
	 * 绘制标题与内容之间的分割线
	 * @param canvas
	 */
	private void drawDivideLine(Canvas canvas){
		mPaint.setColor(Color.LTGRAY);
		mPaint.setStrokeWidth(3);
		canvas.drawLine(
				txtTitle.getX(),                 		//起始点X坐标以标题文本左侧开始
				txtTitle.getHeight()-1,       	//起始点Y坐标以标题文本底部开始，因为标题文本控件与标题行同高
				getWidth(), 							//结束点在本视图最右侧
				txtTitle.getHeight()-1			
				,mPaint);
	}
	
	/**
	 * 绘制标题左侧imageview默认填充的图标
	 */
	private void drawDot(){
		//使用dot的尺寸创建画布
		Canvas c = new Canvas(dot);
		int radius = defaultIndicatorDimens/2;
		//drawCircle的前两个参数指是绘制的圆心中心点，是相对于画布的尺寸而定的
		//先在画布上绘制图形
		c.drawCircle(radius, radius, radius/2, mPaint);
	}
	
	@Override
	public void onClick(View v) {
		if(isCollapseable && mChildsParent.getChildCount() > 0){
			isExpanded = !isExpanded;
			mChildsParent.setVisibility(
					isExpanded ? View.VISIBLE:View.GONE);
			setRightIndicator(isExpanded);
		}
	}
	
	public void setRightIndicator(boolean expanded){
		if(isCollapseable){
			if(indicatorExpanded == null){
				indicatorExpanded = BitmapUtils.drawArrow(getContext()
					,DimensionUtils.dip2Px(getContext(), 20), Color.GRAY, ArrowDirection.TOP);
				indicatorCollapseed = BitmapUtils.drawArrow(getContext()
					,DimensionUtils.dip2Px(getContext(), 20), Color.GRAY, ArrowDirection.BOTTOM);
			}
			imageRightIndicator.setImageBitmap(
					isExpanded ? indicatorExpanded : indicatorCollapseed);
		}
	}
	
	public void setRightIndicator(Bitmap indicator){
		imageRightIndicator.setImageBitmap(indicator);
	}
}
