package com.szboanda.android.platform.view;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Paint.Align;
import android.graphics.Paint.FontMetrics;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.View;

import java.util.ArrayList;
import java.util.List;

public class LabelDrawer {
	
private static final int DEFAULT_LABEL_COLOR = 0xFF666666;
	
	/**标签文本*/
	private String mLabel = "";
	
	private int mLabelTextColor = DEFAULT_LABEL_COLOR;
	
	/**标签跨行时，第一行的文件集合*/
	private List<String> mBreakedLabels;
	
	/**通过{@link View#setPadding(int, int, int, int)}设置的视图左内边距*/
	private int mOriginalPaddingLeft = 0;
	private int mOriginalPaddingTop = 0;
	private int mOriginalPaddingRight = 0;
	private int mOriginalPaddingBottom = 0;
	
	/**通过{@link #setText(CharSequence)}设置的文本在计算标签宽度后的左内边距*/
	private int adjustPadding = 0;
	
	/**标签水平对齐方式*/
	private boolean isHorizontalAlignRight  = false;
	
	/**标签是否垂直居中对齐*/
	private boolean isLabelAlignVerticalCenter = true;
	
	/**标签占整个控件宽度的权重，以百分比计，如值为30表示占30%，
	 * 默认值为-1，表示自适应标签文本宽度*/
	private int labelWeight = -1;
	
	/**标签与内容的间距*/
	private int widthBetweenLableText = 0;
	
	private int mLabelPaddingLeft = 0;
	
	/**绘制标签的画笔*/
	private Paint mLabelPaint = null;
	
	/**是否已经测量标签宽度*/
	boolean hasMeasureLabel;
	
	int mLabelHeight;
	
	Context mContext;
	
	public LabelDrawer(){
		init();
	}
	
	public LabelDrawer(Context context){
		mContext = context;
		init();
	}
	
	public LabelDrawer(String label){
		mLabel = label;
		init();
	}
	
	private void init(){
		
		mLabelPaint = new Paint();
		mLabelPaint.setColor(mLabelTextColor);
		mLabelPaint.setAntiAlias(true);
		mLabelPaint.setTextAlign(Align.LEFT);
		if(mContext == null){
			mLabelPaint.setTextSize(32);
		}else{
			float labelSize = TypedValue.applyDimension(
					TypedValue.COMPLEX_UNIT_SP, 16, mContext.getResources().getDisplayMetrics());
			setLabelSize(labelSize);
		}
	}
	
	public void setLabel(String label){
		mLabel = label;
	}
	
	public String getLabel(){
		return mLabel;
	}
	
	public void setLabelSize(float size){
		mLabelPaint.setTextSize(size);
	}
	
	public void setLableTextColor(int color){
		mLabelTextColor = color;
		mLabelPaint.setColor(color);
	}
	
	public int getLableTextColor(){
		return mLabelTextColor;
	}
	
	public void setLabelWeight(int labelWeight) {
		this.labelWeight = labelWeight > 50 ? 50 : labelWeight;
	}

	public void setLabelPaddingLeft(int padding){
		mLabelPaddingLeft = padding;
	}
	
	public void setLabelMarginRight(int margin){
		widthBetweenLableText = margin;
	}
	
	public void setHorizontalAlignRight(boolean isHorizontalAlignRight) {
		this.isHorizontalAlignRight = isHorizontalAlignRight;
	}

	public void setLabelAlignVerticalCenter(boolean isLabelAlignVerticalCenter) {
		this.isLabelAlignVerticalCenter = isLabelAlignVerticalCenter;
	}
	
	public int getAdjustedPadding(){
		return adjustPadding;
	}
	
	public int getOriginalPaddingLeft(){
		return mOriginalPaddingLeft;
	}
	
	public float getLabelTextSize(){
		return mLabelPaint.getTextSize();
	}
	
	public int measureLabel(View view, int measureWidth, int measureHeight){
		if(!hasMeasureLabel){
			mOriginalPaddingLeft = view.getPaddingLeft();
			mOriginalPaddingTop = view.getPaddingTop();
			mOriginalPaddingRight = view.getPaddingRight();
			mOriginalPaddingBottom = view.getPaddingBottom();
			caculateAdustPadding(measureWidth);
			view.setPadding(adjustPadding, mOriginalPaddingTop, mOriginalPaddingRight, mOriginalPaddingBottom);
			breakLabel(mLabel);
			hasMeasureLabel = true;
		}
		mLabelHeight = mLabelHeightSum + mOriginalPaddingTop + mOriginalPaddingBottom;
		mLabelHeight = Math.max(measureHeight, mLabelHeight);
		return mLabelHeight;
	}
	
	/**
	 * 根据标签的宽度计算需要设置的内边距，以留出绘制标签的空间
	 * @return
	 */
	private int caculateAdustPadding(int measureWidth){
		
		if(labelWeight == -1){
			int labelWidth = (int)mLabelPaint.measureText(mLabel);
			adjustPadding = mLabelPaddingLeft + labelWidth + widthBetweenLableText + mOriginalPaddingLeft;
		}else{
			adjustPadding = measureWidth * labelWeight/100 + mOriginalPaddingLeft;
		}
		return adjustPadding;
	}
	
	/**单行标签的高度*/
	private int mSingleLineLabelHeight;
	/**标签的总高度*/
	private int mLabelHeightSum;
	/**行距*/
	private int mLineSpace;
	
	/**
	 * 当设置了{@link #labelWeight}属性后，过长的标签需要多行绘制
	 * 所以需要对标签进行分割
	 * @param label
	 */
	private void breakLabel(String label){
		int residualWidth = adjustPadding - mLabelPaddingLeft - widthBetweenLableText - mOriginalPaddingLeft;
		int index = mLabelPaint.breakText(label, true, residualWidth, null);
		FontMetrics fontMetrics = mLabelPaint.getFontMetrics();
		//单行标签的高度
		mSingleLineLabelHeight = (int)(fontMetrics.bottom + Math.abs(fontMetrics.top));
		if(index <= label.length()){
			String lineLabel = label.substring(0, index);
			if(mBreakedLabels == null){
				mBreakedLabels = new ArrayList<String>();
			}
			mBreakedLabels.add(lineLabel);
			mLabelHeightSum += (mSingleLineLabelHeight + mLineSpace);
			String residualLabel = label.substring(index, label.length());
			if(!TextUtils.isEmpty(residualLabel)){
				breakLabel(residualLabel);
			}else{
				mLabelHeightSum -= mLineSpace;
			}
		}
	}
	
	/**绘制标签*/
	public void drawLabel(Canvas canvas){
		int x = caculateX();
		int y;
		int lineNum = 0;
		for(String label : mBreakedLabels){
			y = caculateY(lineNum);
			canvas.drawText(label, x, y, mLabelPaint);
			lineNum++;
		}
	}
	
	/**计算绘制标签文本的x坐标*/
	private int caculateX(){
		int x;
		if(labelWeight != -1 && isHorizontalAlignRight){
			mLabelPaint.setTextAlign(Align.RIGHT);
			x = adjustPadding - mOriginalPaddingLeft - widthBetweenLableText;
		}else{
			x = mLabelPaddingLeft;
		}
		return x;
	}
	
	private int caculateY(int lineNum){
		FontMetrics fontMetrics = mLabelPaint.getFontMetrics();
		int y = 0;
		//第一行标签顶部Y值起点
		int startY;
		if(isLabelAlignVerticalCenter){
			//第一行标签顶部Y值起点，标签垂直居中时，使用中心偏移来计算
			startY = mLabelHeight/2 - mLabelHeightSum/2;
		}else{
			startY = mOriginalPaddingTop;
		}
		startY += 1;
		//加上行高
		y += (startY + lineNum * mSingleLineLabelHeight + (int)(Math.abs(fontMetrics.top)));
		//加上行距
		y += lineNum * mLineSpace ;
		return y;
	}

}
