package com.szboanda.android.platform.view;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import android.content.Context;
import android.content.res.TypedArray;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.View;
import android.widget.LinearLayout;

import com.szboanda.android.platform.R;
import com.szboanda.android.platform.util.DateUtils;
import com.szboanda.android.platform.util.DimensionUtils;
import com.szboanda.android.platform.view.wheel.StringWheelAdapter;
import com.szboanda.android.platform.view.wheel.OnWheelChangedListener;
import com.szboanda.android.platform.view.wheel.WheelView;

/**
 * <p>Description: 日期，时间选择控件</p>
 * @Company    深圳市博安达信息技术股份有限公司
 * @author    Siyi Lu    Create Date: 2015-8-14
 * @modify SuHao 2016-05-19 19:33:32
 * @deprecated  
 */
public class DateTimePicker extends LinearLayout{
	
	public final static String UNIT_YEAR = "年";
	public final static String UNIT_MONTH = "月";
	public final static String UNIT_DAY = "日";
	public final static String UNIT_HOUR = "时";
	public final static String UNIT_MIN = "分";
	
	Calendar mCalendar;
	
	PickType mPickType;
	/**年份滚动视图索引*/
	int yearViewIndex;
	/**月份滚动视图索引*/
	int monthViewIndex;
	/**日期滚动视图索引*/
	int dayViewIndex;
	/**小时滚动视图索引*/
	int hourViewIndex;
	/**分钟滚动视图索引*/
	int minViewIndex;
	
	StringWheelAdapter yearAdapter;
	StringWheelAdapter monthAdapter;
	StringWheelAdapter dayAdapter;
	StringWheelAdapter hourAdapter;
	StringWheelAdapter minAdapter;
	
	private OnCalenderChangeListener mChangeListener;
	
	public DateTimePicker(Context context, AttributeSet attrs) {
		super(context, attrs);
		TypedArray ta = context.obtainStyledAttributes(attrs, R.styleable.DateTimePicker);
		int pickTypeIndex = ta.getInt(R.styleable.DateTimePicker_select_type, 0);
		mPickType = PickType.values()[pickTypeIndex];
		ta.recycle();
		init();
	}

	public DateTimePicker(Context context, PickType pickType) {
		super(context);
		mPickType = pickType;
		init();
	}
	
	private void init(){
		setOrientation(HORIZONTAL);
		mCalendar = Calendar.getInstance(Locale.getDefault());
		createView();
		initData();
		initListener();
	}
	
	private void createView(){
		int colCount = 0;
		switch (mPickType) {
		case TIME:
			colCount = 2;
			hourViewIndex = 0;
			minViewIndex = 1;
			break;
		case DATE:
			colCount = 3;
			yearViewIndex = 0;
			monthViewIndex = 1;
			dayViewIndex = 2;
			break;
		case DATE_TIME:
			colCount = 5;
			yearViewIndex = 0;
			monthViewIndex = 1;
			dayViewIndex = 2;
			hourViewIndex = 3;
			minViewIndex = 4;
			break;
		default:
			break;
		}
		for(int i=1; i<=colCount; i++){
			View colView = createSingleColumn(i);
			addView(colView);
		}
	}
	
	/**创建单列滚动视图*/
	private View createSingleColumn(int colIndex){
		int h = DimensionUtils.getScreenHeight(getContext(), 30);
		LayoutParams lp = new LayoutParams(0, h);
		lp.weight = 1;
		WheelView wv = new WheelView(getContext());
		wv.setId(colIndex);
		wv.setLayoutParams(lp);
		return wv;
	}
	
	public void setDate(Date date){
		mCalendar.setTime(date);
		initData();
	}
	
	public void initData(){
		switch (mPickType) {
		case TIME:
			setHourColumnData();
			setMinColumnData();
			break;
		case DATE:
			setYearColumnData();
			setMonthColumnData();
			setDayColumnData();
			break;
		case DATE_TIME:
			setYearColumnData();
			setMonthColumnData();
			setDayColumnData();
			setHourColumnData();
			setMinColumnData();
			break;
		default:
			break;
		}
	}
	
	public Calendar getDate(){
		return mCalendar;
	}
	
	/**获取格式化后的日期、时间字符串*/
	public String getFormatValue(){
		String result = "";
		switch (mPickType) {
		case TIME:
			result = DateUtils.formatDate(mCalendar.getTime(), DateUtils.FORMAT_TIME_HM);
			break;
		case DATE:
			result = DateUtils.formatDate(mCalendar.getTime(), DateUtils.FORMAT_DATE_DASH);
			break;
		case DATE_TIME:
			result = DateUtils.formatDate(mCalendar.getTime(), DateUtils.FORMAT_DATE_TIME_M);
			break;
		default:
			break;
		}
		return result;
	}
	
	private void setYearColumnData(){
		int curYear = mCalendar.get(Calendar.YEAR);
		int startY = curYear -100;
		int endY = curYear + 100;
		WheelView viewYear = (WheelView)getChildAt(yearViewIndex);
		yearAdapter = new StringWheelAdapter(getContext(), getFieldRange(startY, endY, ""));
		viewYear.setViewAdapter(yearAdapter);
		viewYear.setCurrentItem(curYear-startY);
	}
	
	private void setMonthColumnData(){
		int curMonth = mCalendar.get(Calendar.MONTH ) + 1;
		WheelView viewMonth = (WheelView)getChildAt(monthViewIndex);
		monthAdapter = new StringWheelAdapter(getContext(), getFieldRange(1, 12, ""));
		viewMonth.setViewAdapter(monthAdapter);
		viewMonth.setCurrentItem(curMonth-1);
	}
	
	private void setDayColumnData(){
		int curDay = mCalendar.get(Calendar.DAY_OF_MONTH );
		int maxDay = mCalendar.getActualMaximum(Calendar.DAY_OF_MONTH);
		WheelView viewDay = (WheelView)getChildAt(dayViewIndex);
		dayAdapter = new StringWheelAdapter(getContext(), getFieldRange(1, maxDay, ""));
		viewDay.setViewAdapter(dayAdapter);
		viewDay.setCurrentItem(curDay-1);
	}
	
	private void setHourColumnData(){
		int curHour = mCalendar.get(Calendar.HOUR_OF_DAY);
		WheelView viewHour = (WheelView)getChildAt(hourViewIndex);
		hourAdapter = new StringWheelAdapter(getContext(), getFieldRange(0, 23, ""));
		viewHour.setViewAdapter(hourAdapter);
		viewHour.setCurrentItem(curHour);
	}
	
	private void setMinColumnData(){
		int curMin = mCalendar.get(Calendar.MINUTE);
		WheelView viewMin = (WheelView)getChildAt(minViewIndex);
		minAdapter = new StringWheelAdapter(getContext(), getFieldRange(0, 59, ""));
		viewMin.setViewAdapter(minAdapter);
		viewMin.setCurrentItem(curMin);
	}
	
	/**对小于10的值补0*/
	private String prefixNum(int num){
		return num < 10 ? "0" + num : ""+num;
	}
	
	/**获取域（如年、月、日）的范围集合*/
	public List<String> getFieldRange(int start, int end, String unit){
		List<String> ranges = new ArrayList<String>();
		for(int i = start; i <= end; i++){
			ranges.add(prefixNum(i) + unit);
		}
		return ranges;
	}
	
	private void initListener(){
		WheelView yearView;
		WheelView monthView;
		WheelView dayView;
		WheelView hourView;
		WheelView minView;
		switch (mPickType) {
		case TIME:
			hourView 	= (WheelView)getChildAt(hourViewIndex);
			minView 	= (WheelView)getChildAt(minViewIndex);
			
			hourView.addChangingListener(hourChangeListener);
			minView.addChangingListener(minChangeListener);
			break;
		case DATE:
			yearView 		= (WheelView)getChildAt(yearViewIndex);
			monthView 	= (WheelView)getChildAt(monthViewIndex);
			dayView 		= (WheelView)getChildAt(dayViewIndex);
			
			yearView.addChangingListener(yearChangeListener);
			monthView.addChangingListener(monthChangeListener);
			dayView.addChangingListener(dayChangeListener);
			break;
		case DATE_TIME:
			yearView 		= (WheelView)getChildAt(yearViewIndex);
			monthView 	= (WheelView)getChildAt(monthViewIndex);
			dayView 		= (WheelView)getChildAt(dayViewIndex);
			hourView 	= (WheelView)getChildAt(hourViewIndex);
			minView 	= (WheelView)getChildAt(minViewIndex);
			
			yearView.addChangingListener(yearChangeListener);
			monthView.addChangingListener(monthChangeListener);
			dayView.addChangingListener(dayChangeListener);
			hourView.addChangingListener(hourChangeListener);
			minView.addChangingListener(minChangeListener);
			
			break;
		default:
			break;
		}
	}
	
	public int getValue(String valueStr, String unit){
		return TextUtils.isEmpty(valueStr) ? -1:Integer.valueOf(valueStr.replace(unit, ""));
	}
	
	/**年份变化监听*/
	OnWheelChangedListener yearChangeListener = new OnWheelChangedListener() {
		@Override
		public void onChanged(WheelView wheel, int oldIndex, int newIndex) {
			String value = yearAdapter.getItemText(newIndex).toString();
			int year = getValue(value, UNIT_YEAR);
			int preMaxDay = mCalendar.getActualMaximum(Calendar.DAY_OF_MONTH);
			int month = mCalendar.get(Calendar.MONTH);
			//更新年份值
			mCalendar.set(Calendar.YEAR, year);
			
			if(month == 1){//年份变化时，只有2月份可能是闰年变化
				mCalendar.set(Calendar.DAY_OF_MONTH, 1);
				onMouthMaxDaysChange(preMaxDay);
			}
			if(mChangeListener != null){
				mChangeListener.onCalenderChange(getFormatValue());
			}
		}
	};
	
	/**月份值改变监听*/
	OnWheelChangedListener monthChangeListener = new OnWheelChangedListener() {
		@Override
		public void onChanged(WheelView wheel, int oldIndex, int newIndex) {
			String value = monthAdapter.getItemText(newIndex).toString();
			int month = getValue(value, UNIT_MONTH);
			int preMaxDay = mCalendar.getActualMaximum(Calendar.DAY_OF_MONTH);
			int preYear = mCalendar.get(Calendar.YEAR);
			//更新月份值,设置日期值为1，是为防止在onMouthMaxDaysChange方法中计算
			//月份最大天数出错
			mCalendar.set(preYear, month-1, 1);
			onMouthMaxDaysChange(preMaxDay);
			if(mChangeListener != null){
				mChangeListener.onCalenderChange(getFormatValue());
			}
		}
	};
	
	/**月份天数变化时，重新设置数据*/
	private void onMouthMaxDaysChange(int preMaxDay){
		int curMaxDay = mCalendar.getActualMaximum(Calendar.DAY_OF_MONTH);
		WheelView dayView = (WheelView)getChildAt(dayViewIndex);
		//当前选择日期的索引，从0开始
		int curDayIndex = dayView.getCurrentItem();
		if(preMaxDay != curMaxDay){
			List<String> days = getFieldRange(1, curMaxDay, "");
			dayAdapter.setItemsData(days);
			dayView.invalidateWheel(true);
			int curDay = Math.min(curMaxDay, curDayIndex + 1);
			curDayIndex = curDay -1;
			dayView.setCurrentItem(curDayIndex);
		}
		//修改当前日期值，因为索引是从0开始，所以需要+1
		mCalendar.set(Calendar.DAY_OF_MONTH, curDayIndex + 1);
	}
	
	/**日期值改变监听*/
	OnWheelChangedListener dayChangeListener = new OnWheelChangedListener() {
		@Override
		public void onChanged(WheelView wheel, int oldIndex, int newIndex) {
			String value = dayAdapter.getItemText(newIndex).toString();
			int day = getValue(value, UNIT_DAY);
			//更新日期值
			mCalendar.set(Calendar.DAY_OF_MONTH, day);
			if(mChangeListener != null){
				mChangeListener.onCalenderChange(getFormatValue());
			}
		}
	};
	
	/**小时值改变监听*/
	OnWheelChangedListener hourChangeListener = new OnWheelChangedListener() {
		@Override
		public void onChanged(WheelView wheel, int oldIndex, int newIndex) {
			String value = hourAdapter.getItemText(newIndex).toString();
			int hour = getValue(value, UNIT_HOUR);
			//更新小时值
			mCalendar.set(Calendar.HOUR_OF_DAY, hour);
			if(mChangeListener != null){
				mChangeListener.onCalenderChange(getFormatValue());
			}
		}
	};
	
	/**分钟值改变监听*/
	OnWheelChangedListener minChangeListener = new OnWheelChangedListener() {
		@Override
		public void onChanged(WheelView wheel, int oldIndex, int newIndex) {
			String value = minAdapter.getItemText(newIndex).toString();
			int min = getValue(value, UNIT_MIN);
			//更新小时值
			mCalendar.set(Calendar.MINUTE, min);
			if(mChangeListener != null){
				mChangeListener.onCalenderChange(getFormatValue());
			}
		}
	};
	
	public void setOnCalenderChangeListener(OnCalenderChangeListener listener){
		mChangeListener = listener;
	}
	
	public interface OnCalenderChangeListener{
		public void onCalenderChange(String formatValue);
	}
	
	public enum PickType{
		/**日期*/
		DATE,
		/**时间*/
		TIME,
		/**日期+时间*/
		DATE_TIME,
	}

}
