package com.szboanda.android.platform.view;

import android.content.Context;
import android.graphics.Color;
import android.util.AttributeSet;
import android.util.SparseIntArray;
import android.view.View;
import android.view.ViewGroup;

/**
 * <p>Description:自定义控件自动换行的布局容器 </p>
 * @Company    深圳市博安达信息技术股份有限公司
 * @author    Siyi Lu    Create Date: 2015-8-3
 * @modify  
 */
public class AutoLineFeedLayout extends ViewGroup{
	
	/**存储行高的集合，用于出现一行内控件高度不一致时，总是存储高度最大的值*/
	private SparseIntArray lineHeigtCache = null;
	
	/**横向布局时，控件之间的间距*/
	private int horizontalMargin = 10;
	
	/**垂直方向布局时，行与行之间的间距*/
	private int verticalMargin = 10;
	
	private int lastLineResidualWidth = 0;

	public AutoLineFeedLayout(Context context, AttributeSet attrs) {
		super(context, attrs);
		init();
	}

	public AutoLineFeedLayout(Context context) {
		super(context);
		init();
	}
	
	private void init(){
		/**跟随背景色即可***/
		setBackgroundColor(Color.TRANSPARENT);
		lineHeigtCache = new SparseIntArray();
	}
	
	
	public void setHorizontalMargin(int horizontalMargin) {
		this.horizontalMargin = horizontalMargin;
	}
	
	public int getHorizontalMargin() {
		return horizontalMargin;
	}

	public void setVerticalMargin(int verticalMargin) {
		this.verticalMargin = verticalMargin;
	}

	int mWidthMeasureSpec;
	int mHeightMeasureSpec;
	@Override
	protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
		super.onMeasure(widthMeasureSpec, heightMeasureSpec);
		mWidthMeasureSpec = widthMeasureSpec;
		mHeightMeasureSpec = heightMeasureSpec;
		int viewHeight = layoutChilds(true);
		//根据计算出的尺寸设置视图的尺寸，使之生效
		setMeasuredDimension(getMeasuredWidth(), viewHeight);
	}
	
	@Override
	protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
		//因为继承处ViewGroup,子控件的布局完全自定义，
		//所以不需要调super.onLayout(changed, left, top, right, bottom);
		layoutChilds(false);
	}
	
	/**
	 * 布局子视图，该方法有两个用途，一个是在{@link #onMeasure(int, int)}中用于计算整个视图
	 * 的尺寸，另一个就是在{@link #onLayout(boolean, int, int, int, int)}方法中对子视图进行布局，
	 * 使用用了isMock进行区分
	 * @param isMock 是否模拟布局，如果为true,将不会调用{@link #layout(int, int, int, int)}子视图进行布局，
	 *                而只是在{@link #onMeasure(int, int)方法中为了计算视图的高度}
	 * @return 返回控件高度
	 */
	private int layoutChilds(boolean isMock){
		if(isMock){
			//模拟计算本视图尺寸时，必须清空原有缓存
			//正式布局时，不能清空
			lineHeigtCache.clear();
		}
		int measureHeight = 0;
		int sumWidth = 0;
		measureHeight += getPaddingTop();
		sumWidth += getPaddingLeft();
		int lineNum = 0;
		
		for(int i=0; i<getChildCount(); i++){
			View child = getChildAt(i);
			if(child.getVisibility() == View.GONE)
				continue;
			if(isMock){
				//如果是模拟布局，即被onMeasure()调用，必须计算child的尺寸
				//否则下面获取当前视图的宽度值时获取不到
				measureChild(child, mWidthMeasureSpec, mHeightMeasureSpec);
			}
			int curChildWidth = child.getMeasuredWidth();//当前行的宽
			if(isEnoughWidth(sumWidth, curChildWidth)){
				child.setTag(lineNum);
			}else{
				sumWidth = 0;
				sumWidth += getPaddingLeft();
				//换行高度已测量高度计算到前一行，所以这里是加上前一行的最大高度
				//此时的lineNum是前一行的行号
				measureHeight += (verticalMargin + lineHeigtCache.get(lineNum));
				child.setTag(++lineNum);
			}
			if(isMock){
			//缓存当前行高度
				changeLineHeightCache(child);
			}else{
				int lh = lineHeigtCache.get(lineNum);
				//当前子视图的一半高度
				int curChildHalfHeight = child.getMeasuredHeight()/2;
				int t = measureHeight + lh/2 - curChildHalfHeight ;
				int b = measureHeight + lh/2 + curChildHalfHeight;
				child.layout(sumWidth, t, sumWidth + curChildWidth, b);
			}
			sumWidth += horizontalMargin + curChildWidth;
		}
		lastLineResidualWidth = getMeasuredWidth() - sumWidth;
		//加上最后一行的高度
		measureHeight += lineHeigtCache.get(lineNum)+ getPaddingBottom();
		return measureHeight;
	}
	
	/**
	 * 根据当前控件宽度判断当前行是否有足够的空间放置
	 * @param sumWidth  当前行已放置控件的宽度
	 * @param viewWidth 将放置的控件宽度
	 * @return true：还可以放下当前控件，false当前控件应布局到下一行
	 */
	private boolean isEnoughWidth(int sumWidth, int viewWidth){
		return (getMeasuredWidth()-getPaddingRight()-sumWidth) >= viewWidth;
	}
	
	/**
	 * 缓存当前行控件所在行的高度
	 * @param child
	 */
	private void changeLineHeightCache(View child){
		int lineNum = Integer.valueOf(child.getTag().toString());
		if(lineHeigtCache.get(lineNum) == 0){
			lineHeigtCache.put(lineNum, child.getMeasuredHeight());
		}else{
			int exitHeight = lineHeigtCache.get(lineNum);
			lineHeigtCache.put(lineNum, Math.max(exitHeight, child.getMeasuredHeight()));
		}
	}
	
	public int getLineNum(){
		return lineHeigtCache.size();
	}
	
	/**
	 * 获取最后一行剩余的宽度
	 * @return
	 */
	public int getLastLineResidualWidth(){
		return lastLineResidualWidth;
	}
	
	public int getLineHeight(){
		int height = 0;
		for(int i=0; i<lineHeigtCache.size(); i++){
			int h = lineHeigtCache.get(i);
			height = Math.max(height, h);
		}
		return height;
	}
	
	public int getVerticalMargin(){
		return verticalMargin;
	}
	
}
