package com.beaconsinspace.android.beacon.detector;

import android.Manifest;
import android.app.Activity;
import android.app.Application;
import android.app.PendingIntent;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothManager;
import android.bluetooth.le.BluetoothLeScanner;
import android.bluetooth.le.ScanSettings;
import android.content.Context;
import android.content.Intent;
import android.content.ReceiverCallNotAllowedException;
import android.content.ServiceConnection;
import android.content.pm.PackageManager;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Build;
import android.os.Bundle;
import android.os.Looper;
import android.support.v4.app.ActivityCompat;

import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.GeofencingClient;
import com.google.android.gms.location.GeofencingRequest;
import com.google.android.gms.location.LocationServices;

import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.BeaconManager;
import org.altbeacon.beacon.BeaconParser;
import org.altbeacon.beacon.Identifier;
import org.altbeacon.beacon.RangeNotifier;
import org.altbeacon.beacon.Region;
import org.altbeacon.beacon.startup.RegionBootstrap;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;

import static android.location.LocationManager.GPS_PROVIDER;
import static android.location.LocationManager.NETWORK_PROVIDER;
import static android.location.LocationManager.PASSIVE_PROVIDER;

/**
 * Created by kyleshank on 7/24/17.
 */

public class BISCollectionManager implements LocationListener,
        BeaconConsumer,
        Application.ActivityLifecycleCallbacks {
    private static final String TAG = "BIS_COLLECTION_MANAGER";
    private static final Long BEACON_EXIT_THRESHOLD = 30000L;

    private BISReportingManager reportingManager;
    private BISCollectionEvent lastEvent;
    private GeofencingClient mGeofencingClient;
    private PendingIntent mGeofencePendingIntent;
    private List<Geofence> mGeofenceList;
    private Location lastLocation;
    private BeaconManager beaconManager = null;
    private RegionBootstrap regionBootstrap;

    private Timer timer;
    private TimerTask timerTask;

    private Timer stickyIdleTimer;
    private TimerTask stickyIdleTimerTask;

    private boolean stickyService = false;

    private List<Timer> intervalTimers = new ArrayList<Timer>();

    private static final ConcurrentHashMap<Beacon, Long> beaconTimestampMap = new ConcurrentHashMap<Beacon, Long>();
    private static final ConcurrentHashMap<Beacon, BISDetectorRSSICollector> beaconRssiMap = new ConcurrentHashMap<Beacon, BISDetectorRSSICollector>();

    private ConcurrentHashMap<String, Long> lastTriggerEvents = new ConcurrentHashMap<String, Long>();

    private Timer permissionsTimer;
    private TimerTask permissionsTimerTask;

    private Timer eventCollectionFilterTimer;
    private TimerTask eventCollectionFilterTimerTask;
    private List<BISCollectionEvent> gpsBuffer = new ArrayList<BISCollectionEvent>();
    private final BISDetector detector;

    public BISCollectionManager(BISDetector detector, boolean stickyService) {
        this.detector = detector;
        this.reportingManager = new BISReportingManager(detector);
        this.beaconManager = BeaconManager.getInstanceForApplication(detector.getContext());

        this.mGeofencingClient = null;
        this.mGeofenceList = new ArrayList<Geofence>();
        this.lastLocation = null;
        this.stickyService = stickyService;

        setupGeofencingClient();
        startLocationManager();
        startBeaconManager();

        if (detector.getContext() instanceof Application) {
            Application application = (Application) detector.getContext();
            application.registerActivityLifecycleCallbacks(this);
        }

        BISLog.d(TAG, "Collection Manager started.");
    }

    private void startLocationManager() {
        startLocationGathering();
    }

    public Location getLastKnownLocation() {
        return lastLocation;
    }

    @Override
    public void onLocationChanged(Location location) {
        lastLocation = location;
        BISCollectionEvent event = new BISCollectionEvent(detector.getContext(), BISCollectionEvent.BISCollectionEventType.GPS, detector.applicationState, location);
        saveGPSEvent(event);
    }

    @Override
    public void onStatusChanged(String provider, int status, Bundle extras) {
        BISLog.d(this.getClass().getName(), "onStatusChanged(" + provider + ")(" + status + ")");
    }

    @Override
    public void onProviderEnabled(String provider) {
        BISLog.d(this.getClass().getName(), "onProviderEnabled(" + provider + ")");
    }

    @Override
    public void onProviderDisabled(String provider) {
        BISLog.d(this.getClass().getName(), "onProviderDisabled(" + provider + ")");
    }

    public void triggerEvent(BISCollectionEvent event) {
        saveEvent(event);
        String uniqueId = event.getUniqueId();
        BISLog.d(this.getClass().getName(), "Triggered by event: " + uniqueId);
        boolean shouldGather = true;
        Date now = new Date();
        Long lastValue = this.lastTriggerEvents.get(uniqueId);
        if (lastValue != null) {
            long diff = (now.getTime() - lastValue);
            if (diff < (1000.0 * 60.0 * 10.0)) {
                shouldGather = false;
                BISLog.d(this.getClass().getName(), "Skipping location gathering. Similar event was trigged " + (diff / 1000.0) + " seconds ago.");
            }
        }
        if (shouldGather) {
            this.lastTriggerEvents.put(uniqueId, now.getTime());
            try {
                startLocationGathering();
            } catch (SecurityException e) {
                BISLog.e(TAG, e.getMessage());
            }
        }
    }

    public void startLocationGathering() {
        boolean hasFineLocation = ActivityCompat.checkSelfPermission(this.detector.getContext(), Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED;
        boolean hasCoarseLocation = ActivityCompat.checkSelfPermission(this.detector.getContext(), Manifest.permission.ACCESS_COARSE_LOCATION) == PackageManager.PERMISSION_GRANTED;

        if (!hasFineLocation && !hasCoarseLocation) {
            BISLog.d(this.getClass().getName(), "startLocationGathering() ACCESS_FINE_LOCATION and ACCESS_FINE_LOCATION not granted");
            return;
        }

        if (!isCollectingLocationData()) {
            if (permissionsTimer != null) {
                permissionsTimer.cancel();
            }

            permissionsTimer = null;
            permissionsTimerTask = null;

            BISLog.d(TAG, "START LOCATION GATHERING (" + detector.contextDescription + ")");

            try {
                LocationManager locationManager = (LocationManager) detector.getContext().getSystemService(Context.LOCATION_SERVICE);
                try {
                    locationManager.requestLocationUpdates(PASSIVE_PROVIDER, 0, 0, this, Looper.getMainLooper());
                } catch (Exception e) {
                    BISLog.e(TAG, "requestLocationUpdates failed for passive provider: " + e.getMessage());
                }
                try {
                    locationManager.requestLocationUpdates(GPS_PROVIDER, 0, 0, this, Looper.getMainLooper());
                } catch (Exception e) {
                    BISLog.e(TAG, "requestLocationUpdates failed for gps provider: " + e.getMessage());
                }
                try {
                    locationManager.requestLocationUpdates(NETWORK_PROVIDER, 0, 0, this, Looper.getMainLooper());
                } catch (Exception e) {
                    BISLog.e(TAG, "requestLocationUpdates failed for network provider: " + e.getMessage());
                }

                fetchRegions();

                Integer interval = detector.getConfiguration().backgroundDataCollectionMaxLength;

                BISLog.d(TAG, "backgroundDataCollectionMaxLength: " + interval);

                if (stickyService) {
                    if (stickyIdleTimer != null) {
                        stickyIdleTimer.cancel();
                        stickyIdleTimer = null;
                        stickyIdleTimerTask = null;
                    }
                }

                if (timer != null) {
                    timer.cancel();
                    timer = null;
                }

                timer = new Timer();
                timerTask = new TimerTask() {
                    public void run() {
                        try {
                            stopLocationGathering();
                        } catch (Throwable t) {
                            detector.reportCrash(t);
                        }
                    }
                };
                timer.schedule(timerTask, (long) interval);

                if (!detector.getConfiguration().urlBatchEvent.isEmpty()
                        && !detector.getConfiguration().batchEventInterval.isEmpty()) {
                    for (Timer existingTimer : intervalTimers) {
                        existingTimer.cancel();
                    }
                    intervalTimers.clear();
                    for (Integer intervalValue : detector.getConfiguration().batchEventInterval) {
                        Timer intervalTimer = new Timer();
                        TimerTask intervalTimerTask = new TimerTask() {
                            @Override
                            public void run() {
                                reportingManager.report();
                            }
                        };
                        intervalTimer.schedule(intervalTimerTask, intervalValue);
                        intervalTimers.add(intervalTimer);
                    }
                }

            } catch (SecurityException exception) {
                BISLog.d(this.getClass().getName(), "startLocationGathering() " + exception.getMessage());

                permissionsTimer = new Timer();
                permissionsTimerTask = new TimerTask() {
                    public void run() {
                        try {
                            startLocationGathering();
                        } catch (Throwable t) {
                            detector.reportCrash(t);
                        }
                    }
                };
                permissionsTimer.schedule(permissionsTimerTask, (long) 10000);
            } catch (Throwable t) {
                detector.reportCrash(t);
            }
        }
    }

    private void flushGPSEventsAndResetEventCollectionFilterTimer() {
        if (eventCollectionFilterTimer != null) {
            synchronized (eventCollectionFilterTimer) {
                eventCollectionFilterTimer.cancel();
                eventCollectionFilterTimer = null;
                eventCollectionFilterTimerTask = null;
            }
        }
        flushGPSEvents();
    }

    protected void stopLocationGathering() throws SecurityException {
        timer = null;
        timerTask = null;

        for (Timer existingTimer : intervalTimers) {
            existingTimer.cancel();
        }
        intervalTimers.clear();

        flushGPSEventsAndResetEventCollectionFilterTimer();

        reportingManager.report();

        BISLog.d(TAG, "STOP LOCATION GATHERING (" + detector.contextDescription + ")");

        LocationManager locationManager = (LocationManager) detector.getContext().getSystemService(Context.LOCATION_SERVICE);
        locationManager.removeUpdates(this);

        locationManager.requestLocationUpdates(PASSIVE_PROVIDER, 0, 0, this, Looper.getMainLooper());

        if (stickyService) {
            Integer interval = detector.getConfiguration().locationMonitoringInterval;

            stickyIdleTimer = new Timer();
            stickyIdleTimerTask = new TimerTask() {
                public void run() {
                    try {
                        startLocationGathering();
                    } catch (Throwable t) {
                        detector.reportCrash(t);
                    }
                }
            };
            stickyIdleTimer.schedule(stickyIdleTimerTask, (long) interval);
        }
    }

    private boolean isCollectingLocationData() {
        return this.timer != null;
    }

    private void setupGeofencingClient() {
        try {
            this.mGeofencingClient = LocationServices.getGeofencingClient(detector.getContext());
        } catch (Throwable t) {
            // Ignore if geofencing client class not present
            BISLog.e(this.getClass().getName(), "setupGeofencingClient() " + t.getMessage());
        }
    }

    private void saveEvent(BISCollectionEvent event) {
        BISLog.d(TAG, "Saved " + event.getEventType() + " event. (" + detector.contextDescription + ")");
        reportingManager.addEvent(event);
        boolean firstFetch = this.lastEvent == null;
        this.lastEvent = event;
        if (firstFetch) {
            fetchRegions();
        }
    }

    private void saveGPSEvent(BISCollectionEvent event) {
        Integer interval = detector.getConfiguration().eventCollectionFilterLength;
        if (interval == 0) {
            saveEvent(event);
            return;
        }

        synchronized (gpsBuffer) {
            gpsBuffer.add(event);
        }

        if (eventCollectionFilterTimer == null) {
            eventCollectionFilterTimer = new Timer();
            synchronized (eventCollectionFilterTimer) {
                eventCollectionFilterTimerTask = new TimerTask() {
                    public void run() {
                        flushGPSEventsAndResetEventCollectionFilterTimer();
                    }
                };
                eventCollectionFilterTimer.schedule(eventCollectionFilterTimerTask, interval);
            }
        }
    }

    private void flushGPSEvents() {
        BISCollectionEvent bestEvent = null;
        synchronized (gpsBuffer) {
            for (BISCollectionEvent event : gpsBuffer) {
                if (bestEvent == null) {
                    bestEvent = event;
                } else {
                    if ((bestEvent.location != null) && (event.location != null)) {
                        float bestEventAccuracy = bestEvent.location.getAccuracy();
                        float eventAccuracy = event.location.getAccuracy();
                        if (eventAccuracy < bestEventAccuracy) {
                            bestEvent = event;
                        }
                    }
                }
            }
            gpsBuffer.clear();
        }
        if (bestEvent != null) {
            saveEvent(bestEvent);
        }
    }

    private void fetchRegions() {
        if (ActivityCompat.checkSelfPermission(this.detector.getContext(), Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            BISLog.d(this.getClass().getName(), "fetchRegions() ACCESS_FINE_LOCATION not granted");
            return;
        }
        if (this.lastLocation == null) {
            BISLog.d(this.getClass().getName(), "fetchRegions() needs a last location");
            return;
        }

        try {
            getGeofences();
        } catch (Throwable t) {
            detector.reportCrash(t);
        }
    }

    private void getGeofences() {
        getGeofences(detector.getConfiguration().urlGeofence);
    }

    private void getGeofences(final List<String> apiUrls) {
        BISLog.d(TAG, "COLLECTING GEO-FENCE CONFIGURATION");

        if ((getLastKnownLocation() == null) || (apiUrls == null) || apiUrls.isEmpty()) {
            return;
        }

        final String apiUrl = apiUrls.get(0);
        final BISCollectionManager self = this;

        BISNetworking.getGeofences(detector, apiUrl, getLastKnownLocation(), new BISNetworkingThreadListener<String>() {
            public void onSuccess(String responseString) {
                try {
                    if (responseString == null) {
                        throw new JSONException("response is null");
                    }

                    JSONObject response = new JSONObject(responseString);
                    JSONObject data = response.getJSONObject("data");
                    JSONArray array = data.getJSONArray("geofence");

                    List<Geofence> geofences = new ArrayList<Geofence>();

                    for (int i = 0; i < array.length(); i++) {
                        JSONObject geofenceJson = array.getJSONObject(i);
                        double latitude = geofenceJson.getDouble("latitude");
                        double longitude = geofenceJson.getDouble("longitude");
                        float radius = (float) geofenceJson.getDouble("radius");

                        Geofence geofence = new Geofence.Builder()
                                .setRequestId(latitude + "+" + longitude + "+" + radius)
                                .setCircularRegion(
                                        latitude,
                                        longitude,
                                        radius
                                )
                                .setExpirationDuration(Geofence.NEVER_EXPIRE)
                                .setTransitionTypes(Geofence.GEOFENCE_TRANSITION_ENTER |
                                        Geofence.GEOFENCE_TRANSITION_DWELL |
                                        Geofence.GEOFENCE_TRANSITION_EXIT)
                                .setLoiteringDelay(30000)
                                .build();

                        geofences.add(geofence);
                    }

                    if (self.mGeofencingClient != null) {
                        self.mGeofencingClient.removeGeofences(getGeofencePendingIntent());
                        if (geofences != null) {
                            mGeofenceList.clear();
                            mGeofenceList.addAll(geofences);
                        }
                        if (!mGeofenceList.isEmpty()) {
                            if (ActivityCompat.checkSelfPermission(self.detector.getContext(), Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED) {
                                self.mGeofencingClient.addGeofences(getGeofencingRequest(), getGeofencePendingIntent());
                            }
                        }
                    }
                } catch (Exception e) {
                    List<String> newUrls = new ArrayList<String>();
                    for (String newUrl : apiUrls) {
                        if (!newUrl.equals(apiUrl)) {
                            newUrls.add(newUrl);
                        }
                    }
                    getGeofences(newUrls);
                }
            }

            public void onError(Throwable t) {
                List<String> newUrls = new ArrayList<String>();
                for (String newUrl : apiUrls) {
                    if (!newUrl.equals(apiUrl)) {
                        newUrls.add(newUrl);
                    }
                }
                getGeofences(newUrls);
            }
        });
    }

    private GeofencingRequest getGeofencingRequest() {
        GeofencingRequest.Builder builder = new GeofencingRequest.Builder();

        // The INITIAL_TRIGGER_ENTER flag indicates that geofencing service should trigger a
        // GEOFENCE_TRANSITION_ENTER notification when the geofence is added and if the device
        // is already inside that geofence.
        builder.setInitialTrigger(GeofencingRequest.INITIAL_TRIGGER_ENTER);

        // Add the geofences to be monitored by geofencing service.
        builder.addGeofences(mGeofenceList);

        // Return a GeofencingRequest.
        return builder.build();
    }

    private PendingIntent getGeofencePendingIntent() {
        // Reuse the PendingIntent if we already have it.
        if (mGeofencePendingIntent != null) {
            return mGeofencePendingIntent;
        }
        Intent intent = new Intent(this.detector.getContext(), BISGeofenceTransitionsIntentService.class);
        // We use FLAG_UPDATE_CURRENT so that we get the same pending intent back when calling
        // addGeofences() and removeGeofences().
        mGeofencePendingIntent = PendingIntent.getService(this.detector.getContext(), 0, intent, PendingIntent.FLAG_UPDATE_CURRENT);
        return mGeofencePendingIntent;
    }

    private void startBeaconManager() {
        if (!detector.getConfiguration().isModelSupportedForBluetoothUse()) {
            return;
        }

        boolean hasBluetooth = ActivityCompat.checkSelfPermission(this.detector.getContext(), Manifest.permission.BLUETOOTH) == PackageManager.PERMISSION_GRANTED;
        boolean hasBluetoothAdmin = ActivityCompat.checkSelfPermission(this.detector.getContext(), Manifest.permission.BLUETOOTH_ADMIN) == PackageManager.PERMISSION_GRANTED;

        if (!hasBluetooth) {
            BISLog.d(this.getClass().getName(), "startBeaconManager() BLUETOOTH not granted");
            return;
        }

        if (!hasBluetoothAdmin) {
            BISLog.d(this.getClass().getName(), "startBeaconManager() BLUETOOTH_ADMIN not granted");
            return;
        }

        try {

            final String iBeaconLayout = "m:2-3=0215,i:4-19,i:20-21,i:22-23,p:24-24";
            beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(iBeaconLayout));

            final String altBeaconLayout = "m:2-3=beac,i:4-19,i:20-21,i:22-23,p:24-24,d:25-25";
            beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(altBeaconLayout));

            final String eddystoneUIDBeaconLayout = "s:0-1=feaa,m:2-2=00,p:3-3:-41,i:4-13,i:14-19";
            beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(eddystoneUIDBeaconLayout));

            final String eddystoneURLBeaconLayout = "s:0-1=feaa,m:2-2=10,p:3-3:-41,i:4-20v";
            beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(eddystoneURLBeaconLayout));

            try {
                setBeaconScanPeriods();
            } catch (ReceiverCallNotAllowedException e) {
                BISLog.e(TAG, e.getMessage());
            }

            // Android O
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                ScanSettings settings = (new ScanSettings.Builder().setScanMode(ScanSettings.SCAN_MODE_LOW_POWER)).build();
                BluetoothManager bluetoothManager =
                        (BluetoothManager) this.detector.getContext().getSystemService(Context.BLUETOOTH_SERVICE);
                if (bluetoothManager != null) {
                    BluetoothAdapter bluetoothAdapter = bluetoothManager.getAdapter();
                    if (bluetoothAdapter != null) {
                        if (bluetoothAdapter.isEnabled()) {
                            Intent intent = new Intent(this.detector.getContext(), BISBeaconReceiver.class);
                            intent.putExtra("o-scan", true);
                            PendingIntent pendingIntent = PendingIntent.getBroadcast(this.detector.getContext(), 0, intent, PendingIntent.FLAG_UPDATE_CURRENT);
                            BluetoothLeScanner scanner = bluetoothAdapter.getBluetoothLeScanner();
                            if (scanner != null) {
                                // TODO: Fixme. Bullshit saftey wrapping to avoid fatal RuntimeException if BLUETOOTH_PRIVILEGED not granted.
                                try {
                                    scanner.startScan(null, settings, pendingIntent);
                                } catch (Exception permissionException) {
                                    BISLog.e(TAG, "RuntimeException likely. BLUETOOTH_PRIVILEGED not granted? " + permissionException.getMessage());
                                }
                            }
                        }
                    }
                }
            }

        } catch (Throwable t) {
            detector.reportCrash(t);
        }
    }

    private void setBeaconScanPeriods() {
        if (beaconManager == null) {
            return;
        }

        try {

            beaconManager.setForegroundScanPeriod(detector.getConfiguration().foregroundBluetoothScanPeriod);
            beaconManager.setForegroundBetweenScanPeriod(detector.getConfiguration().foregroundBluetoothBetweenScanPeriod);
            beaconManager.setBackgroundScanPeriod(detector.getConfiguration().backgroundBluetoothScanPeriod);
            beaconManager.setBackgroundBetweenScanPeriod(detector.getConfiguration().backgroundBluetoothBetweenScanPeriod);

            doBindBeaconManager();

            if (beaconManager.isBound(this)) {
                beaconManager.updateScanPeriods();
            }

        } catch (Throwable t) {
            detector.reportCrash(t);
        }
    }

    @Override
    public void onBeaconServiceConnect() {
        try {
            beaconManager.setRangeNotifier(

                    new RangeNotifier() {

                        @Override
                        public void didRangeBeaconsInRegion(Collection<Beacon> collection, org.altbeacon.beacon.Region region) {
                            try {
                                // get currentTimestampstamp
                                Long currentTimestamp = System.currentTimeMillis();

                    /*
                     * Loop on beacons that are in range and
                     */
                                for (Beacon beacon : collection) {
                                    // determine if we have seen beacon before
                                    Long beaconLastSeenTimestamp = beaconTimestampMap.get(beacon);

                                    // BRAND-NEW beacon
                                    if (beaconLastSeenTimestamp == null) {
                                        BISLog.d(TAG, "ENTERED BEACON " + uniqueIdentifierForBeacon(beacon));
                                        handleBeaconEnter(beacon);
                                        beaconRssiMap.put(beacon, new BISDetectorRSSICollector());
                                    }

                                    // Update
                                    BISDetectorRSSICollector rssiCollector = beaconRssiMap.get(beacon);
                                    rssiCollector.add(beacon.getRssi());
                                    beaconRssiMap.put(beacon, rssiCollector);
                                    beaconTimestampMap.put(beacon, currentTimestamp);
                                }

                    /*
                     * Loop on beacons and check current timestamps
                     */
                                for (Beacon beacon : beaconTimestampMap.keySet()) {
                                    Long beaconLastSeenTimestamp = beaconTimestampMap.get(beacon);
                                    Long diff = currentTimestamp - beaconLastSeenTimestamp;

                                    if (diff > BEACON_EXIT_THRESHOLD) {
                                        // OLD BEACON
                                        BISLog.d(TAG, "EXITED BEACON " + uniqueIdentifierForBeacon(beacon));
                                        handleBeaconExit(beacon);
                                        beaconTimestampMap.remove(beacon);
                                    } else {
//                                    Log.d(TAG,"BEACON "+uniqueIdentifierForBeacon(beacon)+" TIME SINCE SEEN "+diff+" ");
                                    }
                                }
                            } catch (Throwable t) {
                                detector.reportCrash(t);
                            }
                        }
                    }
            );

            beaconManager.startRangingBeaconsInRegion(new Region("ALL_BEACONS", null, null, null));
        } catch (Throwable t) {
            detector.reportCrash(t);
        }
    }

    @Override
    public Context getApplicationContext() {
        return this.detector.getContext();
    }

    @Override
    public void unbindService(ServiceConnection serviceConnection) {
        try {
            detector.getContext().unbindService(serviceConnection);
        } catch (IllegalArgumentException e) {
            BISLog.e(TAG, e.getMessage());
        } catch (Throwable t) {
            detector.reportCrash(t);
        }
    }

    @Override
    public boolean bindService(Intent intent, ServiceConnection serviceConnection, int i) {
        try {
            return detector.getContext().bindService(intent, serviceConnection, i);
        } catch (ReceiverCallNotAllowedException e) {
            BISLog.e(TAG, e.getMessage());
        } catch (Throwable t) {
            detector.reportCrash(t);
        }
        return false;
    }

    private void doBindBeaconManager() {
        try {
            if (beaconManager != null && !beaconManager.isBound(this)) {
                beaconManager.bind(this);
            }
        } catch (Throwable t) {
            detector.reportCrash(t);
        }
    }

    private void undoBindBeaconManager() {
        try {
            if (beaconManager != null && beaconManager.isBound(this)) {
                beaconManager.unbind(this);
            }
        } catch (Throwable t) {
            detector.reportCrash(t);
        }
    }

    public void startRanging() {
        try {
            stopRanging();
            doBindBeaconManager();
        } catch (Throwable t) {
            detector.reportCrash(t);
        }
    }

    public void stopRanging() {
        try {
            Collection<Region> regions = beaconManager.getRangedRegions();
            for (Region region : regions) {
                beaconManager.stopRangingBeaconsInRegion(region);
            }
            undoBindBeaconManager();
        } catch (Throwable t) {
            detector.reportCrash(t);
        }
    }

    private void handleBeaconEnter(Beacon beacon) {
        BISCollectionEvent event = new BISCollectionEvent(detector.getContext(),
                BISCollectionEvent.BISCollectionEventType.BeaconInteraction,
                detector.applicationState,
                BISCollectionEvent.BISCollectionEventDirection.Enter,
                beacon);
        event.location = lastLocation;
        triggerEvent(event);
    }

    private void handleBeaconExit(Beacon beacon) {
        BISCollectionEvent event = new BISCollectionEvent(detector.getContext(),
                BISCollectionEvent.BISCollectionEventType.BeaconInteraction,
                detector.applicationState,
                BISCollectionEvent.BISCollectionEventDirection.Exit,
                beacon,
                beaconRssiMap.get(beacon));
        event.location = lastLocation;
        triggerEvent(event);
    }

    private static String beaconIdentifierToString(Identifier i) {
        try {
            if (i == null) {
                return null;
            }
            String s = i.toString();
            if (s == null) {
                return null;
            }

            if (s.length() >= 2 && s.substring(0, 2).equals("0x")) {
                s = s.substring(2); // get rid of the 0x
            }
            return s;
        } catch (Throwable t) {
            BISLog.e(TAG, t.getMessage());
        }
        return null;
    }

    private String[] idsForBeacon(Beacon beacon) {

        if (beacon == null) {
            return new String[]{"", "", ""};
        }


        Identifier id1 = null;
        Identifier id2 = null;
        Identifier id3 = null;
        try {
            id1 = beacon.getIdentifiers().size() > 0 ? beacon.getId1() : null;
        } catch (Exception e) {
        }
        try {
            id2 = beacon.getIdentifiers().size() > 1 ? beacon.getId2() : null;
        } catch (Exception e) {
        }
        try {
            id3 = beacon.getIdentifiers().size() > 2 ? beacon.getId3() : null;
        } catch (Exception e) {
        }

        String ID1 = id1 != null ? beaconIdentifierToString(id1) : "";
        String ID2 = id2 != null ? beaconIdentifierToString(id2) : "";
        String ID3 = id3 != null ? beaconIdentifierToString(id3) : "";

        return new String[]{ID1, ID2, ID3};
    }

    private String uniqueIdentifierForBeacon(Beacon beacon) {

        if (beacon == null) {
            return "";
        }

        String s = null;
        try {
            String[] strings = idsForBeacon(beacon);

            String id1 = strings[0];
            String id2 = strings[1];
            String id3 = strings[2];

            String uniqueId = id1 + "_" + id2 + "_" + id3;

            s = id1 + ":" + uniqueId.hashCode();
        } catch (Throwable t) {
            detector.reportCrash(t);
        }
        return s;
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle bundle) {
    }

    @Override
    public void onActivityStarted(Activity activity) {
    }

    @Override
    public void onActivityResumed(Activity activity) {
        startLocationGathering();
    }

    @Override
    public void onActivityPaused(Activity activity) {
        reportingManager.report();
    }

    @Override
    public void onActivityStopped(Activity activity) {
        reportingManager.report();
    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle bundle) {

    }

    @Override
    public void onActivityDestroyed(Activity activity) {

    }
}
