package com.beaconsinspace.android.beacon.detector;

import android.os.Build;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Timer;

/**
 * Created by johnlfoleyiii on 12/14/16.
 */

class BISConfiguration {

    private static final String TAG = "BIS_CONFIG";

    static final String KEY_foregroundBluetoothScanPeriod = "foregroundBluetoothScanPeriod";
    static final String KEY_foregroundBluetoothBetweenScanPeriod = "foregroundBluetoothBetweenScanPeriod";
    static final String KEY_backgroundBluetoothScanPeriod = "backgroundBluetoothScanPeriod";
    static final String KEY_backgroundBluetoothBetweenScanPeriod = "backgroundBluetoothBetweenScanPeriod";
    static final String KEY_locationMonitoringInterval = "locationMonitoringInterval";
    static final String KEY_unsupportedAndroidModels = "unsupportedAndroidModels";
    static final String KEY_collectForegroundProcess = "collectForegroundProcess";
    static final String KEY_eventCollectionFilterLength = "eventCollectionFilterLength";

    static final String KEY_urlConfiguration = "urlConfiguration";
    static final String KEY_urlSingleEvent = "urlSingleEvent";
    static final String KEY_urlBeaconIdentifiers = "urlBeaconIdentifiers";
    static final String KEY_urlBatchEvent = "urlBatchEvent";
    static final String KEY_urlGeofence = "urlGeofence";
    static final String KEY_configurationExpiration = "configurationExpiration";
    static final String KEY_batchEventMax = "batchEventMax";
    static final String KEY_backgroundDataCollectionMaxLength = "backgroundDataCollectionMaxLength";
    static final String KEY_foregroundDataCollectionMaxLength = "foregroundDataCollectionMaxLength";
    static final String KEY_bisData = "bisData";
    static final String KEY_sentryUrl = "sentryUrl";
    static final String KEY_sentryKey = "sentryKey";
    static final String KEY_sentrySecret = "sentrySecret";
    static final String KEY_signficantLocationChangeDistance = "signficantLocationChangeDistance";
    static final String KEY_batchEventInterval = "batchEventInterval";

    static final String URL_configurationRetry1 = "https://E8C34AB4-CFC3-43C8-9484-E0625D47303E.net"
            + BISNetworking.BIS_BASE_ENDPOINT;
    static final String URL_configurationRetry2 = "https://FF4415AE-151D-4C70-B3D7-E883A1CB84EE.net"
            + BISNetworking.BIS_BASE_ENDPOINT;
    static final String URL_configurationRetry3 = "https://438CDE19-0278-4CCD-9E92-32CA51E83021.net"
            + BISNetworking.BIS_BASE_ENDPOINT;

    public Integer foregroundBluetoothScanPeriod = 2000;
    public Integer foregroundBluetoothBetweenScanPeriod = 60000;
    public Integer backgroundBluetoothScanPeriod = 10000;
    public Integer backgroundBluetoothBetweenScanPeriod = 60000;
    public Integer locationMonitoringInterval = 5340000;
    public Integer eventCollectionFilterLength = 9000;

    public List<String> unsupportedAndroidModels = new ArrayList<String>();
    public Integer collectForegroundProcess = -1;
    public Integer configurationExpiration = 10800000;
    public Integer batchEventMax = 1000;
    public Integer backgroundDataCollectionMaxLength = 18000;
    public Integer foregroundDataCollectionMaxLength = 18000;
    public Integer signficantLocationChangeDistance = 50;

    public String sentryUrl = "https://sentry.io/api/284645/store/";
    public String sentryKey = "3d915666beba410c89fe6cb9a3034539";
    public String sentrySecret = "512bdf97ca284100b0eed543bca78654";

    public List<String> defaultUrlConfiguration = new ArrayList<String>(Arrays.asList(BISNetworking.BIS_URL_INITIALIZE));
    public List<String> urlConfiguration = new ArrayList<String>(Arrays.asList(BISNetworking.BIS_URL_INITIALIZE));
    public List<String> urlSingleEvent = new ArrayList<String>();
    public List<String> urlBeaconIdentifiers = new ArrayList<String>();
    public List<String> urlBatchEvent = new ArrayList<String>();
    public List<String> urlGeofence = new ArrayList<String>();

    public List<Integer> batchEventInterval = new ArrayList<Integer>();

    public String bisData = null;

    private enum BISConfigurationState {
        Default,
        Provided,
        DefaultRetry,
        Fallback1,
        Fallback2,
        Fallback3
    }

    private BISConfigurationState state = BISConfigurationState.Default;
    private List<String> stateConfigurationUrls = defaultUrlConfiguration;
    private Integer stateRetryCount = 0;
    private Date configurationReferenceDate = new Date();
    private Timer configurationTimer = null;
    private final BISDetector detector;
    private BISConfigurationInitializationListener initializationListener = null;

    public BISConfiguration(BISDetector detector, BISConfigurationInitializationListener initializationListener) {
        this.detector = detector;
        this.initializationListener = initializationListener;
        fetch();
    }

    private void fetch() {
        try {
            BISLog.d(TAG, "Getting Configuration");
            fetch(stateConfigurationUrls);
        } catch (Throwable t) {
            detector.reportCrash(t);
        }
    }

    private void fetch(final List<String> configurationUrls) throws BISException {

        if (configurationUrls.isEmpty()) {
            BISLog.d(this.getClass().getName(), "Couldn't fetch updated configuration");
            throw new BISException("Couldn't fetch updated configuration");
        }

        String configJson = null;
        try {
            configJson = new BISPersistentStorage(detector.getContext()).getString(BISPersistentStorage.KEY_BIS_CONFIGURATION_JSON);
        } catch (Exception e) {
            BISLog.d(TAG, "Failed to get from persistent storage");
        }

        try {
            if (configJson != null) {
                BISLog.d(TAG, "Found configuration json on disk");
                try {
                    processResponse(new JSONObject(configJson));
                } catch (JSONException e) {
                    BISLog.e(TAG, "JSON parsing exception");
                }

                Long millisecondsSinceConfigurationLastStoredToDisk = null;
                try {
                    final String millisecondString = new BISPersistentStorage(detector.getContext()).getString(BISPersistentStorage.KEY_BIS_CONFIGURATION_MILLISECONDS_SINCE_LAST_STORED);
                    if (millisecondString != null) {
                        millisecondsSinceConfigurationLastStoredToDisk = Long.valueOf(millisecondString);
                    }
                } catch (Exception e) {
                    BISLog.d(TAG, "Failed to get millisecondsSinceConfigurationLastStoredToDisk from disk");
                }

                if (millisecondsSinceConfigurationLastStoredToDisk != null && (System.currentTimeMillis() - millisecondsSinceConfigurationLastStoredToDisk > configurationExpiration)) {
                    BISLog.d(TAG, "Found configuration on disk to be too old, getting new from server");
                    fetchFromServer(configurationUrls);
                }

            } else {
                BISLog.d(TAG, "Getting configuration from server");
                fetchFromServer(configurationUrls);
            }
        } catch (Exception e) {
            BISLog.e(TAG, "Failure in fetch: " + e.getMessage());
        }

    }

    private void fetchFromServer(final List<String> configurationUrls) throws BISException {

        final String configurationUrl = configurationUrls.get(0);

        BISLog.d(this.getClass().getName(), "Fetching configuration from server: " + configurationUrl + " (" + detector.contextDescription + ")");

        final BISConfiguration self = this;
        BISNetworking.getConfiguration(detector, configurationUrl, new BISNetworkingThreadListener<String>() {
            @Override
            public void onSuccess(String responseString) {
                try {
                    if (responseString == null) {
                        throw new JSONException("response is null");
                    }

                    JSONObject response = new JSONObject(responseString);
                    processResponse(response);

                    try {
                        new BISPersistentStorage(detector.getContext()).storeString(
                                BISPersistentStorage.KEY_BIS_CONFIGURATION_JSON,
                                responseString
                        );
                    } catch (Exception e) {
                        BISLog.d(TAG, "Failed to store json string in persistent storage");
                    }


                    try {
                        new BISPersistentStorage(detector.getContext()).storeString(
                                BISPersistentStorage.KEY_BIS_CONFIGURATION_MILLISECONDS_SINCE_LAST_STORED,
                                String.valueOf(System.currentTimeMillis())
                        );
                    } catch (Exception e) {
                        BISLog.d(TAG, "Failed to store milliseconds since configuration gathered from server to disk");
                    }

                } catch (Exception e) {
                    List<String> newConfigurationUrls = new ArrayList<String>();
                    for (String url : configurationUrls) {
                        if (!url.equals(configurationUrl)) {
                            newConfigurationUrls.add(url);
                        }
                    }
                    try {
                        self.fetch(newConfigurationUrls);
                    } catch (BISException bisException) {
                        BISLog.d(this.getClass().getName(), "Attempting to fetch configuration again...");
                        Integer interval = fetchConfigurationFailed();
                    }
                }
            }

            @Override
            public void onError(Throwable t) {

            }
        });
    }

    private void processResponse(JSONObject totalResponse) throws JSONException {
        JSONObject response = totalResponse.getJSONObject("data");
        if (response == null) {
            throw new JSONException("Data key not found for configuration");
        }
        Iterator<String> keyIterator = response.keys();
        while (keyIterator.hasNext()) {
            String key = keyIterator.next();
            switch (key) {
                case KEY_foregroundBluetoothScanPeriod:
                    foregroundBluetoothScanPeriod = response.getInt(key);
                    break;
                case KEY_foregroundBluetoothBetweenScanPeriod:
                    foregroundBluetoothBetweenScanPeriod = response.getInt(key);
                    break;
                case KEY_backgroundBluetoothScanPeriod:
                    backgroundBluetoothScanPeriod = response.getInt(key);
                    break;
                case KEY_backgroundBluetoothBetweenScanPeriod:
                    backgroundBluetoothBetweenScanPeriod = response.getInt(key);
                    break;
                case KEY_locationMonitoringInterval:
                    locationMonitoringInterval = response.getInt(key);
                    break;
                case KEY_unsupportedAndroidModels:
                    List<String> models = new ArrayList<String>();
                    JSONArray modelsArray = response.getJSONArray(key);
                    if (modelsArray != null) {
                        int len = modelsArray.length();
                        for (int i = 0; i < len; i++) {
                            models.add(modelsArray.getString(i));
                        }
                    }
                    unsupportedAndroidModels = models;
                    break;
                case KEY_collectForegroundProcess:
                    collectForegroundProcess = response.getInt(key);
                    break;
                case KEY_configurationExpiration:
                    configurationExpiration = response.getInt(key);
                    break;
                case KEY_batchEventMax:
                    batchEventMax = response.getInt(key);
                    break;
                case KEY_backgroundDataCollectionMaxLength:
                    backgroundDataCollectionMaxLength = response.getInt(key);
                    break;
                case KEY_foregroundDataCollectionMaxLength:
                    foregroundDataCollectionMaxLength = response.getInt(key);
                    break;
                case KEY_eventCollectionFilterLength:
                    eventCollectionFilterLength = response.getInt(key);
                    break;
                case KEY_signficantLocationChangeDistance:
                    signficantLocationChangeDistance = response.getInt(key);
                    break;
                case KEY_urlConfiguration:
                    List<String> config = new ArrayList<String>();
                    JSONArray configArray = response.getJSONArray(key);
                    if (configArray != null) {
                        int len = configArray.length();
                        for (int i = 0; i < len; i++) {
                            config.add(configArray.getString(i));
                        }
                    }
                    urlConfiguration = config;
                    break;
                case KEY_urlSingleEvent:
                    List<String> singleEvent = new ArrayList<String>();
                    JSONArray singleEventArray = response.getJSONArray(key);
                    if (singleEventArray != null) {
                        int len = singleEventArray.length();
                        for (int i = 0; i < len; i++) {
                            singleEvent.add(singleEventArray.getString(i));
                        }
                    }
                    urlSingleEvent = singleEvent;
                    break;
                case KEY_urlBeaconIdentifiers:
                    List<String> beacons = new ArrayList<String>();
                    JSONArray beaconsArray = response.getJSONArray(key);
                    if (beaconsArray != null) {
                        int len = beaconsArray.length();
                        for (int i = 0; i < len; i++) {
                            beacons.add(beaconsArray.getString(i));
                        }
                    }
                    urlBeaconIdentifiers = beacons;
                    break;
                case KEY_urlBatchEvent:
                    List<String> batchEvent = new ArrayList<String>();
                    JSONArray batchEventArray = response.getJSONArray(key);
                    if (batchEventArray != null) {
                        int len = batchEventArray.length();
                        for (int i = 0; i < len; i++) {
                            batchEvent.add(batchEventArray.getString(i));
                        }
                    }
                    urlBatchEvent = batchEvent;
                    break;
                case KEY_urlGeofence:
                    List<String> geofences = new ArrayList<String>();
                    JSONArray geofencesArray = response.getJSONArray(key);
                    if (geofencesArray != null) {
                        int len = geofencesArray.length();
                        for (int i = 0; i < len; i++) {
                            geofences.add(geofencesArray.getString(i));
                        }
                    }
                    urlGeofence = geofences;
                    break;
                case KEY_sentryUrl:
                    sentryUrl = response.getString(key);
                    break;
                case KEY_sentryKey:
                    sentryKey = response.getString(key);
                    break;
                case KEY_sentrySecret:
                    sentrySecret = response.getString(key);
                    break;
                case KEY_bisData:
                    bisData = response.getString(key);
                    break;
                case KEY_batchEventInterval:
                    List<Integer> intervals = new ArrayList<Integer>();
                    JSONArray responseEventIntervals = response.getJSONArray(KEY_batchEventInterval);
                    if (responseEventIntervals != null) {
                        int len = responseEventIntervals.length();
                        for (int i = 0; i < len; i++) {
                            intervals.add(responseEventIntervals.getInt(i));
                        }
                    }
                    batchEventInterval = intervals;
                    break;
                default:
                    break;
            }
        }
        if ((this.urlConfiguration != null) && !this.urlConfiguration.isEmpty()) {
            state = BISConfigurationState.Provided;
            stateConfigurationUrls = urlConfiguration;
            stateRetryCount = 0;
        }
        if (this.initializationListener != null) {
            this.initializationListener.onInitializationSuccess(this);
            this.initializationListener = null;
        }
    }

    private Integer fetchConfigurationFailed() {
        Date now = new Date();
        stateRetryCount = stateRetryCount + 1;
        switch (state) {
            case Default:
                if (stateRetryCount >= 10) {
                    if ((this.urlConfiguration != null) && !this.urlConfiguration.isEmpty()) {
                        state = BISConfigurationState.Provided;
                        stateConfigurationUrls = urlConfiguration;
                        stateRetryCount = 0;
                    } else {
                        state = BISConfigurationState.DefaultRetry;
                        stateConfigurationUrls = defaultUrlConfiguration;
                        stateRetryCount = 0;
                    }
                }
                break;
            case Provided:
                if (stateRetryCount >= 10) {
                    state = BISConfigurationState.DefaultRetry;
                    stateConfigurationUrls = defaultUrlConfiguration;
                    stateRetryCount = 0;
                }
                break;
            case DefaultRetry:
                if (stateRetryCount >= 10) {
                    state = BISConfigurationState.Fallback1;
                    List<String> newUrls = new ArrayList();
                    newUrls.add(BISNetworking.BIS_URL_INITIALIZE);
                    newUrls.add(URL_configurationRetry1);
                    stateConfigurationUrls = newUrls;
                    stateRetryCount = 0;
                    configurationReferenceDate = new Date();
                }
                break;
            case Fallback1:
                if ((now.getTime() - configurationReferenceDate.getTime()) > (1000.0 * 60.0 * 60.0 * 3.0)) {
                    state = BISConfigurationState.Fallback2;
                    List<String> newUrls = new ArrayList();
                    newUrls.add(BISNetworking.BIS_URL_INITIALIZE);
                    newUrls.add(URL_configurationRetry1);
                    newUrls.add(URL_configurationRetry2);
                    stateConfigurationUrls = newUrls;
                    stateRetryCount = 0;
                }
                break;
            case Fallback2:
                if ((now.getTime() - configurationReferenceDate.getTime()) > (1000.0 * 60.0 * 60.0 * 12.0)) {
                    state = BISConfigurationState.Fallback3;
                    List<String> newUrls = new ArrayList();
                    newUrls.add(BISNetworking.BIS_URL_INITIALIZE);
                    newUrls.add(URL_configurationRetry1);
                    newUrls.add(URL_configurationRetry2);
                    newUrls.add(URL_configurationRetry3);
                    stateConfigurationUrls = newUrls;
                    stateRetryCount = 0;
                }
                break;
            default:
                break;
        }
        return 60000;
    }

    public boolean isModelSupportedForBluetoothUse() {
        String deviceModel = Build.MODEL;
        for (String unsupportedModel : unsupportedAndroidModels) {
            if (deviceModel.equals(unsupportedModel)) {
                BISLog.e(TAG, "BeaconsInSpace Detector Library does not run on Android Model " + unsupportedModel + " due to Networking/Bluetooth collision issues.");
                return false;
            }
        }
        return true;
    }
}
