/*
 * Copyright (c) Afilias Technologies Ltd 2017. All rights reserved.
 */

package com.beaconsinspace.android.beacon.detector.deviceatlas;

import android.app.Activity;
import android.util.Log;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.reflect.Field;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.Callable;

/**
 * @author Afilias Technologies Ltd
 */
public class DataCollectorTask implements Callable<Void> {
    private static final String TAG = DataCollectorTask.class.getName();

    interface AsyncResponse {
        void completed(Properties props);
        void partial(Properties props);
    }

    protected static final String BUILD = "build";
    protected static final String BUILD_VERSION = "buildVersion";
    protected static final String WEB = "web";
    protected static final String CPU = "cpu";
    protected static final String MEMORY = "memory";
    protected static final String GPU = "gpu";
    protected static final String TELEPHONY = "telephony";
    protected static final String STORAGE = "storage";
    protected static final String SENSORS = "sensors";
    protected static final String DISPLAY = "display";
    protected static final String USB = "usb";
    protected static final String CAMERA = "camera";


    private DataCollector master;
    private Activity activityContext;
    private boolean gpuComplete = false, webComplete = false;

    public DataCollectorTask(Activity activityContext, DataCollector master) {
        this.activityContext = activityContext;
        this.master = master;
    }


    @Override
    public Void call() throws JSONException, InterruptedException {
        WebPropertiesLoader webLoader = new WebPropertiesLoader(activityContext);

        master.addProperties(BUILD, getBuildProperties());
        master.addProperties(BUILD_VERSION, getBuildVersionProperties());
        master.addProperties(CPU, CpuProperties.getProperties());
        master.addProperties(MEMORY, MemoryProperties.getProperties());
        master.addProperties(DISPLAY, DisplayProperties.getProperties(activityContext));
        master.addProperties(TELEPHONY, TelephonyProperties.getProperties(activityContext));
        master.addProperties(STORAGE, StorageProperties.getProperties());
        master.addProperties(USB, UsbProperties.getProperties());
        master.addProperties(CAMERA, CameraProperties.getProperties(activityContext));
        master.addProperties(SENSORS, SensorProperties.getProperties(activityContext));

        // load GPU and webview data. These are asynchronous so we can't handle results directly here.
        handleAsyncGpuData();
        handleAsyncWebData(webLoader);

        // block until we have the GPU/Web data
        while (!gpuComplete || !webComplete) {
            StringBuilder sb = new StringBuilder();
            sb.append("Waiting for");
            if(!gpuComplete) {
                sb.append(" GPU");
            }
            if(!webComplete) {
                sb.append(" WEB");
            }
            sb.append(" data...");

            Thread.sleep(100); // this may be interrupted if the task times out.
        }



        return null;
    }


    private void handleAsyncGpuData() {
        GpuLoader gpuLoader = new GpuLoader();
        gpuLoader.load(activityContext, new AsyncResponse(){

            @Override
            public void completed(Properties props) {
                try {
                    master.addProperties(GPU, props.getProperties());
                } catch(JSONException ex) {
                } finally {
                    gpuComplete = true;
                }
            }

            @Override
            public void partial(Properties props) {}
        });
    }


    private void handleAsyncWebData(WebPropertiesLoader webProperties) {

        webProperties.load(new AsyncResponse(){
            @Override
            public void completed(Properties props) {

                try {
                    master.addProperties(WEB, props.getProperties());
                } catch (JSONException ex) {
                } finally {
                    webComplete = true;
                }
            }

            @Override
            public void partial(Properties props) {
                try {
                    master.addProperties(WEB, props.getProperties());
                } catch (JSONException ex) {
                }
                // don't mark webComplete = true!
            }
        });
    }



    /**
     *
     * @return
     */
    private JSONObject getBuildProperties() throws JSONException {
        Field[] fields = android.os.Build.class.getDeclaredFields();
        return getPropertiesFromFields(fields, null);
    }


    /**
     *
     * @return
     */
    private JSONObject getBuildVersionProperties() throws JSONException {
        Field[] fields = android.os.Build.VERSION.class.getDeclaredFields();

        Set<String> exclude = new HashSet<>();
        exclude.add("SDK"); // deprecated in API 4 (old!) and superseded by SDK_INT

        return getPropertiesFromFields(fields, exclude);
    }



    /**
     *
     * @param fields
     * @param exclude
     * @return
     */
    private JSONObject getPropertiesFromFields(Field[] fields, Set<String> exclude) throws JSONException {
        JSONObject props = new JSONObject();

        for(Field field : fields) {
            //field.setAccessible(true);
            String key = field.getName();
            if(exclude != null && exclude.contains(key)) {
                continue;
            }

            try {
                key = StrUtil.toCamelCase(field.getName());
                String value = StrUtil.asString(field.get(key));
                props.put(key, value);
            } catch (IllegalAccessException e) {
            }
        }


        return props;
    }

}
