/*
 * Copyright (c) Afilias Technologies Ltd 2017. All rights reserved.
 */

package com.beaconsinspace.android.beacon.detector.deviceatlas;

import android.app.Activity;
import android.util.Log;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;


/**
 * <p>Collects system and hardware properties and returns them back via a callback function.<br>
 * Properties are collected for the following groups:</p>
 *
 * <ul>
 *     <li>build</li>
 *     <li>build_version</li>
 *     <li>web</li>
 *     <li>cpu</li>
 *     <li>memory</li>
 *     <li>gpu</li>
 *     <li>telephony</li>
 *     <li>storage</li>
 *     <li>sensors</li>
 *     <li>display</li>
 *     <li>usb</li>
 *     <li>camera</li>
 * </ul>
 *
 * <p>The data collection runs on a background thread as much as possible but certain properties
 *    require the UI thread. If the background thread does not complete within 10 seconds it will
 *    terminate and send the properties gathered up to that point.</p>
 *
 * @author Afilias Technologies Ltd
 */
public class DataCollector {
    private static final String TAG = DataCollector.class.getName();
    private static final long TIMEOUT_SEC = 10;

    private static final DateFormat DATE_FORMAT;
    static {
        DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mmZ");
        DATE_FORMAT.setTimeZone(TimeZone.getTimeZone("UTC"));
    }

    public static final String INFO = "info";
    private static final String VERSION = "version";
    private static final String SOURCE = "source";
    private static final String VERSION_NUM = "0.2";
    private static final String DATE = "date";
    public static final String DATA = "data";
    private static final String ERRORS = "errors";

    private Activity activityContext;
    private com.beaconsinspace.android.beacon.detector.deviceatlas.HandleDataCallback callback;
    private JSONObject allData = new JSONObject();

    public DataCollector(Activity activityContext, HandleDataCallback callback) {
        this.activityContext = activityContext;
        this.callback = callback;
        collectStats();
    }


    private void collectStats() {
        final DataCollector master = this;

        try {
            // avoid blocking the main UI thread...
            new Thread(new Runnable() {

                @Override
                public void run() {
                    // start another thread to collect the actual data. This one will timeout
                    // after X seconds. The data collected up to this point is then returned.

                    ExecutorService executor = Executors.newSingleThreadExecutor();
                    Future dataResult = executor.submit(new DataCollectorTask(activityContext, master));

                    try {
                        dataResult.get(TIMEOUT_SEC, TimeUnit.SECONDS); // blocks until timeout or thread completes

                    } catch (TimeoutException | InterruptedException | ExecutionException e) {
                        dataResult.cancel(true);

                        addError(prepareException(e));
                    } finally {
                        executor.shutdownNow();
                        doCallback();
                    }
                }
            }).start();

        } catch(Exception e) {
            // hmm do we REALLY want to catch the base Exception?
            // Goal is to avoid crashing the host application...
            Log.d(TAG, e.toString());
        }

    }


    /**
     * Add a collection of properties to the base allData.
     *
     * @param key
     * @param props
     */
    protected void addProperties(String key, Object props) {
        try {
            synchronized(allData) {
                allData.put(key, props);
            }
        } catch (JSONException e) {
            addError(e.toString());
        }
    }

    protected void addError(String err) {
        if ( err == null ) { return; }
        Log.d(TAG, err);

        synchronized(allData) {
            try {
                JSONArray errors;
                if(allData.has(ERRORS)) {
                    errors = allData.getJSONArray(ERRORS);
                } else {
                    errors = new JSONArray();
                }

                errors.put(err);
                allData.put(ERRORS, errors);
            } catch (JSONException e) {
                Log.d(TAG, e.toString());
            }
        }
    }

    /**
     * Iterate over the Throwable to capture just the bits we are interested in.
     *
     * @param t
     * @return
     */
    private String prepareException(Throwable t) {
        try
        {
            int count = 0; // keep counter to avoid potential infinite loop
            while(t.getCause() != null && count < 50) {
                t = t.getCause();
                count++;
            }

            StringBuilder sb = new StringBuilder();
            sb.append(t.toString());
            sb.append(" - ");

            // append the class/method/line if the exception came from our class
            String packageName = getClass().getPackage().getName();
            for(StackTraceElement st : t.getStackTrace()) {
                String className = st.getClassName();
                if(className.startsWith(packageName)) {
                    className = className.substring(packageName.length()+1); // +1 for .

                    sb.append(className)
                            .append(".")
                            .append(st.getMethodName())
                            .append("(")
                            .append(st.getLineNumber())
                            .append(")");
                    break;
                }
            }

            return sb.toString();
        }
        catch( Exception e )
        {
            return t.getMessage();
        }
    }


    private void doCallback() {
        final String toReturn = prepareFinalJsonStr();

        // run on UI thread in case the calling application wants to display the results.
        activityContext.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                callback.handleData(toReturn);
            }
        });
    }


    private String prepareFinalJsonStr() {
        String jsonStr;

        synchronized(allData) {
            try {
                JSONObject info = new JSONObject();
                info.put(VERSION, VERSION_NUM);
                info.put(DATE, DATE_FORMAT.format(new Date()));
                info.put(SOURCE, getAppName());

                JSONObject all = new JSONObject();
                all.put(INFO, info);
                all.put(DATA, allData);

                jsonStr = all.toString();
            } catch(JSONException ex) {
                jsonStr = "";
                Log.d(TAG, ex.toString());
            }
        }

        return jsonStr;
    }


    private String getAppName() {
        return activityContext.getApplicationInfo().loadLabel(activityContext.getPackageManager()).toString();
    }
}
