/*
 * Copyright (c) Afilias Technologies Ltd 2017. All rights reserved.
 */


package com.beaconsinspace.android.beacon.detector.deviceatlas;

import android.app.Activity;
import android.content.Context;
import android.os.Build;
import android.util.Log;
import android.webkit.WebSettings;
import android.webkit.WebView;

import java.lang.reflect.Constructor;

/**
 * @author Afilias Technologies Ltd
 */
public class WebPropertiesLoader {
    private static final String TAG = WebPropertiesLoader.class.getName();

    private final Activity activityContext;

    public WebPropertiesLoader(Activity activityContext) {
        this.activityContext = activityContext;
    }


    public void load(final DataCollectorTask.AsyncResponse callback) {
        callback.partial(new WebProperties()); // this will store some properties contained within WebProperties but not all the webview data.

        // get the UA for a webview, this tries to avoid creating a webview if possible as it can be
        // a slow operation.
        String ua = null;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            ua = WebSettings.getDefaultUserAgent(activityContext);
            Log.d(TAG, "User-Agent obtained from normal WebSettings call.");

        } else {
            // see if we can use reflection to get it, if not then fallback to creating a webview

            Constructor<WebSettings> constructor = null;
            try {
                constructor = WebSettings.class.getDeclaredConstructor(Context.class, WebView.class);
                constructor.setAccessible(true);

                WebSettings settings = constructor.newInstance(activityContext, null);
                ua = settings.getUserAgentString();
                Log.d(TAG, "User-Agent obtained from WebSettings via reflection.");
            } catch (Exception e) {
                Log.d(TAG, "Problem using reflection, fall back to WebView... "+e.toString());
                // fallback to creating webview - see below
            } finally {
                if(constructor != null) {
                    constructor.setAccessible(false);
                }
            }
        }

        if(ua != null) {
            // we're all done :)
            WebProperties props = new WebProperties();
            props.setWebviewUserAgent(ua);
            callback.completed(props);

        } else {
            // create a webiew - this is asynchronous as it needs to run on UI thread.
            createWebViewForUa(callback);
        }
    }




    private void createWebViewForUa(final DataCollectorTask.AsyncResponse callback) {
        activityContext.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                String ua = null;
                try {
                    WebView webview = new WebView(activityContext);
                    ua = webview.getSettings().getUserAgentString();

                    webview.clearHistory();
                    webview.clearCache(true);
                    webview.freeMemory();
                    webview.pauseTimers();
                    webview.destroy();
                    webview = null;

                } catch (Exception ex) {
                    // not sure what exceptions may get thrown... various stackoverflow pages
                    // suggest many possible...
                    Log.d(TAG, "Problem creating/destroying Webview. "+ex.toString());
                } finally {
                    if(ua!=null) {
                        Log.d(TAG, "User-Agent obtained from Webview");
                    }
                    WebProperties props = new WebProperties();
                    props.setWebviewUserAgent(ua);
                    callback.completed(props);
                }
            }
        });
    }


}
