package com.beaconsinspace.android.beacon.detector;

import android.location.Location;
import android.os.AsyncTask;
import android.os.Build;
import android.text.TextUtils;
import android.util.Log;

import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.Identifier;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;


/**
 * Created by kyleshank on 9/28/17.
 */

class BISNetworkingEventTask extends AsyncTask<BISCollectionEvent, Integer, String> {

    private String apiUrl;
    private String bisData;

    public BISNetworkingEventTask(String apiUrl, String bisData){
        this.apiUrl = apiUrl;
        this.bisData = bisData;
    }

    @Override
    protected String doInBackground(BISCollectionEvent... events) {
        if (events.length == 0){
            return null;
        }

        HttpURLConnection urlConnection;
        String result = null;

        BISCollectionEvent event = events[0];

        String authorizationHeader = BISNetworking.getSharedManger().getAuthorizationHeader();

        Map<String, String> parameters = new HashMap<String, String>();

        parameters.putAll(BISNetworking.getSharedManger().getCommonRequestArguments());
        parameters.putAll(BISNetworking.getSharedManger().getLocationRequestArguments(event.location));
        parameters.putAll(getEventArguments(event));

        String data = BISNetworking.encodeParams( parameters );

        try {
            urlConnection = (HttpURLConnection) new URL(apiUrl).openConnection();
            urlConnection.setDoOutput(true);
            urlConnection.setRequestProperty("Authorization", authorizationHeader);
            if (bisData != null){
                urlConnection.setRequestProperty("bisData", bisData);
            }
            urlConnection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
            urlConnection.setRequestProperty("Accept", "application/json");
            urlConnection.setRequestMethod("POST");
            urlConnection.connect();

            // Write
            OutputStream outputStream = urlConnection.getOutputStream();
            BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(outputStream, "UTF-8"));
            writer.write(data);
            writer.close();
            outputStream.close();

            // Read
            BufferedReader reader = new BufferedReader(new InputStreamReader(urlConnection.getInputStream(), "UTF-8"));

            String line;
            StringBuilder sb = new StringBuilder();

            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }

            reader.close();
            result = sb.toString();
        }
        catch (UnsupportedEncodingException e) {
            BISLog.e(this.getClass().getName(), "UnsupportedEncodingException: " + e.getMessage());
        }
        catch (IOException e) {
            BISLog.e(this.getClass().getName(), "IOException: " + e.getMessage());
        }
        catch( Exception e )
        {
            BISLog.e(this.getClass().getName(), "Exception: " + e.getMessage());
        }

        return result;
    }

    private Map<String, String> getEventArguments( BISCollectionEvent event )
    {
        Map<String,String> request = new HashMap<>();

        if (event != null){
            request.put("detect", event.getDetect());
            request.put("eventType", event.getEventType());
            request.put("batteryLevel", Float.toString(event.batteryLevel));
            request.put("batteryState", event.batteryState);

            if (event.date != null){
                request.put("creationTimeInEpochMillis", String.valueOf(event.date.getTime()));
            }

            String applicationState = "background";
            if (event.applicationState != null) {
                switch (event.applicationState) {
                    case ACTIVE:
                        applicationState = "active";
                        break;
                    case INACTIVE:
                        applicationState = "inactive";
                        break;
                    default:
                        applicationState = "background";
                        break;
                }
            }
            request.put("appStateDetection", applicationState);

            if (event.beacon != null){
                String[] identifiers = idsForBeacon(event.beacon);
                request.put("uuid", identifiers[0]);
                request.put("major", identifiers[1]);
                request.put("minor", identifiers[2]);
                if (event.eventDirection != null){
                    if(event.eventDirection == BISCollectionEvent.BISCollectionEventDirection.Exit){
                        String rssi = event.beaconRssiCollector.toJson();
                        if (rssi != null) {
                            request.put("rssi", rssi);
                        }
                    }
                }
                if (event.location!=null){
                    double dateCreatedGeographicPosition = event.location.getTime() / 1000.0;
                    request.put("dateCreatedGeographicPosition", Double.toString(dateCreatedGeographicPosition));
                }
            }
        }

        return request;
    }

    private String beaconIdentifierToString ( Identifier i )
    {
        try
        {
            if ( i == null ) { return null; }
            String s = i.toString();
            if ( s == null ) { return null; }

            if ( s.length() >= 2 && s.substring(0,2).equals("0x") )
            {
                s = s.substring(2); // get rid of the 0x
            }
            return s;
        }
        catch( Exception e )
        {
            BISLog.e( this.getClass().getName() ,"Exception: " + e.getMessage() );
        }
        return null;
    }

    private String[] idsForBeacon(Beacon beacon) {

        if(beacon == null) {
            return new String[]{"", "", ""};
        }


        Identifier id1 = null;
        Identifier id2 = null;
        Identifier id3 = null;
        try { id1 = beacon.getIdentifiers().size() > 0 ? beacon.getId1() : null; } catch( Exception e ){}
        try { id2 = beacon.getIdentifiers().size() > 1 ? beacon.getId2() : null; } catch( Exception e ){}
        try { id3 = beacon.getIdentifiers().size() > 2 ? beacon.getId3() : null; } catch( Exception e ){}

        String ID1 = id1 != null ? beaconIdentifierToString(id1) : "";
        String ID2 = id2 != null ? beaconIdentifierToString(id2) : "";
        String ID3 = id3 != null ? beaconIdentifierToString(id3) : "";

        return new String[]{ID1,ID2,ID3};
    }
}
