package com.beaconsinspace.android.beacon.detector;

import android.bluetooth.BluetoothAdapter;
import android.location.Location;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Base64;

import com.google.android.gms.location.Geofence;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.ExecutionException;

/**
 * Created by kyleshank on 9/28/17.
 */

class BISNetworking {
    private static final String TAG = "BIS_NETWORKING";

    static private BISNetworking sharedManger = null;
    static final String BIS_BASE_URL = "https://api.beaconsinspace.com";
    static final String BIS_BASE_ENDPOINT = "/v1/secure/initialize?uam=1";
    static final String BIS_URL_INITIALIZE = BIS_BASE_URL + BIS_BASE_ENDPOINT;

    private BISNetworking(){

    }

    public static BISNetworking getSharedManger(){
        if (sharedManger == null){
            sharedManger = new BISNetworking();
        }
        return sharedManger;
    }

    @NonNull
    public JSONObject getConfiguration(String apiUrl)
            throws InterruptedException, ExecutionException, JSONException {
        BISNetworkingGetTask urlCaller = new BISNetworkingGetTask();

        urlCaller.execute(apiUrl);

        String responseString = urlCaller.get();

        if (responseString==null){
            throw new JSONException("response is null");
        }

        return new JSONObject(responseString);
    }

    @NonNull
    public List<Geofence> getGeofences(@NonNull String apiUrl, @NonNull Location location)
            throws ExecutionException, InterruptedException, JSONException {
        BISNetworkingGetTask urlCaller = new BISNetworkingGetTask();

        Map<String,String> params = new HashMap<String, String>();
        params.put("latitude", Double.toString(location.getLatitude()));
        params.put("longitude", Double.toString(location.getLongitude()));
        params.put("gpsHorizontalAccuracy", Float.toString(location.getAccuracy()));
        params.put("gpsSpeed", Double.toString(location.getSpeed()));

        String apiUrlWithLocation = apiUrl + "?" +
                encodeParams(params);

        urlCaller.execute(apiUrlWithLocation);

        String responseString = urlCaller.get();

        if (responseString==null){
            throw new JSONException("response is null");
        }

        JSONObject response = new JSONObject(responseString);
        JSONObject data = response.getJSONObject("data");
        JSONArray array = data.getJSONArray("geofence");

        List<Geofence> geofences = new ArrayList<Geofence>();

        for(int i=0; i < array.length(); i++){
            JSONObject geofenceJson = array.getJSONObject(i);
            double latitude = geofenceJson.getDouble("latitude");
            double longitude = geofenceJson.getDouble("longitude");
            float radius = (float)geofenceJson.getDouble("radius");

            Geofence geofence = new Geofence.Builder()
                    .setRequestId(latitude+"+"+longitude+"+"+radius)
                    .setCircularRegion(
                            latitude,
                            longitude,
                            radius
                    )
                    .setExpirationDuration(Geofence.NEVER_EXPIRE)
                    .setTransitionTypes(Geofence.GEOFENCE_TRANSITION_ENTER |
                            Geofence.GEOFENCE_TRANSITION_DWELL |
                            Geofence.GEOFENCE_TRANSITION_EXIT)
                    .setLoiteringDelay(30000)
                    .build();

            geofences.add(geofence);
        }

        return geofences;
    }

    public Boolean reportSingleEvent( @NonNull String apiUrl,
                                   @NonNull BISCollectionEvent event,
                                   @Nullable String bisData) throws Exception {
        BISNetworkingEventTask task = new BISNetworkingEventTask(apiUrl, bisData);

        task.execute(event);

        String result = task.get();

        return (result != null);
    }

    public Boolean reportBatchEvent( @NonNull String apiUrl,
                                      @NonNull List<BISCollectionEvent> events,
                                      @Nullable String bisData) throws Exception {
        if (events.isEmpty()){
            return false;
        }

        BISLog.d(this.getClass().getName(), "reportBatchEvent("+events.size()+")");

        BISNetworkingEventsTask task = new BISNetworkingEventsTask(apiUrl, bisData);

        task.execute(events);

        String result = task.get();

        return (result != null);
    }

    public String getAuthorizationHeader() {
        String USER_KEY = BISDetector.API_KEY;
        String PACKAGE_NAME = BISDetector.getContext() != null ? BISDetector.getContext().getPackageName() : "";
        String authString = USER_KEY +":"+ PACKAGE_NAME;
        byte[] authBytes = authString.getBytes();
        String base64AuthString = Base64.encodeToString(authBytes, Base64.DEFAULT, authString.length(), Base64.NO_WRAP );
        String authorizationHeader = "Basic " + base64AuthString;
        return authorizationHeader;
    }

    public Map<String,String> getCommonRequestArguments()
    {
        Map<String,String> common = new HashMap<>();

        common.put("userId", ((BISDetector.UUID != null && BISDetector.UUID.toString() != null)? BISDetector.UUID.toString() : ""));
        common.put("userIdType", "AUUID");
        common.put("userId2", (BISDetector.ADID != null ? BISDetector.ADID : ""));
        common.put("userIdType2","ADID");
        common.put("os","ANDROID" + " " + Build.VERSION.RELEASE);
        common.put("device",Build.DEVICE);
        common.put("brand",Build.BRAND);
        common.put("model", Build.MODEL);
        common.put("manufacturer",Build.MANUFACTURER);
        common.put("userAgent",System.getProperty( "http.agent" ));
        common.put("tz",TimeZone.getDefault().getID());
        common.put("language",Locale.getDefault().toString());
        common.put("country",Locale.getDefault().toString());
        common.put("sdkVersion",BISDetector.SDK_VERSION);
        common.put("creationTimeInEpochMillis", String.valueOf(System.currentTimeMillis()));
        common.put("isBluetoothEnabled", isBluetoothEnabled() ? "1" : "0");

        if (BISPersistentStorage.isDeviceMetaDataCollected() && !BISPersistentStorage.isdeviceMetaDataSent()){
            String deviceMetaDataString = BISPersistentStorage.getDeviceMetaDataString();
            if(deviceMetaDataString!=null) {
                BISPersistentStorage.setDeviceMetaDataSent(true);
                common.put("deviceMetaData", deviceMetaDataString);
            }
        }

        return common;
    }

    public Map<String, String> getLocationRequestArguments( Location location )
    {
        Map<String,String> request = new HashMap<>();

        if (location != null){
            request.put("gpsLatitude", Double.toString(location.getLatitude()));
            request.put("gpsLongitude", Double.toString(location.getLongitude()));
            request.put("gpsHorizontalAccuracy", Float.toString(location.getAccuracy()));
            request.put("gpsAltitude", Double.toString(location.getAltitude()));
            request.put("gpsSpeed", Float.toString(location.getSpeed()));
            request.put("gpsBearing", Float.toString(location.getBearing()));
        }

        return request;
    }

    public static String encodeParams( Map<String,String> parameters ){
        List<String> encoded = new ArrayList<>();

        for (Map.Entry<String, String> entry : parameters.entrySet()) {
            try {
                if (entry.getValue() != null) {
                    encoded.add(entry.getKey() + "=" + URLEncoder.encode(entry.getValue(), "UTF-8"));
                }
            } catch (UnsupportedEncodingException e) {
                BISLog.e(TAG, "UnsupportedEncodingException: " + e.getMessage());
            }
        }

        return TextUtils.join("&", encoded.toArray());
    }

    private Boolean isBluetoothEnabled() {
        BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();

        if ( adapter == null ) { return false; }
        return adapter.isEnabled();
    }
}
