package com.beaconsinspace.android.beacon.detector;

import android.app.Activity;
import android.app.Application;
import android.app.Service;
import android.app.job.JobInfo;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;
import android.util.Log;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.common.GooglePlayServicesRepairableException;

import java.io.IOException;
import java.util.UUID;

public class BISDetector extends Service implements
        Application.ActivityLifecycleCallbacks{

    static public String SDK_VERSION = "2.0.0";
    static private final String TAG = "BIS_API";

    static String API_KEY = "";
    static String ADID = "";
    static UUID UUID;

    private static BISDetector sharedInstance;
    static BISLogDelegate logDelegate;
    private BISCollectionManager collectionManager;

    private Thread primaryProcessThread;

    static Context appContext;

    public static BISDetector getSharedInstance(){
        if (sharedInstance == null){
            sharedInstance = new BISDetector();
        }
        return sharedInstance;
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle bundle) {
        applicationState = ApplicationState.INACTIVE;
    }

    @Override
    public void onActivityStarted(Activity activity) {
        applicationState = ApplicationState.ACTIVE;
    }

    @Override
    public void onActivityResumed(Activity activity) {
        applicationState = ApplicationState.ACTIVE;
    }

    @Override
    public void onActivityPaused(Activity activity) {
        applicationState = ApplicationState.BACKGROUND;
    }

    @Override
    public void onActivityStopped(Activity activity) {
        applicationState = ApplicationState.BACKGROUND;
    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle bundle) {

    }

    @Override
    public void onActivityDestroyed(Activity activity) {

    }

    enum ApplicationState {
        ACTIVE,
        INACTIVE,
        BACKGROUND
    }

    public ApplicationState applicationState = ApplicationState.BACKGROUND;

    /**
     * Constructor for the BeaconsInSpace Detector
     *
     * @param key String
     * @param ctx Context
     */
    public static void configure(String key, Context ctx) {
        Log.d(TAG, "Configuring BISDetector SDK " + SDK_VERSION);

        // Set props
        API_KEY = key;
        appContext = ctx.getApplicationContext();//always get appContext
        logDelegate=null;

        if (appContext instanceof Application) {
            Application application = (Application) appContext;
            application.registerActivityLifecycleCallbacks(getSharedInstance());
        }

        // Store in persistent storage
        BISPersistentStorage.storeString( BISPersistentStorage.KEY_APIKEY, key );

        //init deviceAtlas lib
        if(!BISPersistentStorage.isDeviceMetaDataCollected()) {
            //data is not yet collected from deviceAtlas lib, so booting deviceAtlas..
            getSharedInstance().bootDeviceAtlas();
        }

        scheduleJob(appContext);

        // spawn sticky service
        if ( Build.VERSION.SDK_INT >= Build.VERSION_CODES.O ) {
            BISDetector detector = getSharedInstance();
            if (detector != null) {
                detector.foregroundService();
            }
        }else {
            appContext.sendBroadcast(new Intent("BootstrapBeaconsInSpace"));
        }
    }

    private void foregroundService(){
        if (primaryProcessThread == null ) {
            primaryProcessThread = new Thread()
            {
                public void run()
                {
                    try {
                        bootstrap(API_KEY, appContext);
                    }
                    catch (Throwable e) {
                        BISLog.e(TAG,e.getMessage());
                    }
                }
            };
            primaryProcessThread.start();
        }
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        appContext = this.getApplicationContext();
        API_KEY = BISPersistentStorage.getString( BISPersistentStorage.KEY_APIKEY );

        if (primaryProcessThread == null ) {
            primaryProcessThread = new Thread()
            {
                public void run()
                {
                    try {
                        bootstrap(API_KEY, appContext);
                    }
                    catch (Throwable e) {
                        BISLog.e(TAG,e.getMessage());
                    }
                }
            };
            primaryProcessThread.start();
        }

        return START_STICKY;
    }

    private static void scheduleJob(Context context){
        if ( Build.VERSION.SDK_INT >= Build.VERSION_CODES.O ) {
            BISLog.d(TAG, "Scheduling job");

            int jobId = 1337;
            ComponentName mServiceComponent = new ComponentName(context, BISJobService.class);
            JobInfo.Builder builder = new JobInfo.Builder(jobId, mServiceComponent);
            builder.setPeriodic(BISConfiguration.getCurrentConfiguration().locationMonitoringInterval);
            builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);

            JobScheduler tm = (JobScheduler) context.getSystemService(Context.JOB_SCHEDULER_SERVICE);
            tm.cancel(jobId);
            tm.schedule(builder.build());
        }
    }

    public void bootstrap(String key, Context ctx) {
        // Ensure library should run
        if (!isSafeToBootstrap()) {
            Log.d(TAG, "This device is not supported. BeaconsInSpace Detector shutting down");
            return;
        }

        // Set API Key
        API_KEY = key != null ? key : "";

        // Get config
        BISConfiguration.getCurrentConfiguration();

        // Get UUID
        BISDeviceUUID deviceUUID = new BISDeviceUUID(ctx);
        UUID = deviceUUID.getDeviceUuid();

        // Read Users Advertisement Identifier
        BISDetector.getUserADID();

        // Create and start Collection Manager
        collectionManager = new BISCollectionManager(ctx);
    }

    /*
     * ===================================SERVICE METHODS=======================================
     */
    @Override
    public IBinder onBind(Intent arg0) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public void onDestroy() {
        //send broadcast before destroying service
        sendBroadcast(new Intent("BootstrapBeaconsInSpace"));
        super.onDestroy();
    }
    /*
     * ===========================================================================================
     */

    /**
     * Checks for unsupported device models
     * @return true if the device is supported; false otherwise
     */
    private static boolean isSafeToBootstrap() {
        // Check that BLE is available
        if ( Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR2 ) {
            Log.e(TAG,"BeaconsInSpace Detector Library does not run on Android: "+Build.VERSION.SDK_INT);
            return false;
        }
        return true;
    }

    /**
     * Checks for required services like location, internet, bluetooth
     * Gets beacons data from servers
     */
    static void performInitialSetup() {
        Log.i(TAG,"BeaconsInSpace has bootstrapped successfully");
    }

    static void getUserADID() {

        if(appContext == null) { return; }

        Thread thread = new Thread() {

            public void run() {
                AdvertisingIdClient.Info adInfo = null;

                try {
                    adInfo = AdvertisingIdClient.getAdvertisingIdInfo(appContext);
                } catch (IOException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } catch (GooglePlayServicesRepairableException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } catch (GooglePlayServicesNotAvailableException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } catch (Exception e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } finally {
                    if(adInfo == null) {
                        onUserADIDReceiveFail();
                    }
                    else {
                        boolean limitAdTrackingIsEnabled = adInfo.isLimitAdTrackingEnabled();
                        ADID = limitAdTrackingIsEnabled ? "00000000-0000-0000-0000-000000000000" : adInfo.getId();
//                            final boolean isLAT = adInfo.isLimitAdTrackingEnabled();
                        final boolean success = ADID != null && ADID.length() > 0;

                        // NOTE: we need to post in to execute on UI thread, because user can have UI updates in his overload.
                        Handler handler = new Handler(Looper.getMainLooper());
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                if (success)
                                    onUserADIDReceiveSuccess();
                                else
                                    onUserADIDReceiveFail();
                            }
                        });
                    }
                } // finally
            } // Thread run()
        }; // new Thread()

        thread.start();
    }

    static void onUserADIDReceiveSuccess() {
        BISDetector.performInitialSetup();
    }

    static void onUserADIDReceiveFail() {
        BISDetector.performInitialSetup();
    }

    /**
     * boot deviceAtlas lib with a translucent Activity and collect data
     */
    private void bootDeviceAtlas(){
        //BISDeviceAtlas is used to provide deviceAtlas with activityContext
        Intent transparentActivityIntent = new Intent(appContext, BISDeviceAtlas.class);
        transparentActivityIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        appContext.startActivity(transparentActivityIntent);
    }

    public static Context getContext() {
        return appContext;
    }

    public BISCollectionManager getCollectionManager(Context context){
        if (this.collectionManager != null){
            return this.collectionManager;
        }
        String apiKey = API_KEY = BISPersistentStorage.getString( BISPersistentStorage.KEY_APIKEY );
        if (apiKey != null){
            bootstrap(apiKey, context);
            return this.collectionManager;
        }
        return null;
    }
}
