package com.beaconsinspace.android.beacon.detector;

import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.location.Location;
import android.os.Build;
import android.util.Base64;
import android.util.Log;

import org.altbeacon.beacon.Beacon;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;
import java.util.concurrent.ExecutionException;

class BISDetectorREST {

    // Defines
    static final String TAG = "BIS_REST";

    static final String BIS_BASE_URL = "https://api.beaconsinspace.com";
    static final String BIS_URL_EVENT = BIS_BASE_URL+"/v1/event";
    static final String BIS_URL_BEACON = BIS_BASE_URL+"/v1/beacon";
    static final String BIS_URL_INITIALIZE = BIS_BASE_URL+"/v1/secure/initialize?uam=1";

    // Local
    static BISDetectorInternalDelegate delegate;

    // Setup
    static public void setDelegate(BISDetectorInternalDelegate _delegate) {
        delegate = _delegate;
    }

    static public ArrayList<String> getBeaconsInfoFromServer() {

        Log.i( TAG, "Getting beacon identifiers from BeaconsInSpace API" );

        String responseString = getBeaconsList();

        /*
         * Verify API returned JSON
         */
        if(responseString == null || responseString.length() == 0)
        {
            BISDetector.sharedInstance.onBeaconsInfoReceiveFail( BISDetectorInternalDelegate.ERROR_CODE_SERVICE_UNAVAILABLE, null );
            return null;
        }

        /*
        *   As response we receive JSON.
        *   We need to parse it and get array of UUIDs of iBeacons to be monitored for this user.
        *
        *   JSON structure of response in format (key : value):
        *
        *   code        :   value (Int)
        *   message     :   value (String)
        *   data        :   value (Array of Dictionaries)
        *   |
        *   |-> single element of 'data'
        *       protocol    :   value (String)
        *       uuid        :   value (String)
        *
        */

        ArrayList<String> uuids = new ArrayList<String>();

        try {
            JSONObject jsonObject = new JSONObject(responseString);

            /*
             * Verify status code of request
             */
            int responseCode = jsonObject.getInt("code");
            if ( responseCode != 200 )
            {
                BISDetector.sharedInstance.onBeaconsInfoReceiveFail( responseCode, null );
                return null;
            }

            JSONArray jsonArrayData = jsonObject.getJSONArray("data");
            if (jsonArrayData == null) return null;


            for (int i = 0; i < jsonArrayData.length(); i++) {

                JSONObject jsonObjectBeacon = jsonArrayData.getJSONObject(i);

                String protocol = jsonObjectBeacon.getString("protocol");

                if(protocol.equalsIgnoreCase("ibeacon")) {
                    String uuid = jsonObjectBeacon.getString("uuid");

                    if(uuid != null)
                        uuids.add(uuid);
                }
            }

            // If there are some uuids - start monitoring
            if (uuids.size() > 0) {
                BISDetector.beaconsManager().updateUUIDs(uuids);

                if(delegate != null)
                    delegate.onBeaconsInfoReceiveSuccess();
            }
        }
        catch (JSONException e) {
            Log.e(TAG, "JSONException: " + e.getMessage());
            BISDetector.sharedInstance.onBeaconsInfoReceiveFail( BISDetectorInternalDelegate.ERROR_CODE_SERVICE_UNAVAILABLE, null );
        }

        return uuids;
    }


    static private String getBeaconsList() {

        // Perform URL call to BISDetector API

        String urlToCall = BISDetectorREST.BIS_URL_BEACON + "?" + getCommonRequestArguments();

        BISDetectorAsyncRequest urlCaller = new BISDetectorAsyncRequest();

        String responseString = null;

        try {
            urlCaller.execute(urlToCall);
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "IllegalStateException: " + e.getMessage());
        }
        catch( Exception e )
        {
            e.printStackTrace();
        }
        finally {

            try {
                responseString = urlCaller.get();
            }
            catch (InterruptedException e) {
                Log.e(TAG, "InterruptedException: " + e.getMessage());
            }
            catch (ExecutionException e) {
                Log.e(TAG, "ExecutionException: " + e.getMessage());
            }
            catch( Exception e )
            {
                e.printStackTrace();
            }
        }

        return responseString;
    }


    static public String notifyAboutBeaconEnter(Beacon beacon) {
        return sendBeaconEvent(beacon, true);
    }

    static public String notifyAboutBeaconExit(Beacon beacon)
    {
        return sendBeaconEvent(beacon, false);
    }

    static private String getCommonRequestArguments()
    {
        String os = "";
        String device = "";
        String brand = "";
        String model = "";
        String manufacturer = "";
        String userAgent = "";
        String tz = "";
        String language = "";
        String country = "";
        String sdkVersion = "";

        try
        {
            os = "ANDROID" + URLEncoder.encode(" " + Build.VERSION.RELEASE, "UTF-8");
            device = URLEncoder.encode(Build.DEVICE, "UTF-8");
            brand = URLEncoder.encode(Build.BRAND, "UTF-8");
            model = URLEncoder.encode(Build.MODEL, "UTF-8");
            tz = URLEncoder.encode(TimeZone.getDefault().getID(), "UTF-8" );
            language = URLEncoder.encode(Locale.getDefault().toString(), "UTF-8" );
            country = URLEncoder.encode(Locale.getDefault().toString(), "UTF-8" );
            userAgent = URLEncoder.encode(System.getProperty( "http.agent" ), "UTF-8" );
            sdkVersion = URLEncoder.encode(BISDetector.SDK_VERSION, "UTF-8" );
            manufacturer = URLEncoder.encode(Build.MANUFACTURER, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            Log.e(TAG, "UnsupportedEncodingException: " + e.getMessage());
        }
        double dateCreated = System.currentTimeMillis()/1000.0;

        return String.format
                (
                        "userId=%s" +
                        "&userIdType=%s" +
                        "&userId2=%s" +
                        "&userIdType2=%s" +
                        "&os=%s" +
                        "&device=%s" +
                        "&brand=%s" +
                        "&model=%s" +
                        "&manufacturer=%s" +
                        "&userAgent=%s" +
                        "&tz=%s" +
                        "&language=%s" +
                        "&country=%s" +
                        "&sdkVersion=%s" +
                        "&createdAt=%f",
                        (BISDetector.UUID.toString() != null ? BISDetector.UUID.toString() : ""),
                        "AUUID",
                        (BISDetector.ADID != null ? BISDetector.ADID : ""),
                        "ADID",
                        os,
                        device,
                        brand,
                        model,
                        manufacturer,
                        userAgent,
                        tz,
                        language,
                        country,
                        sdkVersion,
                        dateCreated
                );
    }

    static private String getLocationRequestArguments( Location location )
    {
        String s = "";

        if ( location != null )
        {
            String latitude = Double.toString(location.getLatitude());
            String longitude = Double.toString(location.getLongitude());
            String horizationAccuracy = Float.toString(location.getAccuracy());
            String altitude = Double.toString(location.getAltitude());
            String speed = Float.toString(location.getSpeed());
            String bearing = Float.toString(location.getBearing());
            s +=
                    "&gpsLatitude="+latitude
                    +"&gpsLongitude="+longitude
                    +"&gpsHorizontalAccuracy="+horizationAccuracy
                    +"&gpsAltitude="+altitude
                    +"&gpsSpeed="+speed
                    +"&gpsBearing="+bearing;
        }

        return s;
    }

    static private String sendBeaconEvent(Beacon beacon, boolean isEnter) {

        String[] beaconIds = BISDetectorManager.idsForBeacon(beacon);
        String beaconId = BISDetectorManager.uniqueIdentifierForBeacon(beacon);

        if(beaconIds == null || beaconIds.length < 3)
            return null;

        Location location = BISLocationListener.getLocationByBeaconId(beaconId);
        String locationArguments = getLocationRequestArguments( location );

        // Build custom data string
        String customDataString = String.format("uuid=%s&major=%s&minor=%s&detect=%s", beaconIds[0], beaconIds[1], beaconIds[2], (isEnter ? "enter" : "exit")); // Attempt to add GPS coordinates
        String commonDataString = getCommonRequestArguments();
        String dataString = customDataString + "&" + commonDataString;
        if ( ! locationArguments.equals("") )
        {
            dataString += "&" + locationArguments;
        }


        // Collect additional beacon information
        HashMap<String,String> beaconData = BISDetectorManager.collectBeaconInfo( beacon );
        for ( String key : beaconData.keySet() )
        {
            String value = beaconData.get( key );
            if ( value != null )
            {
                try
                {
                    String encodedKey = URLEncoder.encode( key, "UTF-8" );
                    String encodedValue = URLEncoder.encode( value, "UTF-8" );
                    dataString+="&"+encodedKey+"="+encodedValue;
                }
                catch (UnsupportedEncodingException e){}
            }
        }

        // If its exit get rssi data
        if ( ! isEnter )
        {
            try
            {
                String rssiJson = BISDetectorManager.getBeaconRssiJson( beacon );
                Integer gpsRssi = BISDetectorManager.getBeaconGPSRssi( beacon );
                String gpsRssiString = "";
                if ( gpsRssi != null ) { gpsRssiString = gpsRssi.toString(); }
                dataString+="&rssi="+URLEncoder.encode( rssiJson, "UTF-8" );
                dataString+="&gpsRssi="+URLEncoder.encode( gpsRssiString, "UTF-8" );
            }
            catch (UnsupportedEncodingException e) {}
        }

        if(!BISPersistentStorage.isdeviceMetaDataSent()){
            String deviceMetaDataEncoded = getDeviceMetaData();
            dataString += "&deviceMetaData="+deviceMetaDataEncoded;
            BISPersistentStorage.setDeviceMetaDataSent(true);
        }

        final String finalDataString = dataString;
        final String[] resultStringData = new String[1];

        Thread thread = new Thread() {

            public void run() {
                resultStringData[0] = sendData(finalDataString);
            }
        };

        try
        {
            thread.start();
            thread.join();
        }
        catch( Exception e )
        {
            Log.e( TAG, "sendBeaconEvent thread error:"+ e.getMessage() );
        }
        return resultStringData[0];
    }

    static String getAuthorizationHeader()
    {
        String USER_KEY = BISDetector.API_KEY;
        String PACKAGE_NAME = BISDetector.getContext() != null ? BISDetector.getContext().getPackageName() : "";
        String authString = USER_KEY +":"+ PACKAGE_NAME;
        byte[] authBytes = authString.getBytes();
        String base64AuthString = Base64.encodeToString( authBytes, Base64.DEFAULT, authString.length(), Base64.NO_WRAP );
        String authorizationHeader = "Basic " + base64AuthString;
        return authorizationHeader;
    }

    private static String getDeviceMetaData()
    {
        String deviceMetaData = BISPersistentStorage.getDeviceMetaDataString();
        String deviceMetaDataEncoded="";
        try
        {
            deviceMetaDataEncoded = URLEncoder.encode(deviceMetaData,"UTF-8");
        }
        catch( Exception e )
        {
            Log.e(TAG,"Failed to encode deviceMetaData string: "+e.getMessage() );
        }
        return deviceMetaDataEncoded;
    }

    static String sendGPSEvent(Location location, boolean sendAdditionalData )
    {
        if ( location == null ) { return null; }

        // Build data string
        String customDataString = String.format("detect=%s", "gps");
        String commonDataString = getCommonRequestArguments();
        String locationArguments = getLocationRequestArguments( location );
        String dataString = customDataString + "&" + commonDataString;
        if ( ! locationArguments.equals("") )
        {
            dataString += "&" + locationArguments;
        }

        if(!BISPersistentStorage.isdeviceMetaDataSent()){
            String deviceMetaDataEncoded = getDeviceMetaData();
            dataString += "&deviceMetaData="+deviceMetaDataEncoded;
            BISPersistentStorage.setDeviceMetaDataSent(true);
        }

        final String finalDataString = dataString;
        final String[] resultStringData = new String[1];

        Thread thread = new Thread() {
            public void run() {
                resultStringData[0] = sendData(finalDataString);
            }
        };

        try
        {
            thread.start();
            thread.join();
        }
        catch( Exception e )
        {
            Log.e( TAG, "sendBeaconEvent thread error: "+ e.getMessage() );
        }

        return resultStringData[0];
    }

    static String collectInstalledPackages()
    {
        String appString = "";
        try
        {
            final PackageManager pm = BISDetector.sharedInstance.getPackageManager();
            List<ApplicationInfo> applications = pm.getInstalledApplications(PackageManager.GET_META_DATA);
            boolean isFirst = true;
            for (ApplicationInfo appInfo : applications)
            {
                if ( appInfo.sourceDir.toLowerCase().startsWith("/system") ) { continue; }
                if ( ! isFirst ) { appString += "|"; }
                else { isFirst = false; }
                appString += appInfo.packageName;
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        return appString;
    }

    static String sendData(String data) {

        if(data == null) { return null; }

        HttpURLConnection urlConnection;
        String url = BISDetectorREST.BIS_URL_EVENT;
        String result = null;

        // prepare auth header
        String authorizationHeader = getAuthorizationHeader();

        try {
            urlConnection = (HttpURLConnection) new URL(url).openConnection();
            urlConnection.setDoOutput(true);
            urlConnection.setRequestProperty("Authorization", authorizationHeader);
            urlConnection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
            urlConnection.setRequestProperty("Accept", "application/json");
            urlConnection.setRequestMethod("POST");
            urlConnection.connect();

            // Write
            OutputStream outputStream = urlConnection.getOutputStream();
            BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(outputStream, "UTF-8"));
            writer.write(data);
            writer.close();
            outputStream.close();

            // Read
            BufferedReader reader = new BufferedReader(new InputStreamReader(urlConnection.getInputStream(), "UTF-8"));

            String line;
            StringBuilder sb = new StringBuilder();

            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }

            reader.close();
            result = sb.toString();
        }
        catch (UnsupportedEncodingException e) {
            Log.e(TAG, "UnsupportedEncodingException: " + e.getMessage());
            return null;
        }
        catch (IOException e) {
            Log.e(TAG, "IOException: " + e.getMessage());
            return null;
        }
        catch( Exception e )
        {
            e.printStackTrace();
        }

        return result;
    }

    static HashMap<String,String> getInitializationData()
    {
        HashMap<String, String> data = new HashMap<String,String>();

        BISDetectorAsyncRequest urlCaller = new BISDetectorAsyncRequest();

        String responseString = null;

        try {
            urlCaller.execute(BIS_URL_INITIALIZE);
        }
        catch (IllegalStateException e) {
            Log.e(TAG, "IllegalStateException: " + e.getMessage());
        }
        catch( Exception e )
        {
            e.printStackTrace();
        }
        finally
        {
            try
            {
                responseString = urlCaller.get();
            }
            catch (InterruptedException e)
            {
                Log.e(TAG, "InterruptedException: " + e.getMessage());
            }
            catch (ExecutionException e)
            {
                Log.e(TAG, "ExecutionException: " + e.getMessage());
            }
        }

        if ( responseString == null ) { return null; }

        try
        {
            JSONObject jsonObject = new JSONObject(responseString);

            /*
             * Verify status code of request
             */
            int responseCode = jsonObject.getInt("code");
            if ( responseCode != 200 )
            {
                BISDetector.sharedInstance.onBeaconsInfoReceiveFail( responseCode, null );
                return null;
            }

            JSONObject jsonObjectData = jsonObject.getJSONObject("data");
            Iterator<String> keyIterator = jsonObjectData.keys();
            while ( keyIterator.hasNext() )
            {
                String key = keyIterator.next();
                String value = jsonObjectData.getString(key);
                data.put(key,value);
            }
        }
        catch (JSONException e)
        {
            Log.e(TAG, "JSONException: " + e.getMessage());
            BISDetector.sharedInstance.onBeaconsInfoReceiveFail( BISDetectorInternalDelegate.ERROR_CODE_SERVICE_UNAVAILABLE, null );
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }

        return data;
    }


}















