package com.beaconsinspace.android.beacon.detector;

import android.bluetooth.BluetoothAdapter;
import android.content.Context;
import android.location.LocationManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.provider.Settings;
import android.util.Log;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.common.GooglePlayServicesRepairableException;

import java.io.IOException;
import java.util.UUID;

public class BISDetector implements BISDetectorInternalDelegate
{
    /*
     * VERSION
     */
    static public String SDK_VERSION = "1.2.1";

    /*
     * Definitions
     */
    static private final String TAG = "BIS_API";

    static String API_KEY = "";
    static String ADID = "";
    static UUID UUID;

    static BISDetector sharedInstance = new BISDetector();
    static Context context;
    static BISDetectorManager beaconsManager = new BISDetectorManager();
    static BISDetectorDelegate delegate;

    /**
     * Constructor for the BeaconsInSpace Detector
     *
     * @param key String
     * @param ctx Context
     * @param dlgt BISDetectorDelegate
     */
    public static void configure(String key, Context ctx, BISDetectorDelegate dlgt) {

        // Log occurrance
        Log.d(TAG, "Configuring BISDetector SDK " + SDK_VERSION);

        // Ensure library should run
        if ( ! isSafeToBootstrap() )
        {
            Log.d(TAG, "This device is not supported. BeaconsInSpace Detector shutting down");
            return;
        }

        // 0. Set Delegate
        delegate = dlgt;

        // 1. Set API Key
        API_KEY = key != null ? key : "";

        // 2. Set Context
        context = ctx;
        beaconsManager.setContextAndInit(ctx);

        // 3. Set Internal Delegate
        BISDetectorInternalDelegate internalDelegate = sharedInstance;
        BISDetectorREST.setDelegate(internalDelegate);
        BISDetectorServicesListener.setDelegate(internalDelegate);
        BISDetectorManager.setDelegate(internalDelegate);

        // 4. Get UUID
        BISDeviceUUIDFactory UUIDFactory = new BISDeviceUUIDFactory(context);
        UUID = UUIDFactory.getDeviceUuid();

        // 4. Read Users Advertisement Identifier
        BISDetector.getUserADID();
    }

    private static boolean isSafeToBootstrap()
    {
        // Check that BLE is available
        if ( Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR2 )
        {
            Log.e(TAG,"BeaconsInSpace Detector Library does not run on Android: "+Build.VERSION.SDK_INT);
            return false;
        }
        // Check that it is not a Bluetooth/WIFI collision problem device

        String[] unsupportedDeviceModels = { "Nexus 4","Nexus 7","XT1028","XT1031","XT937C","XT1002","XT1003","XT1008","XT1032","XT1034","XT939G","XT1033","XT1069","titan_retbr_dstv","XT1063","XT1064","XT1068","titan_niibr_ds","MotoG3","Moto G (4)","XT1032","MotoG3","TE","XT1039","XT1040","XT1042","XT1045","XT1072","XT1077","XT1079","XT1063","XT1078","XT1225","XT1225","XT1049","XT1050","XT1052","XT1053","XT1055","XT1056","XT1058","XT1060","XT1085","XT1092","XT1093","XT1094","XT1095","XT1096","XT1097","XT1098","XT1580","XT1580","XT1581","XT1021","XT1562","XT1563","XT1564","XT1021","XT1562","XT1563","XT1575","XT1572","XT1570","XT1572" };
        String deviceModel = Build.MODEL;
        for( String unsupportedModel : unsupportedDeviceModels )
        {
            if ( deviceModel.equals(unsupportedModel) )
            {
                return false;
            }
        }

        return true;
    }

    /**
     *
     * Set duration of bluetooth scan and sleep periods in both the foreground and background
     * If any of the values are null then the respective setting will not be modified
     *
     * @param foregroundScanPeriod
     * @param foregroundBetweenScanPeriod
     * @param backgroundScanPeriod
     * @param backgroundBetweenScanPeriod
     */
    public static void setBeaconManagerScanPeriods
    (
        Long foregroundScanPeriod,
        Long foregroundBetweenScanPeriod,
        Long backgroundScanPeriod,
        Long backgroundBetweenScanPeriod
    )
    {
        beaconsManager.setScanPeriods( foregroundScanPeriod, foregroundBetweenScanPeriod, backgroundScanPeriod, backgroundBetweenScanPeriod );
    }

    static void performInitialSetup()
    {

        /*
         * Verify all required services are enabled
         */
        String errorMessage = "";
        if (!isLocationServiceEnabled())
        {
            errorMessage = "Location Services are not enabled. Please enable them in Settings.";
            Log.e( TAG, errorMessage );
            if ( delegate != null ) { delegate.onBISError( ERROR_CODE_DEPENDENCIES_DISABLED, errorMessage ); }
            return;
        }

        if (!isInternetAvailable())
        {
            errorMessage = "Network Services are not enabled. Please enable them in Settings.";
            Log.e( TAG, errorMessage );
            if ( delegate != null ) { delegate.onBISError( ERROR_CODE_DEPENDENCIES_DISABLED, errorMessage ); }
            return;
        }

        if (!isBluetoothEnabled())
        {
            errorMessage = "Bluetooth is not enabled. Please turn it on to proceed";
            Log.e( TAG, errorMessage );
            if ( delegate != null ) { delegate.onBISError( ERROR_CODE_DEPENDENCIES_DISABLED, errorMessage ); }
            return;
        }

        /*
         * Get beacon identifiers
         */
        BISDetectorREST.getBeaconsInfoFromServer();
    }

    static BISDetectorManager beaconsManager() {
        return beaconsManager;
    }

    static void getUserADID() {

        if(context == null) { return; }

        Thread thread = new Thread() {

            public void run() {
                AdvertisingIdClient.Info adInfo = null;

                try {
                    adInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
                } catch (IOException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } catch (GooglePlayServicesRepairableException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } catch (GooglePlayServicesNotAvailableException e) {
                    Log.e(TAG, "getAdvertisingIdInfo exception: " + e.toString());
                } finally {
                    if(adInfo == null) {
                        onUserADIDReceiveFail();
                    }
                    else {
                        ADID = adInfo.getId();
//                            final boolean isLAT = adInfo.isLimitAdTrackingEnabled();
                        final boolean success = ADID != null && ADID.length() > 0;

                        // NOTE: we need to post in to execute on UI thread, because user can have UI updates in his overload.
                        Handler handler = new Handler(Looper.getMainLooper());
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                if (success)
                                    onUserADIDReceiveSuccess();
                                else
                                    onUserADIDReceiveFail();
                            }
                        });
                    }
                } // finally
            } // Thread run()
        }; // new Thread()

        thread.start();
    }

    static public boolean isLocationServiceEnabled()
    {
        if(context == null)
            return false;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT)
        {
            int locationMode = 0;
            try
            {
                locationMode = Settings.Secure.getInt(context.getContentResolver(), Settings.Secure.LOCATION_MODE);
            }
            catch (Settings.SettingNotFoundException e)
            {
                Log.e(TAG,"Failed to check for location services: "+e.getMessage());
                return false;
            }
            return locationMode != Settings.Secure.LOCATION_MODE_OFF;
        }
        else
        {
            LocationManager lm = (LocationManager)context.getSystemService(Context.LOCATION_SERVICE);
            boolean gps_enabled = false;
            boolean network_enabled = false;

            try { gps_enabled = lm.isProviderEnabled(LocationManager.GPS_PROVIDER); } catch(Exception ex) {}
            try { network_enabled = lm.isProviderEnabled(LocationManager.NETWORK_PROVIDER); } catch(Exception ex) {}

            return gps_enabled || network_enabled;
        }
    }

    static public boolean isInternetAvailable() {

        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();

        return activeNetworkInfo != null && activeNetworkInfo.isConnected();
    }

    static public boolean isBluetoothEnabled() {

        BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();

        if ( adapter == null ) { return false; }
        return adapter.isEnabled();
    }

    static public void startRanging() {
        beaconsManager.setContextAndInit(context);
        beaconsManager.startRanging();
    }

    static public void stopRanging() {
        beaconsManager.stopRanging();
    }

    static void onUserADIDReceiveSuccess() {
        BISDetector.performInitialSetup();
    }

    static void onUserADIDReceiveFail() {
        BISDetector.performInitialSetup();
    }

    @Override
    public void onBeaconsInfoReceiveSuccess() {
        Log.d(TAG, "BLE ranging has begun");
        startRanging();
    }

    @Override
    public void onBeaconsInfoReceiveFail( int errorCode, String errorMessage ) {
        if ( errorMessage == null ) {
            switch( errorCode )
            {
                case 401:
                    errorMessage = "Invalid Authentication";
                    break;
                default:
                    errorMessage = "An error occurred";
                    break;
            }
        }
        errorMessage = errorCode+" : "+errorMessage;
        Log.e(TAG, errorMessage);
        if ( delegate != null ) { delegate.onBISError( BISDetectorInternalDelegate.ERROR_CODE_SERVICE_UNAVAILABLE, errorMessage ); }
    }

    @Override
    public void checkIfShouldUpdateBeaconsInfo() {
        if(isInternetAvailable() && beaconsManager.uuids.size() == 0) {
            BISDetectorREST.getBeaconsInfoFromServer();
        }
    }

    @Override
    public void onBeaconEnter(String beaconId) {
        if(delegate != null)
        {
            delegate.didEnterBISRegion(beaconId);
        }
    }

    @Override
    public void onBeaconExit(String beaconId) {
        if(delegate != null)
        {
            delegate.didExitBISRegion(beaconId);
        }
    }
}
