package com.beaconsinspace.android.beacon.detector;

import android.content.Context;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Looper;
import android.util.Log;

import java.util.HashMap;
import java.util.Map;

/**
 * Created by johnlfoleyiii on 6/2/16.
 */
class BISLocationListener extends AsyncTask<String, String, Location> implements LocationListener {

    private static final String TAG = "BIS_LOCATION_LISTENER";

    private static final int TWO_MINUTES = 1000 * 60 * 2;
    private static final int LOCATION_HALF_LIFE = 1000 * 30; // seconds until location is deemed stale/invalid
    private static HashMap<String, Location> beaconLocations = new HashMap<String, Location>();
    static Location currentBestLocation;
    static boolean isCurrentlyMonitoring = false;

    private Context context;
    private Location foundLocation;
    private LocationManager locationManager;


    BISLocationListener(Context ctx)
    {
        context = ctx;
    }

    public static void assignLocationToBeaconId( String beaconId )
    {
        if ( locationIsStillValid( currentBestLocation ) )
        {
            setBeaconIdLocation( beaconId, currentBestLocation );
        }
        else
        {
            setBeaconIdLocation( beaconId, null );
            if ( ! isCurrentlyMonitoring )
            {
                BISLocationListener locationListener = new BISLocationListener(BISDetector.context);
                locationListener.execute(beaconId);
            }
        }
    }
    public static Location getLocationByBeaconId(String beaconId)
    {
        Location loc = beaconLocations.get(beaconId);
        beaconLocations.remove( beaconId );
        return loc;
    }
    private static void setBeaconIdLocation(String beaconId, Location location)
    {
        beaconLocations.put( beaconId, location );
    }

    @Override
    protected Location doInBackground(String... beaconIdArr)
    {
        beginLocationMonitoring();
        while(foundLocation==null){}
        return foundLocation;
    }
    @Override
    protected void onPostExecute(Location location)
    {
        for ( Map.Entry<String, Location> entry : beaconLocations.entrySet() )
        {
            // Set valid location on all of the beaconLocation entries with null locations
            String entryBeaconId = entry.getKey();
            Location entryLocation = entry.getValue();
            if ( entryLocation == null )
            {
                setBeaconIdLocation( entryBeaconId, location );
            }
        }
        super.onPostExecute(location);
    }

    static boolean locationIsStillValid(Location loc)
    {
        if ( loc == null ) { return false; }
        long locTime = loc.getTime();
        long currentTime = System.currentTimeMillis();
        return currentTime - locTime < LOCATION_HALF_LIFE;
    }

    void beginLocationMonitoring()
    {
        try
        {
            locationManager = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);
            locationManager.requestLocationUpdates(LocationManager.GPS_PROVIDER, 5000, 10, this, Looper.getMainLooper());
            locationManager.requestLocationUpdates(LocationManager.NETWORK_PROVIDER, 5000, 10, this, Looper.getMainLooper());
            isCurrentlyMonitoring = true;
        }
        catch (SecurityException e)
        {
            Log.e(TAG, "LOCATION EXCEPTION"+ e.getMessage());
            endLocationMonitoring();
        }
    }

    void endLocationMonitoring()
    {
        try
        {
            locationManager.removeUpdates(this);
            isCurrentlyMonitoring=false;
        }
        catch( SecurityException e )
        {
            Log.e(TAG,"FAILED TO END LOCATION MONITORING"+e.getMessage());
        }
    }

    @Override
    public void onLocationChanged(Location loc)
    {
        logLocationInfo(loc);
        makeUseOfNewLocation(loc);
    }

    /**
     * This method modify the last know good location according to the arguments.
     *
     * @param location The possible new location.
     */
    void makeUseOfNewLocation(Location location)
    {
        if ( isBetterLocation( location, currentBestLocation ) )
        {
            endLocationMonitoring();
            currentBestLocation = location;
            foundLocation = location; // must be last, setting this value finalizes and closes thread
        }
    }

    /** Determines whether one location reading is better than the current location fix
     * @param location  The new location that you want to evaluate
     * @param currentBestLocation  The current location fix, to which you want to compare the new one.
     */
    protected boolean isBetterLocation(Location location, Location currentBestLocation) {
        if (currentBestLocation == null) {
            // A new location is always better than no location
            return true;
        }

        // Check whether the new location fix is newer or older
        long timeDelta = location.getTime() - currentBestLocation.getTime();
        boolean isSignificantlyNewer = timeDelta > TWO_MINUTES;
        boolean isSignificantlyOlder = timeDelta < -TWO_MINUTES;
        boolean isNewer = timeDelta > 0;

        // If it's been more than two minutes since the current location, use the new location,
        // because the user has likely moved.
        if (isSignificantlyNewer) {
            return true;
            // If the new location is more than two minutes older, it must be worse.
        } else if (isSignificantlyOlder) {
            return false;
        }

        // Check whether the new location fix is more or less accurate
        int accuracyDelta = (int) (location.getAccuracy() - currentBestLocation.getAccuracy());
        boolean isLessAccurate = accuracyDelta > 0;
        boolean isMoreAccurate = accuracyDelta < 0;
        boolean isSignificantlyLessAccurate = accuracyDelta > 200;

        // Check if the old and new location are from the same provider
        boolean isFromSameProvider = isSameProvider(location.getProvider(),
                currentBestLocation.getProvider());

        // Determine location quality using a combination of timeliness and accuracy
        if (isMoreAccurate) {
            return true;
        } else if (isNewer && !isLessAccurate) {
            return true;
        } else if (isNewer && !isSignificantlyLessAccurate && isFromSameProvider) {
            return true;
        }
        return false;
    }

    /** Checks whether two providers are the same */
    private boolean isSameProvider(String provider1, String provider2)
    {
        if ( provider1 == null )
        {
            return provider2 == null;
        }
        return provider1.equals( provider2 );
    }

    @Override
    public void onStatusChanged(String var1, int var2, Bundle var3) { Log.i( TAG, "STATUS CHANGED" ); }

    @Override
    public void onProviderEnabled(String var1)
    {
        beginLocationMonitoring();
    }

    @Override
    public void onProviderDisabled(String var1)
    {
        endLocationMonitoring();
    }

    public static void logLocationInfo(Location loc)
    {
        if ( loc == null ) { Log.i(TAG,"LOCATION CANNOT BE LOGGED IT IS NULL"); return; }
//        Log.i(TAG,"Location: "+loc.toString());
    }


}
