package com.bandyer.communication_center.live_pointer

import com.bandyer.communication_center.CommunicationCenter
import com.bandyer.communication_center.call.Call
import com.bandyer.communication_center.call.CallException
import com.bandyer.communication_center.call.OnCallEventObserver
import com.bandyer.communication_center.live_pointer.utils.isActive
import com.bandyer.core_av.room.RoomToken
import com.google.gson.Gson

/**
 * LivePointer provides functionality to listen to live pointer events. A live pointer event represent a pointer that should be represented upon a video stream.
 */
object LivePointer {

    internal val gson: Gson by lazy { CommunicationCenter.instance!!.gson }

    private val activeCustomEventObservers = hashMapOf<Call, CallAwareLivePointerInteractor>()

    /**
     * Default sampling period of the events
     */
    const val SAMPLING_PERIOD = 300L

    /**
     * Default idle timeout
     */
    const val IDLE_TIMEOUT = 5000L

    private val callEventObserver = object : OnCallEventObserver {
        override fun onCallStarted(call: Call, roomToken: RoomToken) = Unit
        override fun onCallEnded(call: Call, callEndReason: Call.EndReason) = stopListeningToPointerEvents(call)
        override fun onCallError(call: Call, reason: CallException) = stopListeningToPointerEvents(call)
        override fun onCallUpgraded() = Unit
        override fun onCallStatusChanged(call: Call, status: Call.Status) = Unit
    }

    /**
     * Start listening to live pointer events for the provided call.
     * @param call Call input call for the live pointer event listening.
     * @param pointerEventListener LivePointerEventsObserver live pointer events observer.
     * @param samplingPeriod milliseconds for the sampling period of the event stream.
     * @param idleTimeout milliseconds to trigger the idle state.
     */
    fun startListeningToPointerEvents(call: Call, pointerEventListener: PointerEventListener, samplingPeriod: Long? = SAMPLING_PERIOD, idleTimeout: Long? = IDLE_TIMEOUT) {
        if (!call.isActive()) return
        activeCustomEventObservers[call] = CallAwareLivePointerInteractor(call, pointerEventListener, samplingPeriod = samplingPeriod, idleTimeout = idleTimeout)
        call.addEventObserver(callEventObserver)
    }

    /**
     * Stop listening to live pointer events for the provided call.
     * @param call The call that should not be used anymore for live pointer events listening.
     */
    fun stopListeningToPointerEvents(call: Call) = removeObservers(call)

    private fun removeObservers(call: Call) {
        activeCustomEventObservers.remove(call)?.let { customEventObserver ->
            customEventObserver.dispose()
        }
    }
}