/*
 * Copyright (C) 2016-2019 Authlete, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific
 * language governing permissions and limitations under the
 * License.
 */
package com.authlete.common.types;


/**
 * An interface that represents a resource owner (in the context of
 * <a href="https://tools.ietf.org/html/rfc6749">OAuth 2.0</a>) or
 * an end user (in the context of <a href="http://openid.net/connect/"
 * >OpenID Connect</a>).
 *
 * @since 1.26
 *
 * @author Takahiko Kawasaki
 */
public interface User
{
    /**
     * Get the subject (= unique identifier) of the user.
     *
     * @return
     *         The subject (= unique identifier) of the user.
     */
    String getSubject();


    /**
     * Get the value of a claim of the user.
     *
     * @param claimName
     *         A claim name such as {@code name} and {@code family_name}.
     *         Standard claim names are listed in "<a href=
     *         "http://openid.net/specs/openid-connect-core-1_0.html#StandardClaims"
     *         >5.1. Standard Claims</a>" of <a href=
     *         "http://openid.net/specs/openid-connect-core-1_0.html">OpenID
     *         Connect Core 1.0</a>. Java constant values that represent the
     *         standard claims are listed in {@link com.authlete.common.types.StandardClaims
     *         StandardClaims} class.
     *
     * @param languageTag
     *         A language tag such as {@code en} and {@code ja}. Implementations
     *         should take this into account whenever possible. See "<a href=
     *         "http://openid.net/specs/openid-connect-core-1_0.html#ClaimsLanguagesAndScripts"
     *         >5.2. Claims Languages and Scripts</a>" in <a href=
     *         "http://openid.net/specs/openid-connect-core-1_0.html">OpenID
     *         Connect Core 1.0</a> for details.
     *
     * @return
     *         The claim value. {@code null} if the claim value of the claim
     *         is not available.
     */
    Object getClaim(String claimName, String languageTag);


    /**
     * Get the value of an attribute of the user.
     *
     * @param attributeName
     *         An attribute name.
     *
     * @return
     *         The attribute value. {@code null} if the attribute value of the
     *         attribute is not available.
     *
     * @since 2.33
     */
    Object getAttribute(String attributeName);
}
