/*
 * Copyright (C) 2016-2020 Authlete, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.authlete.common.dto;


import java.io.Serializable;


/**
 * Request to Authlete's {@code /auth/token/update} API.
 *
 * <p>
 * The API is used to update an existing access token.
 * </p>
 *
 *
 * <blockquote>
 * <dl>
 * <dt><b><code>accessToken</code></b></dt>
 * <dd>
 * <p>
 * An existing access token.
 * </p>
 * </dd>
 *
 * <dt><b><code>accessTokenExpiresAt</code></b></dt>
 * <dd>
 * <p>
 * A new date at which the access token will expire in milliseconds
 * since the Unix epoch (1970-01-01).  If the {@code accessTokenExpiresAt}
 * request parameter is not included in a request or its value is 0
 * (or negative), the expiration date of the access token is not changed.
 * </p>
 * </dd>
 *
 * <dt><b><code>scopes</code></b></dt>
 * <dd>
 * <p>
 * A new set of scopes assigned to the access token. Scopes that are
 * not supported by the service and those that the client application
 * associated with the access token is not allowed to request are
 * ignored on the server side. If the {@code scopes} request parameter
 * is not included in a request or its value is {@code null}, the scopes
 * of the access token are not changed.
 * </p>
 * </dd>
 *
 * <dt><b><code>properties</code></b></dt>
 * <dd>
 * <p>
 * A new set of properties assigned to the access token. If the
 * {@code properties} request parameter is not included in a request
 * or its value is {@code null}, the properties of the access token
 * are not changed.
 * </p>
 * </dd>
 *
 * <dt><b><code>accessTokenExpiresAtUpdatedOnScopeUpdate</code></b></dt>
 * <dd>
 * <p>
 * A boolean request parameter which indicates whether the API attempts to update
 * the expiration date of the access token when the scopes linked to the access
 * token are changed by this request. The default value is {@code false}. For more
 * details, see the description of
 * {@link #setAccessTokenExpiresAtUpdatedOnScopeUpdate(boolean)}.
 * </p>
 * </dd>
 *
 * <dt><b><code>accessTokenHash</code></b></dt>
 * <dd>
 * <p>
 * The hash of the access token value. Used when the hash of the token
 * is known (perhaps from lookup) but the value of the token itself is not.
 * </p>
 * <p>
 * The value of the {@code accessToken} parameter takes precedence.
 * </p>
 * </dd>
 *
 * <dt><b><code>accessTokenValueUpdated</code></b></dt>
 * <dd>
 * <p>
 * A boolean request parameter which indicates whether to update the value of
 * the access token in the data store. If this parameter is set to {@code true}
 * then a new access token value is generated by the server and returned in the
 * response. The default value is {@code false} and the access token's previous
 * value is retained.
 * </p>
 * </dd>
 *
 * <dt><b><code>certificateThumbprint</code></b> (OPTIONAL)</dt>
 * <dd>
 * <p>
 * The thumbprint of the MTLS certificate bound to this token. If this field
 * is set, a certificate with the corresponding value MUST be presented with the
 * access token when it is used by a client.
 * </p>
 * </dd>
 *
 * <dt><b><code>dpopKeyThumbprint</code></b> (OPTIONAL)</dt>
 * <dd>
 * <p>
 * The thumbprint of the public key used for DPoP presentation of this token.
 * If this field is set, a DPoP proof signed with the corresponding private key
 * MUST be presented with the access token when it is used by a client. Additionally,
 * the token's {@code token_type} will be set to 'DPoP'.
 * </p>
 * </dd>
 *
 * </dl>
 * </blockquote>
 *
 * @see TokenUpdateResponse
 *
 * @since 1.34
 */
public class TokenUpdateRequest implements Serializable
{
    private static final long serialVersionUID = 5L;


    private String accessToken;
    private long accessTokenExpiresAt;
    private String[] scopes;
    private Property[] properties;
    private boolean accessTokenExpiresAtUpdatedOnScopeUpdate;
    private boolean accessTokenPersistent;
    private String accessTokenHash;
    private boolean accessTokenValueUpdated;
    private String certificateThumbprint;
    private String dpopKeyThumbprint;


    /**
     * Get the access token to update.
     *
     * @return
     *         The access token to update.
     */
    public String getAccessToken()
    {
        return accessToken;
    }


    /**
     * Set an existing access token to update.
     *
     * @param accessToken
     *         An existing access token to update.
     *
     * @return
     *         {@code this} object.
     */
    public TokenUpdateRequest setAccessToken(String accessToken)
    {
        this.accessToken = accessToken;

        return this;
    }


    /**
     * Get the new date at which the access token will expire.
     *
     * @return
     *         The new expiration date in milliseconds since the Unix epoch (1970-01-01).
     */
    public long getAccessTokenExpiresAt()
    {
        return accessTokenExpiresAt;
    }


    /**
     * Set the new date at which the access token will expire.
     *
     * <p>
     * If 0 or a negative value is given, the expiration date of the access token
     * is not changed.
     * </p>
     *
     * @param expiresAt
     *         The new expiration date in milliseconds since the Unix epoch (1970-01-01).
     *
     * @return
     *         {@code this} object.
     */
    public TokenUpdateRequest setAccessTokenExpiresAt(long expiresAt)
    {
        this.accessTokenExpiresAt = expiresAt;

        return this;
    }


    /**
     * Get the new set of scopes assigned to the access token.
     *
     * @return
     *         The new set of scopes.
     */
    public String[] getScopes()
    {
        return scopes;
    }


    /**
     * Set a new set of scopes assigned to the access token.
     *
     * <p>
     * If {@code null} is given, the scope set associated with the access token
     * is not changed.
     * </p>
     *
     * @param scopes
     *         A new set of scopes. {@code null} means that scopes are not changed.
     *
     * @return
     *         {@code this} object.
     */
    public TokenUpdateRequest setScopes(String[] scopes)
    {
        this.scopes = scopes;

        return this;
    }


    /**
     * Get a new set of properties assigned to the access token.
     *
     * @return
     *         A new set of properties.
     */
    public Property[] getProperties()
    {
        return properties;
    }


    /**
     * Set a new set of properties assigned to the access token.
     *
     * <p>
     * If {@code null} is given, the property set associated with the access token
     * is not changed.
     * </p>
     *
     * @param properties
     *         A new set of properties. {@code null} means that properties are not changed.
     *
     * @return
     *         {@code this} object.
     */
    public TokenUpdateRequest setProperties(Property[] properties)
    {
        this.properties = properties;

        return this;
    }


    /**
     * Get the flag which indicates whether {@code /auth/token/update} API attempts
     * to update the expiration date of the access token when the scopes linked to
     * the access token are changed by this request.
     *
     * @return
     *         The flag which indicates whether {@code /auth/token/update} API
     *         attempts to update the expiration date of the access token when
     *         the scopes linked to the access token are changed by this request.
     *
     * @since 2.29
     */
    public boolean isAccessTokenExpiresAtUpdatedOnScopeUpdate()
    {
        return accessTokenExpiresAtUpdatedOnScopeUpdate;
    }


    /**
     * Set the flag which indicates whether {@code /auth/token/update} API attempts
     * to update the expiration date of the access token when the scopes linked to
     * the access token are changed by this request. This request parameter is optional
     * and its default value is {@code false}. If this request parameter is set
     * to {@code true} and all of the following conditions are satisfied, the API
     * performs an update on the expiration date of the access token even if the
     * <code>accessTokenExpiresAt</code> request parameter is not explicitly specified
     * in the request.
     *
     * <ol>
     * <li>The <code>accessTokenExpiresAt</code> request parameter is not included
     * in the request or its value is <code>0</code> (or negative).
     * <li>The scopes linked to the access token are changed by the <code>scopes</code>
     *     request parameter in the request.
     * <li>Any of the new scopes to be linked to the access token has one or more
     *     attributes specifying access token duration.
     * </ol>
     *
     * <p>
     * When multiple access token duration values are found in the attributes of
     * the specified scopes, the smallest value among them is used.
     * </p>
     *
     * <p>
     * For more details, see the following examples.
     * </p>
     *
     * <p>
     * <b>Example 1.</b>
     *
     * <p>
     * Let's say we send the following request to {@code /auth/token/update} API
     * </p>
     *
     * <pre style="border: 1px solid black; padding: 0.5em; margin: 0.5em;">
     * {
     *   "accessToken" : "JDGiiM9PuWT63FIwGjG9eYlGi-aZMq6CQ2IB475JUxs",
     *   "scopes" : ["read_profile"]
     * }</pre>
     *
     * <p>
     * and <code>"read_profile"</code> has the following attributes.
     * </p>
     *
     * <pre style="border: 1px solid black; padding: 0.5em; margin: 0.5em;">
     * {
     *   "key" : "access_token.duration",
     *   "value" : "10000"
     * }</pre>
     *
     * <p>
     * In this case, the API evaluates <code>"10000"</code> as a new value of the
     * duration of the access token (in seconds) and updates the expiration date
     * of the access token using the duration.
     * </p>
     *
     * <b>Example 2.</b>
     *
     * <p>
     * Let's say we send the following request to {@code /auth/token/update} API
     * </p>
     *
     * <pre style="border: 1px solid black; padding: 0.5em; margin: 0.5em;">
     * {
     *   "accessToken" : "JDGiiM9PuWT63FIwGjG9eYlGi-aZMq6CQ2IB475JUxs",
     *   "scopes" : ["read_profile", "write_profile"]
     * }</pre>
     *
     * <p>
     * and <code>"read_profile"</code> has the following attributes
     * </p>
     *
     * <pre style="border: 1px solid black; padding: 0.5em; margin: 0.5em;">
     * {
     *   "key" : "access_token.duration",
     *   "value" : "10000"
     * }</pre>
     *
     * <p>
     * and <code>"write_profile"</code> has the following attributes.
     * </p>
     *
     * <pre style="border: 1px solid black; padding: 0.5em; margin: 0.5em;">
     * {
     *   "key" : "access_token.duration",
     *   "value" : "5000"
     * }</pre>
     *
     * <p>
     * In this case, the API evaluates <code>"10000"</code> and <code>"5000"</code>
     * as candidate values for new duration of the access token (in seconds) and
     * chooses the smallest value of them (i.e. "5000" is adopted) and updates the
     * expiration date of the access token using the duration.
     * </p>
     *
     * @param updated
     *         The flag which indicates whether {@code /auth/token/update} API
     *         attempts to update the expiration date of the access token when
     *         the scopes linked to the access token are changed by this request.
     *
     * @return
     *         {@code this} object.
     *
     * @since 2.29
     */
    public TokenUpdateRequest setAccessTokenExpiresAtUpdatedOnScopeUpdate(boolean updated)
    {
        this.accessTokenExpiresAtUpdatedOnScopeUpdate = updated;

        return this;
    }


    /**
     * Get whether the access token expires or not. By default, all access tokens
     * expire after a period of time determined by their service. If this request
     * parameter is {@code true} then the access token will not automatically
     * expire and must be revoked or deleted manually at the service.
     *
     * <p>
     * If this request parameter is {@code true}, the {@code accessTokenExpiresAt}
     * request parameter is ignored. If this request parameter is {@code false},
     * the {@code accessTokenExpiresAt} request parameter is processed normally.
     * </p>
     *
     * @return
     *         {@code false} if the access token expires (default).
     *         {@code true} if the access token does not expire.
     *
     * @since 2.30
     */
    public boolean isAccessTokenPersistent()
    {
        return accessTokenPersistent;
    }


    /**
     * Set whether the access token expires or not. By default, all access tokens
     * expire after a period of time determined by their service. If this request
     * parameter is {@code true} then the access token will not automatically
     * expire and must be revoked or deleted manually at the service.
     *
     * <p>
     * If this request parameter is {@code true}, the {@code accessTokenExpiresAt}
     * request parameter is ignored. If this request parameter is {@code false},
     * the {@code accessTokenExpiresAt} request parameter is processed normally.
     * </p>
     *
     * @param persistent
     *         {@code false} to make the access token expire (default).
     *         {@code true} to make the access token be persistent.
     *
     * @return
     *         {@code this} object.
     *
     * @since 2.30
     */
    public TokenUpdateRequest setAccessTokenPersistent(boolean persistent)
    {
        this.accessTokenPersistent = persistent;

        return this;
    }


    /**
     * Get the hash of the access token value. Used when the hash of the token
     * is known (perhaps from lookup) but the value of the token itself is not.
     *
     * <p>
     * The value of the {@code accessToken} parameter takes precedence.
     * </p>
     *
     * @return
     *         The hash of the access token value.
     *
     * @since 2.31
     */
    public String getAccessTokenHash()
    {
        return accessTokenHash;
    }


    /**
     * Set the hash of the access token value. Used when the hash of the token
     * is known (perhaps from lookup) but the value of the token itself is not.
     *
     * <p>
     * The value of the {@code accessToken} parameter takes precedence.
     * </p>
     *
     * @param accessTokenHash
     *         The hash of the access token value.
     *
     * @return
     *         {@code this} object.
     *
     * @since 2.31
     */
    public TokenUpdateRequest setAccessTokenHash(String accessTokenHash)
    {
        this.accessTokenHash = accessTokenHash;

        return this;
    }


    /**
     * Get whether to update the value of the access token in the data store. If
     * this parameter is set to {@code true} then a new access token value is
     * generated by the server and returned in the response.
     *
     * @return
     *         {@code false} to keep the access token's current value (default).
     *         {@code true} to have the server update the access token's value.
     *
     * @since 2.31
     */
    public boolean isAccessTokenValueUpdated()
    {
        return accessTokenValueUpdated;
    }


    /**
     * Set whether to update the value of the access token in the data store. If
     * this parameter is set to {@code true} then a new access token value is
     * generated by the server and returned in the response.
     *
     * @param updated
     *         {@code false} to keep the access token's current value (default).
     *         {@code true} to have the server update the access token's value.
     *
     * @return
     *         {@code this} object.
     *
     * @since 2.31
     */
    public TokenUpdateRequest setAccessTokenValueUpdated(boolean updated)
    {
        this.accessTokenValueUpdated = updated;

        return this;
    }


    /**
     * Get the thumbprint of the MTLS certificate bound to this token. If this field
     * is set, a certificate with the corresponding value MUST be presented with the
     * access token when it is used by a client.
     *
     * @return
     *         The SHA256 certificate thumbprint, base64url encoded.
     *
     * @since 2.72
     */
    public String getCertificateThumbprint()
    {
        return certificateThumbprint;
    }


    /**
     * Set the thumbprint of the MTLS certificate bound to this token. If this field
     * is set, a certificate with the corresponding value MUST be presented with the
     * access token when it is used by a client.
     *
     * @param certificateThumbprint
     *            The SHA256 certificate thumbprint, base64url encoded.
     *
     * @return
     *         {@code this} object.
     *
     * @since 2.72
     */
    public TokenUpdateRequest setCertificateThumbprint(String certificateThumbprint)
    {
        this.certificateThumbprint = certificateThumbprint;

        return this;
    }


    /**
     * Get the thumbprint of the public key used for DPoP presentation of this token.
     * If this field is set, a DPoP proof signed with the corresponding private key
     * MUST be presented with the access token when it is used by a client. Additionally,
     * the token's {@code token_type} will be set to 'DPoP'.
     *
     * @return
     *         The JWK public key thumbprint.
     *
     * @since 2.72
     */
    public String getDpopKeyThumbprint()
    {
        return dpopKeyThumbprint;
    }


    /**
     * Set the thumbprint of the public key used for DPoP presentation of this token.
     * If this field is set, a DPoP proof signed with the corresponding private key
     * MUST be presented with the access token when it is used by a client. Additionally,
     * the token's {@code token_type} will be set to 'DPoP'.
     *
     * @param dpopKeyThumbprint
     *            The JWK public key thumbprint.
     *
     * @return
     *         {@code this} object.
     *
     * @since 2.72
     */
    public TokenUpdateRequest setDpopKeyThumbprint(String dpopKeyThumbprint)
    {
        this.dpopKeyThumbprint = dpopKeyThumbprint;

        return this;
    }
}
