package com.atlassian.vcache.internal;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.regex.Pattern;

/**
 * Provides utility methods to validate a name as being legitimate for use with VCache.
 *
 * @since 1.0
 */
public class NameValidator {
    // If this simple implementation provides to be a performance bottleneck, then consider an optimized algorithm
    // looking at the characters.
    private static final Pattern LEGAL_PATTERN = Pattern.compile("^[\\w\\./\\-_\\$]+$");

    /**
     * Returns whether the specified name is valid for a product identifier.
     *
     * @param name the name to check
     * @return whether the specified name is valid for a product identifier.
     */
    public static boolean isValidProductIdentifier(@Nullable String name) {
        return (name != null) && LEGAL_PATTERN.matcher(name).matches();
    }

    /**
     * Verifies that the specified name is valid for a product identifier,
     * and throws an {@link IllegalArgumentException} if it not.
     *
     * @param name the name to verify
     * @return the name if it is valid
     */
    @Nonnull
    public static String requireValidProductIdentifier(@Nullable String name) {
        if (!isValidProductIdentifier(name)) {
            throw new IllegalArgumentException("Invalid product identifier: " + name);
        }

        return name;
    }

    /**
     * Returns whether the specified name is valid for a partition identifier.
     *
     * @param name the name to check
     * @return whether the specified name is valid for a partition identifier.
     */
    public static boolean isValidPartitionIdentifier(@Nullable String name) {
        return (name != null) && LEGAL_PATTERN.matcher(name).matches();
    }

    /**
     * Verifies that the specified name is valid for a partition identifier,
     * and throws an {@link IllegalArgumentException} if it not.
     *
     * @param name the name to verify
     * @return the name if it is valid
     */
    @Nonnull
    public static String requireValidPartitionIdentifier(@Nullable String name) {
        if (!isValidPartitionIdentifier(name)) {
            throw new IllegalArgumentException("Invalid partition identifier: " + name);
        }

        return name;
    }

    /**
     * Returns whether the specified name is valid for a key.
     *
     * @param name the name to check
     * @return whether the specified name is valid for a key.
     */
    public static boolean isValidKeyName(@Nullable String name) {
        return (name != null) && LEGAL_PATTERN.matcher(name).matches();
    }

    /**
     * Verifies that the specified name is valid for a key, and throws an {@link IllegalArgumentException} if it
     * not.
     *
     * @param name the name to verify
     * @return the name if it is valid
     */
    @Nonnull
    public static String requireValidKeyName(@Nullable String name) {
        if (!isValidKeyName(name)) {
            throw new IllegalArgumentException("Invalid key name: " + name);
        }

        return name;
    }

    /**
     * Returns whether the specified name is valid for a cache.
     *
     * @param name the name to check
     * @return whether the specified name is valid for a cache.
     */
    public static boolean isValidCacheName(String name) {
        return LEGAL_PATTERN.matcher(name).matches();
    }

    /**
     * Verifies that the specified name is valid for a cache, and throws an {@link IllegalArgumentException} if it
     * not.
     *
     * @param name the name to verify
     * @return the name if it is valid
     */
    @Nonnull
    public static String requireValidCacheName(String name) {
        if (!isValidCacheName(name)) {
            throw new IllegalArgumentException("Invalid cache name: " + name);
        }

        return name;
    }
}
