package com.atlassian.vcache;

import com.atlassian.annotations.ExperimentalApi;

import javax.annotation.Nonnull;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * Provides a helper utilities.
 *
 * @since 1.0
 */
@ExperimentalApi
public class VCacheUtils {
    /**
     * Syntactic sugar for writing {@code stage.toCompletableFuture().join()}.
     *
     * @param stage the stage to call {@link CompletableFuture#join()} on
     * @param <V>   the value type
     * @return the result of joining.
     * @see CompletableFuture#join()
     */
    @Nonnull
    public static <V> V join(CompletionStage<V> stage) {
        return stage.toCompletableFuture().join();
    }

    /**
     * Syntactic sugar for handling the result (either success or failure) from a {@link CompletionStage}.
     *
     * @param stage   the stage to handle the results
     * @param success called if the stage returned a successful result
     * @param failure called if the stage failed
     * @param <T>     the type returned by the stage
     * @param <R>     the type to be returned
     * @return if the stage is successful, then the result of calling <tt>success</tt>, otherwise the result of
     * calling <tt>failure</tt>
     */
    @Nonnull
    public static <T, R> R fold(CompletionStage<T> stage, Function<T, R> success, Function<Throwable, R> failure) {
        return join(stage.handle((val, err) ->
                (val != null) ? success.apply(val) : failure.apply(err)));
    }

    /**
     * Syntactic sugar for writing {@code stage.handle(fn).toCompletableFuture().join()}.
     *
     * @param stage the stage to handle the results
     * @param fn    called to handle the result
     * @param <T>   the type returned by the stage
     * @param <R>   the type to be returned
     * @return the result of calling <tt>fn</tt> on the stage outcome.
     */
    @Nonnull
    public static <T, R> R fold(CompletionStage<T> stage, BiFunction<T, Throwable, R> fn) {
        return join(stage.handle(fn));
    }
}
