package com.atlassian.vcache;

import com.atlassian.annotations.PublicApi;

import javax.annotation.Nonnull;

/**
 * Represents the factory for creating caches.
 * <p>
 * Notes:
 * </p>
 * <ul>
 * <li>
 * {@link JvmCache}'s and {@link RequestCache}'s are identified by their name and cache type. I.e. A
 * {@link JvmCache} called <tt>'Williams'</tt> and a {@link RequestCache} called <tt>'Williams'</tt> are
 * considered to be separate caches.
 * </li>
 * <li>
 * {@link ExternalCache}'s are identified by their name and cache type.
 * </li>
 * <li>
 * A cache name may only contain the characters {@code A-Z}, {@code a-z}, {@code 0-9}, {@code ./-_$}.
 * </li>
 * <li>
 * The returned cache instances are multi-thread safe.
 * </li>
 * </ul>
 *
 * @since 1.0
 */
@PublicApi
public interface VCacheFactory {
    /**
     * Obtains a {@link JvmCache} with the specified details.
     *
     * @param name     the name of the cache
     * @param settings the settings for the cache
     * @param <K>      the key type
     * @param <V>      the value type
     * @return {@link JvmCache} with the specified details
     */
    @Nonnull
    <K, V> JvmCache<K, V> getJvmCache(String name, JvmCacheSettings settings);

    /**
     * Obtains a {@link RequestCache} with the specified details.
     *
     * @param name the name of the cache
     * @param <K>  the key type
     * @param <V>  the value type
     * @return {@link RequestCache} with the specified details
     */
    @Nonnull
    <K, V> RequestCache<K, V> getRequestCache(String name);

    /**
     * Obtains a {@link TransactionalExternalCache} with the specified details.
     *
     * @param name            the name of the cache
     * @param valueMarshaller the marshaller for the values. See {@link com.atlassian.vcache.marshallers.MarshallerFactory} for provided implementations.
     * @param settings        the settings for the cache
     * @param <V>             the value type
     * @return {@link TransactionalExternalCache} with the specified details
     */
    @Nonnull
    <V> TransactionalExternalCache<V> getTransactionalExternalCache(
            String name,
            Marshaller<V> valueMarshaller,
            ExternalCacheSettings settings);

    /**
     * Obtains a {@link StableReadExternalCache} with the specified details.
     *
     * @param name            the name of the cache
     * @param valueMarshaller the marshaller for the values.
     *                        See {@link com.atlassian.vcache.marshallers.MarshallerFactory} for provided implementations.
     * @param settings        the settings for the cache
     * @param <V>             the value type
     * @return {@link StableReadExternalCache} with the specified details
     */
    @Nonnull
    <V> StableReadExternalCache<V> getStableReadExternalCache(
            String name,
            Marshaller<V> valueMarshaller,
            ExternalCacheSettings settings);

    /**
     * Obtains a {@link DirectExternalCache} with the specified details.
     *
     * @param name            the name of the cache
     * @param valueMarshaller the marshaller for the values.
     *                        See {@link com.atlassian.vcache.marshallers.MarshallerFactory} for provided implementations.
     * @param settings        the settings for the cache
     * @param <V>             the value type
     * @return {@link DirectExternalCache} with the specified details
     */
    @Nonnull
    <V> DirectExternalCache<V> getDirectExternalCache(
            String name,
            Marshaller<V> valueMarshaller,
            ExternalCacheSettings settings);
}
