package com.atlassian.vcache;

import com.atlassian.annotations.PublicApi;

import javax.annotation.Nonnull;
import java.time.Duration;
import java.util.Optional;

import static com.atlassian.vcache.SettingsUtils.ifPresent;
import static java.util.Objects.requireNonNull;

/**
 * An immutable representation of settings for an {@link JvmCache}.
 *
 * @since 1.0
 */
@PublicApi
public class JvmCacheSettings {
    private final Optional<Integer> maxEntries;
    private final Optional<Duration> defaultTtl;

    JvmCacheSettings(Optional<Integer> maxEntries, Optional<Duration> defaultTtl) {
        this.maxEntries = requireNonNull(maxEntries);
        this.defaultTtl = requireNonNull(defaultTtl);
    }

    /**
     * Returns a new {@link JvmCacheSettings} instance where the current settings
     * are overridden with settings specified in <tt>overrides</tt>. Only properties
     * in <tt>overrides</tt> that are present will be applied.
     *
     * @param overrides contains the settings to override
     * @return a new {@link JvmCacheSettings} instance with the <tt>overrides</tt> settings applied.
     */
    @Nonnull
    public JvmCacheSettings override(JvmCacheSettings overrides) {
        return new JvmCacheSettings(
                ifPresent(overrides.getMaxEntries(), getMaxEntries()),
                ifPresent(overrides.getDefaultTtl(), getDefaultTtl()));
    }

    /**
     * Returns the maximum number of entries allowed.
     *
     * @return the maximum number of entries allowed.
     */
    @Nonnull
    public Optional<Integer> getMaxEntries() {
        return maxEntries;
    }

    /**
     * Returns the default time-to-live for entries when added.
     *
     * @return the default time-to-live for entries when added.
     */
    @Nonnull
    public Optional<Duration> getDefaultTtl() {
        return defaultTtl;
    }
}
