package com.atlassian.vcache;

import com.atlassian.annotations.PublicApi;

import javax.annotation.Nonnull;
import java.time.Duration;
import java.util.Optional;

import static com.atlassian.vcache.SettingsUtils.ifPresent;
import static java.util.Objects.requireNonNull;

/**
 * An immutable representation of settings for an {@link ExternalCache}.
 *
 * @since 1.0
 */
@PublicApi
public class ExternalCacheSettings {
    private final Optional<Duration> defaultTtl;
    private final Optional<Integer> entryCountHint;
    private final Optional<ChangeRate> dataChangeRateHint;
    private final Optional<ChangeRate> entryGrowthRateHint;

    ExternalCacheSettings(
            Optional<Duration> defaultTtl,
            Optional<Integer> entryCountHint,
            Optional<ChangeRate> dataChangeRateHint,
            Optional<ChangeRate> entryGrowthRateHint) {
        this.defaultTtl = requireNonNull(defaultTtl);
        this.entryCountHint = requireNonNull(entryCountHint);
        this.dataChangeRateHint = requireNonNull(dataChangeRateHint);
        this.entryGrowthRateHint = requireNonNull(entryGrowthRateHint);
    }

    /**
     * Returns a new {@link ExternalCacheSettings} instance where the current settings
     * are overridden with settings specified in <tt>overrides</tt>.
     *
     * @param overrides contains the settings to override
     * @return a new {@link ExternalCacheSettings} instance with the <tt>overrides</tt> settings applied.
     */
    @Nonnull
    public ExternalCacheSettings override(ExternalCacheSettings overrides) {
        return new ExternalCacheSettings(
                ifPresent(overrides.getDefaultTtl(), getDefaultTtl()),
                ifPresent(overrides.getEntryCountHint(), getEntryCountHint()),
                ifPresent(overrides.getDataChangeRateHint(), getDataChangeRateHint()),
                ifPresent(overrides.getEntryGrowthRateHint(), getEntryGrowthRateHint()));
    }

    /**
     * Returns the default time-to-live for entries when added.
     *
     * @return the default time-to-live for entries when added.
     */
    @Nonnull
    public Optional<Duration> getDefaultTtl() {
        return defaultTtl;
    }

    /**
     * Returns the hint for the expected number of entries
     *
     * @return the hint for the expected number of entries
     */
    @Nonnull
    public Optional<Integer> getEntryCountHint() {
        return entryCountHint;
    }

    /**
     * Returns the hint for the expected change rate for data updates.
     *
     * @return the hint for the expected change rate for data updates.
     */
    @Nonnull
    public Optional<ChangeRate> getDataChangeRateHint() {
        return dataChangeRateHint;
    }

    /**
     * Returns the hint for the expected change rate for entry additions.
     *
     * @return the hint for the expected change rate for entry additions.
     */
    @Nonnull
    public Optional<ChangeRate> getEntryGrowthRateHint() {
        return entryGrowthRateHint;
    }
}
