package com.atlassian.vcache;

import com.atlassian.annotations.PublicApi;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * Represents a failure occurred with an {@link ExternalCache}.
 *
 * @since 1.0
 */
@PublicApi
public class ExternalCacheException extends VCacheException {
    /**
     * The reasons for failure. Note that even though a failure may be reported, the operation may have still been
     * performed on the external cache.
     */
    public enum Reason {
        /**
         * Operation timed out.
         */
        TIMEOUT,
        /**
         * Operation failed due to a network error.
         */
        NETWORK_FAILURE,
        /**
         * Operation failed due a {@link Marshaller} failure.
         */
        MARSHALLER_FAILURE,
        /**
         * Operation failed due to the supplied Function not returning the expected number of values.
         */
        FUNCTION_INCORRECT_RESULT,
        /**
         * Creation of cache failed.
         */
        CREATION_FAILURE,
        /**
         * Operation failed due to a transaction sync failure.
         */
        TRANSACTION_FAILURE,
        /**
         * Operation failed due to an unclassified failure.
         */
        UNCLASSIFIED_FAILURE
    }

    private final Reason reason;

    /**
     * Creates an instance for the specified reason.
     *
     * @param reason the reason for the failure
     */
    public ExternalCacheException(Reason reason) {
        super("Failed due to " + reason.name());
        this.reason = requireNonNull(reason);
    }

    /**
     * Creates an instance for the specified reason and cause.
     *
     * @param reason the reason for the failure
     * @param cause  the cause of the failure
     */
    public ExternalCacheException(Reason reason, Throwable cause) {
        super("Failed due to " + reason.name(), cause);
        this.reason = requireNonNull(reason);
    }

    /**
     * Returns the reason for failure.
     *
     * @return the reason for failure.
     */
    @Nonnull
    public Reason getReason() {
        return reason;
    }
}
