package com.atlassian.vcache;

import com.atlassian.annotations.PublicApi;
import com.atlassian.marshalling.api.MarshallingPair;

/**
 * Represents the factory for creating caches.
 * <p>
 * Notes:
 * </p>
 * <ul>
 * <li>
 * {@link JvmCache}'s and {@link RequestCache}'s are identified by their name and cache type. I.e. A
 * {@link JvmCache} called <tt>'Williams'</tt> and a {@link RequestCache} called <tt>'Williams'</tt> are
 * considered to be separate caches.
 * </li>
 * <li>
 * {@link ExternalCache}'s are identified by their name and cache type.
 * </li>
 * <li>
 * A cache name may only contain the characters {@code A-Z}, {@code a-z}, {@code 0-9}, {@code ./-_$}.
 * </li>
 * <li>
 * The returned cache instances are multi-thread safe.
 * </li>
 * </ul>
 *
 * @since 1.0
 */
@PublicApi
public interface VCacheFactory {
    /**
     * Obtains a {@link JvmCache} with the specified details.
     * <p>
     * Note: multiple calls to this method will always return the instance created on the first invocation. As such,
     * any changes in the supplied settings will be ignored.
     * </p>
     *
     * @param name     the name of the cache
     * @param settings the settings for the cache
     * @param <K>      the key type
     * @param <V>      the value type
     * @return {@link JvmCache} with the specified details
     */
    <K, V> JvmCache<K, V> getJvmCache(String name, JvmCacheSettings settings);

    /**
     * Obtains a {@link RequestCache} with the specified details.
     * <p>
     * Note: multiple calls to this method will always return the instance created on the first invocation. Simply calls
     * {@link VCacheFactory#getRequestCache(String, RequestCacheSettings)} using the default settings.
     *
     * </p>
     *
     * @param name the name of the cache
     * @param <K>  the key type
     * @param <V>  the value type
     * @return {@link RequestCache} with the specified details
     * @deprecated Please use {@link VCacheFactory#getRequestCache(String, RequestCacheSettings)} instead.
     */
    @Deprecated
    default <K, V> RequestCache<K, V> getRequestCache(String name) {
        return getRequestCache(name, new RequestCacheSettingsBuilder().build());
    }

    /**
     * Obtains a {@link RequestCache} with the specified settings.
     *
     * @param name     the name of the cache.
     * @param settings the settings for this cache.
     * @param <K>      the key type.
     * @param <V>      the value type.
     * @return {@link RequestCache} with the specified configuration.
     */
    <K, V> RequestCache<K, V> getRequestCache(String name, RequestCacheSettings settings);

    /**
     * Obtains a {@link TransactionalExternalCache} with the specified details.
     * <p>
     * Note: each call to this method will return a new instance using the specified settings. However, the existing
     * data stored in the external cache will remain. It is not expected in a production configuration that this
     * method will be called multiple times, but is allowed is a development configuration.
     * </p>
     *
     * @param name            the name of the cache
     * @param valueMarshaller the marshaller for the values.
     *                        See {@link com.atlassian.vcache.marshallers.MarshallerFactory} for provided implementations.
     * @param settings        the settings for the cache
     * @param <V>             the value type
     * @return {@link TransactionalExternalCache} with the specified details
     * @deprecated since 1.5.0. Use {@link #getTransactionalExternalCache(String, MarshallingPair, ExternalCacheSettings)}
     */
    @Deprecated
    <V> TransactionalExternalCache<V> getTransactionalExternalCache(
            String name,
            Marshaller<V> valueMarshaller,
            ExternalCacheSettings settings);

    /**
     * Obtains a {@link TransactionalExternalCache} with the specified details.
     * <p>
     * Note: each call to this method will return a new instance using the specified settings. However, the existing
     * data stored in the external cache will remain. It is not expected in a production configuration that this
     * method will be called multiple times, but is allowed is a development configuration.
     * </p>
     *
     * @param name             the name of the cache
     * @param valueMarshalling the marshaller for the values.
     * @param settings         the settings for the cache
     * @param <V>              the value type
     * @return {@link TransactionalExternalCache} with the specified details
     */
    <V> TransactionalExternalCache<V> getTransactionalExternalCache(
            String name,
            MarshallingPair<V> valueMarshalling,
            ExternalCacheSettings settings);

    /**
     * Obtains a {@link StableReadExternalCache} with the specified details.
     * <p>
     * Note: each call to this method will return a new instance using the specified settings. However, the existing
     * data stored in the external cache will remain. It is not expected in a production configuration that this
     * method will be called multiple times, but is allowed is a development configuration.
     * </p>
     *
     * @param name            the name of the cache
     * @param valueMarshaller the marshaller for the values.
     *                        See {@link com.atlassian.vcache.marshallers.MarshallerFactory} for provided implementations.
     * @param settings        the settings for the cache
     * @param <V>             the value type
     * @return {@link StableReadExternalCache} with the specified details
     * @deprecated since 1.5.0. Use {@link #getStableReadExternalCache(String, MarshallingPair, ExternalCacheSettings)}
     */
    @Deprecated
    <V> StableReadExternalCache<V> getStableReadExternalCache(
            String name,
            Marshaller<V> valueMarshaller,
            ExternalCacheSettings settings);

    /**
     * Obtains a {@link StableReadExternalCache} with the specified details.
     * <p>
     * Note: each call to this method will return a new instance using the specified settings. However, the existing
     * data stored in the external cache will remain. It is not expected in a production configuration that this
     * method will be called multiple times, but is allowed is a development configuration.
     * </p>
     *
     * @param name             the name of the cache
     * @param valueMarshalling the marshalling pair for the values.
     * @param settings         the settings for the cache
     * @param <V>              the value type
     * @return {@link StableReadExternalCache} with the specified details
     */
    <V> StableReadExternalCache<V> getStableReadExternalCache(
            String name,
            MarshallingPair<V> valueMarshalling,
            ExternalCacheSettings settings);

    /**
     * Obtains a {@link DirectExternalCache} with the specified details.
     * <p>
     * Note: each call to this method will return a new instance using the specified settings. However, the existing
     * data stored in the external cache will remain. It is not expected in a production configuration that this
     * method will be called multiple times, but is allowed is a development configuration.
     * </p>
     *
     * @param name            the name of the cache
     * @param valueMarshaller the marshaller for the values.
     *                        See {@link com.atlassian.vcache.marshallers.MarshallerFactory} for provided implementations.
     * @param settings        the settings for the cache
     * @param <V>             the value type
     * @return {@link DirectExternalCache} with the specified details
     * @deprecated since 1.5.0. Use {@link #getDirectExternalCache(String, MarshallingPair, ExternalCacheSettings)}
     */
    @Deprecated
    <V> DirectExternalCache<V> getDirectExternalCache(
            String name,
            Marshaller<V> valueMarshaller,
            ExternalCacheSettings settings);

    /**
     * Obtains a {@link DirectExternalCache} with the specified details.
     * <p>
     * Note: each call to this method will return a new instance using the specified settings. However, the existing
     * data stored in the external cache will remain. It is not expected in a production configuration that this
     * method will be called multiple times, but is allowed is a development configuration.
     * </p>
     *
     * @param name             the name of the cache
     * @param valueMarshalling the marshalling pair to be used for marshalling.
     * @param settings         the settings for the cache
     * @param <V>              the value type
     * @return {@link DirectExternalCache} with the specified details
     */
    <V> DirectExternalCache<V> getDirectExternalCache(
            String name,
            MarshallingPair<V> valueMarshalling,
            ExternalCacheSettings settings);
}
