package com.atlassian.vcache;

import com.atlassian.annotations.PublicApi;

import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletionStage;
import java.util.function.Supplier;

/**
 * Represents an {@link ExternalCache} where all operations are performed directly on the external cache system.
 * See the {@link ExternalCache} documentation for more information.
 *
 * @param <V> the value type
 * @since 1.0
 */
@PublicApi
public interface DirectExternalCache<V>
        extends ExternalCache<V>, ExternalWriteOperationsUnbuffered<V> {
    /**
     * Returns an identified value that is associated with a specified key. The {@link IdentifiedValue#identifier()}
     * may be used in subsequent CAS operations, such as
     * {@link #removeIf(String, CasIdentifier)} and
     * {@link #replaceIf(String, CasIdentifier, Object)}.
     *
     * @param key the key to check.
     * @return an {@link Optional} which may contain the identified value associated with the key.
     */
    CompletionStage<Optional<IdentifiedValue<V>>> getIdentified(String key);

    /**
     * Returns an identified value that is associated with a specified key. The {@link IdentifiedValue#identifier()}
     * may be used in subsequent CAS operations, such as
     * {@link #removeIf(String, CasIdentifier)} and
     * {@link #replaceIf(String, CasIdentifier, Object)}.
     *
     * @param key the key to check.
     * @param supplier used to generate the value, if one does not exist already for the key. The supplier may not
     *                 return {@code null}.
     * @return an {@link Optional} which may contain the identified value associated with the key.
     */
    CompletionStage<IdentifiedValue<V>> getIdentified(String key, Supplier<V> supplier);

    /**
     * Returns the {@link IdentifiedValue}'s that are associated with the specified keys. It is equivalent to calling
     * {@link #getBulkIdentified(Iterable)} passing {@code Arrays.asList(keys)} as the parameter.
     *
     * @param keys the keys to check.
     * @return A {@link Map} that is keyed on the {@code keys} specified. Each entry in the {@link Map} will have
     * {@link Optional} which may contain the {@link IdentifiedValue} associated with the key.
     */
    @SuppressWarnings("unchecked")
    default CompletionStage<Map<String, Optional<IdentifiedValue<V>>>> getBulkIdentified(String... keys) {
        return getBulkIdentified(Arrays.asList(keys));
    }

    /**
     * Returns the {@link IdentifiedValue}'s that are associated with the specified keys.
     *
     * @param keys the keys to check.
     * @return A {@link Map} that is keyed on the {@code keys} specified. Each entry in the {@link Map} will have
     * {@link Optional} which may contain the {@link IdentifiedValue} associated with the key.
     */
    CompletionStage<Map<String, Optional<IdentifiedValue<V>>>> getBulkIdentified(Iterable<String> keys);

    /**
     * Removes the specified entry, iff the specified CAS identifier matches.
     *
     * @param key   the key of the entry to remove
     * @param casId the CAS identifier to match
     * @return whether the entry was removed
     */
    CompletionStage<Boolean> removeIf(String key, CasIdentifier casId);

    /**
     * Replaces the value for a specified entry, iff the specified CAS identifier matches.
     *
     * @param key      the key of the entry to replace it's value
     * @param casId    the CAS identifier to match
     * @param newValue the new value to replace with
     * @return whether the value was replaced
     */
    CompletionStage<Boolean> replaceIf(String key, CasIdentifier casId, V newValue);
}
