UPM.define('MultipleLicenseUpdateDialog', [
  'UpmDialog',
  'UpmAnalytics',
  'AuiFlagManager',
  'UpmEventBus'
], function (UpmDialog,
             UpmAnalytics,
             AuiFlagManager,
             UpmEventBus) {

  'use strict';

  return UpmDialog.extend({

    Pages: {
      LICENSE_ENTRY: 1,
      FULL_LICENSE_APP_WARNING: 2,
      EVAL_LICENSE_APP_INFO: 3,
    },

    template: com.atlassian.upm.application.multipleLicenseUpdateDialog,

    events: {
      'click #show-dc-apps': '_openAppListingTab',
      'click .learn-more': '_fireLearnMoreAnalytics',
    },

    _postShow: function () {
      if (!this.options.data.otherApplicationKey && !this.options.data.removeApplicationKey) {
        // we are only showing this dialog to show the apps, so go directly to that page
        this.options.page = this.Pages.LICENSE_ENTRY; // current page to hide
        this._showPage(this.options.data.singleEvalLicense ? this.Pages.EVAL_LICENSE_APP_INFO : this.Pages.FULL_LICENSE_APP_WARNING);
        this.$el.find(".hide-when-single-page").addClass("hidden");
      }
      UpmAnalytics.logEvent(
        'manageapps-multiple-license-update-show',
        this._getAnalyticsData()
      );
    },

    _getAnalyticsData: function() {
      const dialogPageNames = ["license-compatibility", "apps-compatibility-full", "apps-compatibility-eval"];
      return {
        appCrossgradeCount: this.options.data.appCrossgrades.length,
        newLicenseApplicationKey: this.options.data.newLicenseApplicationKey,
        otherApplicationKey: this.options.data.otherApplicationKey,
        removeApplicationKey: this.options.data.removeApplicationKey,
        dialogPage: this.options.page ? dialogPageNames[this.options.page - 1] : undefined
      }
    },

    _getReturnValue: function() {
      return {};
    },

    _openAppListingTab: function(e) {
      e.preventDefault();
      UpmAnalytics.logEvent('manageapps-see-data-center-apps-clicked', this._getAnalyticsData());
      var new_page = window.open();
      new_page.document.write(com.atlassian.upm.application.multipleLicenseAppDetailsPage({appCrossgrades: this.options.data.appCrossgrades}));
    },

    _fireLearnMoreAnalytics: function(e) {
      // don't preventDefault, we want the link to be followed
      UpmAnalytics.logEvent('manageapps-learn-more-clicked', this._getAnalyticsData());
    },

    _getLicenseUpdateParameters: function() {
      var makeLicense = function(key, text, del) {
        return { applicationKey: key, license: text, "delete": del};
      };

      var data = this.options.data;

      var licenses =  [
        makeLicense(data.newLicenseApplicationKey, data.newLicense, false)
      ];
      if (data.otherApplicationKey) {
        licenses.push(makeLicense(data.otherApplicationKey, this._otherLicense().val(), false));
      }
      if (data.removeApplicationKey) {
        licenses.push(makeLicense(data.removeApplicationKey, data.removeApplicationLicense, true));
      }
      return licenses;
    },

    _onConfirm: function () {
      this._hideErrors();
      var errorSpans = this._findErrorSpans(this, "other-license-group");
      errorSpans.errorSpan.addClass('hidden');
      if (this.options.data.otherApplicationKey && this._isBlank(this._otherLicense().val())) {
        this._showError(errorSpans, AJS.I18n.getText('upm.atlassian.other-license.missing'));
      } else {
        const showErrors = this._showErrors;
        const showWarnings = this._showWarnings;
        const options = this.options;
        const that = this;
        const Pages = this.Pages;

        this.options.model._multipleLicenseKeysOperation(
          this._getLicenseUpdateParameters(),
          (options.data.appCrossgrades.length === 0 && options.page === Pages.LICENSE_ENTRY)
          || options.page === Pages.FULL_LICENSE_APP_WARNING || options.page === Pages.EVAL_LICENSE_APP_INFO ? 'update' : 'validate'
        ).then(function (resp) {
            if (options.data.appCrossgrades.length !== 0 && options.page === Pages.LICENSE_ENTRY) {
              UpmAnalytics.logEvent('manageapps-multiple-license-update-continue', that._getAnalyticsData());
              that._showPage(resp.allEvaluationLicenses ? Pages.EVAL_LICENSE_APP_INFO : Pages.FULL_LICENSE_APP_WARNING);
            } else {
              UpmAnalytics.logEvent('manageapps-multiple-license-update-finish', that._getAnalyticsData());
              AuiFlagManager.flag({
                  type: 'success',
                  title: (AJS.I18n.getText('upm.applications.flags.license.added.multiple.dc.title')),
                  body: (options.data.otherApplicationName ?
                    AJS.I18n.getText('upm.applications.flags.license.added.multiple.dc.body', options.data.newLicenseApplicationName, options.data.otherApplicationName) :
                    AJS.I18n.getText('upm.applications.flags.license.added.single.dc.body', options.data.newLicenseApplicationName)),
                  close: 'manual',
                  delay: 300
                });
              AJS.$("#getting-started-link").click(
                function() {
                  UpmAnalytics.logEvent('manageapps-getting-started-with-dc-clicked', that._getAnalyticsData());
                });
              showWarnings(resp);
              UpmEventBus.trigger("refreshAllApplications");
              UpmDialog.prototype._onConfirm.apply(that);
            }
          }).fail(
          function (xhr, status, msg) {
            if (options.page !== 1) {
              // go back to the first page to show any errors
              that._showPage(Pages.LICENSE_ENTRY);
            }
            showErrors(xhr.responseJSON, options.data, that);
          }
        );
      }
    },

    _showPage: function(pageNumber) {
      var createPageSelector = function(n) {
        return ".page-" + n;
      };
      var oldSelectorName = createPageSelector(this.options.page);
      this.options.page = pageNumber;
      var newSelectorName = createPageSelector(pageNumber);
      this.$el.find(oldSelectorName).addClass("hidden");
      this.$el.find(newSelectorName).removeClass("hidden");
    },

    _hideErrors: function() {
      this.$el.find(".errorSpan").addClass("hidden");
    },

    _showErrors: function (json, data, that) {
      for (var applicationKey in json.results) {
        var errorSpans;
        if (json.results[applicationKey].errors.length > 0 &&
          (!json.results[applicationKey].errors[0].mismatchError || applicationKey === data.otherApplicationKey)) {
          if (applicationKey === data.newLicenseApplicationKey) {
            errorSpans = that._findErrorSpans(that, 'new-license-group');
          } else if (applicationKey === data.otherApplicationKey) {
            errorSpans = that._findErrorSpans(that, 'other-license-group');
          } else {
            var span = that.$el.find('#general-error');
            errorSpans = { errorSpan: span, errorTextSpan: span };
          }
          that._showError(errorSpans, json.results[applicationKey].errors[0].message);
        }
      }
    },

    _showError: function(errorSpans, message) {
      errorSpans.errorSpan.removeClass("hidden");
      errorSpans.errorTextSpan.html(message);
    },

    _findErrorSpans: function(that, group) {
      return { errorSpan: that.$el.find("div." + group + " span.error-span"),
               errorTextSpan: that.$el.find("div." + group + " span.error-text")};
    },

    _showWarnings: function(json) {
      for (var applicationKey in json.results) {
        for (var warning in json.results[applicationKey].warnings) {
          AuiFlagManager.flag({
            type: 'warning',
            body: (json.results[applicationKey].warnings[warning].message),
            close: 'manual',
            delay: 300
          });
        }
      }
    },

    _otherLicense: function() {
      return this.$el.find("#upm-atlassian-other-license");
    },

    _isBlank: function(str) {
      return (!str || /^\s*$/.test(str));
    },

    getResult: function () {
      var that = this;
      return UpmDialog.prototype.getResult.apply(this)
        .done(function () {
        })
        .fail(function () {
          UpmAnalytics.logEvent('manageapps-multiple-license-update-cancel', that._getAnalyticsData());
        });
    }
  });
});
