package com.atlassian.upm.license.role.confluence;

import com.atlassian.fugue.Option;
import com.atlassian.plugin.Plugin;
import com.atlassian.upm.license.role.spi.LicensingRole;
import com.atlassian.upm.license.role.spi.LicensingRoleCreationFailedException;
import com.atlassian.upm.license.role.spi.RoleBasedLicenseService;
import com.atlassian.user.Group;
import com.atlassian.util.concurrent.LazyReference;

import com.google.common.base.Function;

import static com.atlassian.fugue.Option.some;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Confluence-specific implementation of the role-based licensing SPI. This implementation is based on Confluence's
 * use of Atlassian User groups. This implementation can be changed to be backed by other means in the future, for
 * example, if/when Confluence supports pluggable global permissions.
 */
public class ConfluenceRoleBasedLicenseService implements RoleBasedLicenseService
{
    private final ConfluenceRoleBasedLicenseManager manager;

    public ConfluenceRoleBasedLicenseService(ConfluenceRoleBasedLicenseManager manager)
    {
        this.manager = checkNotNull(manager, "manager");
    }

    @Override
    public LicensingRole createLicensingRole(Plugin plugin, final String key, String nameI18nKey, String descriptionI18nKey) throws LicensingRoleCreationFailedException
    {
        LazyReference<Option<Group>> createGroup = new LazyReference<Option<Group>>()
        {
            @Override
            protected Option<Group> create() throws Exception
            {
                return some(manager.createGroup(key));
            }
        };

        Option<LicensingRole> licensingRole = manager.getGroup(key).orElse(createGroup).map(new Function<Group, LicensingRole>()
        {
            @Override
            public LicensingRole apply(Group group)
            {
                return new ConfluenceLicensingRole(manager, group, key);
            }
        });

        for (LicensingRole role : licensingRole)
        {
            return role;
        }

        throw new LicensingRoleCreationFailedException("Could not create a group for the licensing role: " + key);
    }

    @Override
    public LicensingRole getLicensingRole(Plugin plugin, final String key)
    {
        return manager.getGroup(key).map(new Function<Group, LicensingRole>()
        {
            @Override
            public LicensingRole apply(Group group)
            {
                return new ConfluenceLicensingRole(manager, group, key);
            }
        }).getOrNull();
    }

    @Override
    public void onPluginUnlicensedEvent(Plugin plugin)
    {
        // do nothing. this implementation doesn't need to clean up anything.
    }
}
