package com.atlassian.upm.license.storage.lib;

import com.atlassian.plugin.JarPluginArtifact;
import com.atlassian.plugin.PluginArtifact;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;

import static org.apache.commons.io.FileUtils.openOutputStream;
import static org.apache.commons.io.IOUtils.closeQuietly;

/**
 * Builds {@link PluginArtifact}s to be installed.
 */
public abstract class PluginArtifactFactory
{
    private static final int DOWNLOAD_BUFFER_SIZE = 1024 * 4;

    /**
     * Downloads a file from a URI and then builds a {@link PluginArtifact} from it.
     * Will return null if no artifact can be found at the given URI. We cannot use {@code Option}
     * here as this code is executed prior to binding to UPM's API.
     *
     * @param uri the URI
     * @return the plugin artifact
     * @throws IOException
     */
    public static PluginArtifact fromURI(URI uri) throws IOException
    {
        return fromInputStream(uri.toURL().openStream(), uri.getPath());
    }

    /**
     * Builds a {@link PluginArtifact} from a file on the local classpath.
     * Will return null if no artifact can be found at the given URI. We cannot use {@code Option}
     * here as this code is executed prior to binding to UPM's API.
     *
     * @param resourcePath the resource path
     * @return the plugin artifact
     * @throws IOException
     */
    public static PluginArtifact fromFile(String resourcePath) throws IOException
    {
        final InputStream in = PluginArtifactFactory.class.getClassLoader().getResourceAsStream(resourcePath);
        return fromInputStream(in, resourcePath);
    }

    /**
     * Creates a {@link PluginArtifact} from an {@link InputStream}. Includes closing the {@link InputStream} when finished.
     * @param in the input stream
     * @param resourcePath the path where the stream is located
     * @return a {@link PluginArtifact} from the stream
     * @throws IOException
     */
    public static PluginArtifact fromInputStream(InputStream in, String resourcePath) throws IOException
    {
        File file = File.createTempFile("plugin.", "." + extractFileName(resourcePath));
        FileOutputStream out = null;

        //read the plugin jar
        try
        {
            if (in == null)
            {
                return null;
            }
            
            out = openOutputStream(file);
            copy(in, out);
        }
        finally
        {
            closeQuietly(in);
            closeQuietly(out);
        }

        return new JarPluginArtifact(file);
    }

    private static String extractFileName(String path)
    {
        return path.substring(path.lastIndexOf('/') + 1);
    }

    private static void copy(InputStream in, FileOutputStream out) throws IOException
    {
        byte[] buffer = new byte[DOWNLOAD_BUFFER_SIZE];
        int n;
        while (-1 != (n = in.read(buffer)))
        {
            out.write(buffer, 0, n);
        }
    }
}
