package com.atlassian.marketplace.client.model;

import com.atlassian.upm.api.util.Option;

import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonProperty;

import static com.atlassian.marketplace.client.util.ModelUtil.requireProperty;
import static com.atlassian.upm.api.util.Option.option;

/**
 * A specific pricing tier for Paid-via-Atlassian plugin, corresponding to some combination
 * of license type, user count, etc.
 */
public final class PricingItem
{
    @JsonProperty private final long id;
    @JsonProperty private final String description;
    @JsonProperty private final String unitDescription;
    @JsonProperty private final float usdAmount;
    @JsonProperty private final int unitCount;
    @JsonProperty private final boolean enterprise;
    @JsonProperty private final boolean starter;
    @JsonProperty private final String saleType;
    @JsonProperty private final String licenseType;
    @JsonProperty private final boolean legacyDisplay;
    
    @JsonCreator
    PricingItem(@JsonProperty("id") Long id,
                @JsonProperty("description") String description,
                @JsonProperty("unitDescription") String unitDescription,
                @JsonProperty("usdAmount") Float usdAmount,
                @JsonProperty("unitCount") Integer unitCount,
                @JsonProperty("enterprise") Boolean enterprise,
                @JsonProperty("starter") Boolean starter,
                @JsonProperty("saleType") String saleType,
                @JsonProperty("licenseType") String licenseType,
                @JsonProperty("legacyDisplay") Boolean legacyDisplay)
    {
        this.id = requireProperty(id, "id");
        this.description = description;  // optional
        this.unitDescription = unitDescription;  // optional
        this.usdAmount = requireProperty(usdAmount, "usdAmount");
        this.unitCount = requireProperty(unitCount, "unitCount");
        this.enterprise = requireProperty(enterprise, "enterprise");
        this.starter = requireProperty(starter, "starter");
        this.saleType = requireProperty(saleType, "saleType");
        this.licenseType = requireProperty(licenseType, "licenseType");
        this.legacyDisplay = (legacyDisplay == null) ? false : legacyDisplay;  // optional
    }

    /**
     * A unique identifier for this pricing tier.
     */
    public long getId()
    {
        return id;
    }

    /**
     * A human-readable description (e.g. "Commercial 25 Users").
     */
    @JsonIgnore
    public Option<String> getDescription()
    {
        return option(description);
    }

    /**
     * A human-readable description of the meaning of {@link #getUnitCount()} (e.g. "Users");
     */
    @JsonIgnore
    public Option<String> getUnitDescription()
    {
        return option(unitDescription);
    }

    /**
     * The price in U.S. dollars.
     */
    public float getUsdAmount()
    {
        return usdAmount;
    }

    /**
     * The number of licensed users/agents/etc. for which this price applies. 
     */
    public int getUnitCount()
    {
        return unitCount;
    }

    /**
     * True if this price applies to enterprise licenses.
     */
    public boolean isEnterprise()
    {
        return enterprise;
    }

    /**
     * True if this price applies to starter licenses.
     */
    public boolean isStarter()
    {
        return starter;
    }

    /**
     * A short string describing whether this is for first-time purchase ("purchase") or renewal ("renewal").
     */
    public String getSaleType()
    {
        return saleType;
    }

    /**
     * Name of the license type for which this price applies (e.g. "Commercial").
     */
    public String getLicenseType()
    {
        return licenseType;
    }

    /**
     * True if this is legacy pricing.
     */
    public boolean isLegacyDisplay()
    {
        return legacyDisplay;
    }
}
