package com.atlassian.marketplace.client.model;

import java.net.URI;
import java.util.Collection;
import java.util.Date;
import java.util.Map;

import com.atlassian.upm.api.util.Option;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;

import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonProperty;

import static com.atlassian.marketplace.client.util.ModelUtil.requireLink;
import static com.atlassian.marketplace.client.util.ModelUtil.requireList;
import static com.atlassian.marketplace.client.util.ModelUtil.requireProperty;
import static com.atlassian.upm.api.util.Option.option;

/**
 * Information about a specific version of a plugin.
 */
public final class PluginVersion
{
    @JsonProperty private final Links links;
    @JsonProperty private final long buildNumber;
    @JsonProperty private final Date releaseDate;
    @JsonProperty private final String releasedBy;
    @JsonProperty private final String releaseFocus;
    @JsonProperty private final boolean stable;
    @JsonProperty private final String status;
    @JsonProperty private final LicenseType license;
    @JsonProperty private final String version;
    @JsonProperty private final boolean deployable;
    @JsonProperty private final String supportType;
    @JsonProperty private final String pluginSystemVersion;
    @JsonProperty private final String releaseNotes;
    @JsonProperty private final String summary;
    @JsonProperty private final Collection<ApplicationSummary> compatibleApplications;
    @JsonProperty private final Collection<VersionCompatibility> compatibilities;
    @JsonProperty("marketplaceType") private final Map<String, String> marketplaceTypeMap;
    @JsonProperty private final Collection<Screenshot> screenshots;
    @JsonProperty private final String youtubeId;

    private final URI webUri;
    
    // fields omitted:  marketplaceAgreementAccepted, rejectionReason
    
    @JsonCreator
    PluginVersion(@JsonProperty("links") Links links,
                  @JsonProperty("buildNumber") Long buildNumber,
                  @JsonProperty("releaseDate") Date releaseDate,
                  @JsonProperty("releasedBy") String releasedBy,
                  @JsonProperty("releaseFocus") String releaseFocus,
                  @JsonProperty("stable") Boolean stable,
                  @JsonProperty("status") String status,
                  @JsonProperty("license") LicenseType license,
                  @JsonProperty("version") String version,
                  @JsonProperty("deployable") Boolean deployable,
                  @JsonProperty("supportType") String supportType,
                  @JsonProperty("pluginSystemVersion") String pluginSystemVersion,
                  @JsonProperty("releaseNotes") String releaseNotes,
                  @JsonProperty("summary") String summary,
                  @JsonProperty("compatibleApplications") Collection<ApplicationSummary> compatibleApplications,
                  @JsonProperty("compatibilities") Collection<VersionCompatibility> compatibilities,
                  @JsonProperty("marketplaceType") Map<String, String> marketplaceTypeMap,
                  @JsonProperty("screenshots") Collection<Screenshot> screenshots,
                  @JsonProperty("youtubeId") String youtubeId)
    {
        this.links = requireProperty(links, "links");
        this.buildNumber = requireProperty(buildNumber, "buildNumber");
        this.releaseDate = requireProperty(releaseDate, "releaseDate");
        this.releasedBy = releasedBy;  // optional
        this.releaseFocus = releaseFocus;  // optional
        this.stable = requireProperty(stable, "stable");
        this.status = requireProperty(status, "status");
        this.license = requireProperty(license, "license");
        this.version = requireProperty(version, "version");
        this.deployable = requireProperty(deployable, "deployable");
        this.supportType = requireProperty(supportType, "supportType");
        this.pluginSystemVersion = requireProperty(pluginSystemVersion, "pluginSystemVersion");
        this.releaseNotes = releaseNotes;  // optional
        this.summary = summary;  // optional
        this.compatibleApplications = requireList(compatibleApplications, "compatibleApplications");
        this.compatibilities = requireList(compatibilities, "compatibilities");
        this.marketplaceTypeMap = ImmutableMap.copyOf(requireProperty(marketplaceTypeMap, "marketplaceType"));
        this.screenshots = requireList(screenshots, "screenshots");
        this.youtubeId = youtubeId;  // optional
        
        this.webUri = requireLink(links, "alternate");
    }

    public Links getLinks()
    {
        return links;
    }

    /**
     * An integer that uniquely identifies this version and determines the sort order of versions.
     */
    public long getBuildNumber()
    {
        return buildNumber;
    }

    /**
     * The date on which the version was made available.
     */
    public Date getReleaseDate()
    {
        return releaseDate;
    }

    /**
     * The name of the user who released the version, if available.
     */
    @JsonIgnore
    public Option<String> getReleasedBy()
    {
        return option(releasedBy);
    }

    /**
     * A short summary of the nature of the version (major feature changes, bug fixes, etc.).
     */
    @JsonIgnore
    public Option<String> getReleaseFocus()
    {
        return option(releaseFocus);
    }

    /**
     * True if this is a stable (non-alpha, non-beta) release.
     */
    public boolean isStable()
    {
        return stable;
    }

    /**
     * Indicates whether the version is published or is in a draft state.
     */
    public String getStatus()
    {
        return status;
    }

    /**
     * True if this version is visible to everyone, false if it is a draft version.  Draft versions
     * can only be seen by an authenticated user who has permission to edit the plugin.
     */
    public boolean isPublished()
    {
        return status.equalsIgnoreCase("Published");
    }
    
    /**
     * Information about how this version is licensed.
     */
    public LicenseType getLicense()
    {
        return license;
    }

    /**
     * The version number string (e.g. "1.0.0");
     */
    public String getVersion()
    {
        return version;
    }

    /**
     * True if this version can be installed directly via the Plugin Manager.
     */
    public boolean isDeployable()
    {
        return deployable;
    }

    /**
     * A short string describing what type of support is available (e.g. "Vendor").
     */
    public String getSupportType()
    {
        return supportType;
    }

    /**
     * The Atlassian plugins framework version of the plugin (e.g. "Two").
     */
    public String getPluginSystemVersion()
    {
        return pluginSystemVersion;
    }

    /**
     * Optional release notes for this version.
     */
    @JsonIgnore
    public Option<String> getReleaseNotes()
    {
        return option(releaseNotes);
    }

    /**
     * An optional short description of this version.
     */
    @JsonIgnore
    public Option<String> getSummary()
    {
        return option(summary);
    }

    /**
     * The applications compatible with this plugin version.
     */
    @JsonIgnore
    public Iterable<ApplicationSummary> getCompatibleApplications()
    {
        return ImmutableList.copyOf(compatibleApplications);
    }

    /**
     * The specific application version(s) compatible with this plugin version.
     */
    @JsonIgnore
    public Iterable<VersionCompatibility> getCompatibilities()
    {
        return ImmutableList.copyOf(compatibilities);
    }

    /**
     * Indicates which payment system, if any, is used for buying licenses for the plugin.
     */
    public MarketplaceType getMarketplaceType()
    {
        return MarketplaceType.fromName(marketplaceTypeMap.get("type")).getOrElse(MarketplaceType.FREE);
    }

    /**
     * A human-readable explanation of the value of {@link #getMarketplaceType()}.
     */
    public String getMarketplaceTypeTooltip()
    {
        return option(marketplaceTypeMap.get("tooltip")).getOrElse("");
    }

    /**
     * The screenshot images associated with this version, if any.
     */
    @JsonIgnore
    public Iterable<Screenshot> getScreenshots()
    {
        return ImmutableList.copyOf(screenshots);
    }
    
    /**
     * The YouTube video key associated with this version, if any.
     */
    @JsonIgnore
    public Option<String> getYoutubeId()
    {
        return option(youtubeId);
    }
    
    /**
     * Shortcut for getting the "alternate" link, if any, from the links map.  This is the
     * URI of the plugin version details on the Marketplace website.
     */
    @JsonIgnore
    public URI getWebUri()
    {
        return webUri;
    }
    
    /**
     * Shortcut for getting the "binary" link, if any, from the links map.  This is the URI
     * from which the plugin version can be downloaded.
     */
    @JsonIgnore
    public Option<URI> getBinaryUri()
    {
        return links.get("binary");
    }
}
