package com.atlassian.marketplace.client.model;

import java.util.Collection;
import java.util.Date;

import com.atlassian.upm.api.util.Option;

import com.google.common.collect.ImmutableList;

import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonProperty;

import static com.atlassian.marketplace.client.util.ModelUtil.requireList;
import static com.atlassian.marketplace.client.util.ModelUtil.requireProperty;
import static com.atlassian.upm.api.util.Option.option;

/**
 * Abbreviated information about a plugin listed on the Atlassian Marketplace.
 */
public final class PluginSummary
{
    @JsonProperty private final Links links;
    @JsonProperty private final String name;
    @JsonProperty private final String pluginKey;
    @JsonProperty private final Vendor vendor;
    @JsonProperty private final PluginVersion version;
    @JsonProperty private final Icons icons;
    @JsonProperty private final String summary;
    @JsonProperty private final ReviewSummary reviewSummary;
    @JsonProperty private final Collection<PluginCategory> categories;
    @JsonProperty private final long downloadCount;
    @JsonProperty private final Modification lastModified;
    @JsonProperty private final Date lastReleasedDate;
    @JsonProperty private final Collection<ApplicationSummary> compatibleApplications;
    @JsonProperty private final Pricing pricing;
    
    @JsonCreator
    PluginSummary(@JsonProperty("links") Links links,
                  @JsonProperty("name") String name,
                  @JsonProperty("pluginKey") String pluginKey,
                  @JsonProperty("vendor") Vendor vendor,
                  @JsonProperty("version") PluginVersion version,
                  @JsonProperty("icons") Icons icons,
                  @JsonProperty("summary") String summary,
                  @JsonProperty("reviewSummary") ReviewSummary reviewSummary,
                  @JsonProperty("categories") Collection<PluginCategory> categories,
                  @JsonProperty("downloadCount") Long downloadCount,
                  @JsonProperty("lastModified") Modification lastModified,
                  @JsonProperty("lastReleasedDate") Date lastReleasedDate,
                  @JsonProperty("compatibleApplications") Collection<ApplicationSummary> compatibleApplications,
                  @JsonProperty("pricing") Pricing pricing)
    {
        this.links = requireProperty(links, "links");
        this.name = requireProperty(name, "name");
        this.pluginKey = requireProperty(pluginKey, "pluginKey");
        this.vendor = vendor;  // optional
        this.version = requireProperty(version, "version");
        this.icons = requireProperty(icons, "icons");
        this.summary = summary;  // optional
        this.reviewSummary = requireProperty(reviewSummary, "reviewSummary");
        this.categories = requireList(categories, "categories");
        this.downloadCount = requireProperty(downloadCount, "downloadCount");
        this.lastModified = lastModified;  // optional
        this.lastReleasedDate = lastReleasedDate;  // optional
        this.compatibleApplications = requireList(compatibleApplications, "compatibleApplications");
        this.pricing = pricing;  // optional
    }

    public Links getLinks()
    {
        return links;
    }

    /**
     * The plugin name.
     */
    public String getName()
    {
        return name;
    }

    /**
     * The unique plugin key.
     */
    public String getPluginKey()
    {
        return pluginKey;
    }

    /**
     * The plugin's vendor.
     */
    @JsonIgnore
    public Option<Vendor> getVendor()
    {
        return option(vendor);
    }
    
    /**
     * The specific version being queried.  By default, this is the latest version compatible
     * with the application version that was specified in the query, or if none was specified,
     * then the latest published version.
     */
    public PluginVersion getVersion()
    {
        return version;
    }

    /**
     * Icon/logo resources associated with the plugin.
     */
    public Icons getIcons()
    {
        return icons;
    }
    
    /**
     * A shorter description of the plugin.
     */
    @JsonIgnore
    public Option<String> getSummary()
    {
        return option(summary);
    }

    /**
     * A summary of reviews of the plugin.
     */
    public ReviewSummary getReviewSummary()
    {
        return reviewSummary;
    }

    /**
     * Categories to which this plugin belongs.
     */
    @JsonIgnore
    public Iterable<PluginCategory> getCategories()
    {
        return ImmutableList.copyOf(categories);
    }

    /**
     * The number of times this plugin has been downloaded.
     */
    public long getDownloadCount()
    {
        return downloadCount;
    }

    /**
     * Information about the last change to this plugin.
     */
    @JsonIgnore
    public Option<Modification> getLastModified()
    {
        return option(lastModified);
    }

    /**
     * The date on which the most recent version was released.
     */
    @JsonIgnore
    public Option<Date> getLastReleasedDate()
    {
        return option(lastReleasedDate);
    }

    /**
     * The Atlassian application(s) that this plugin is compatible with.
     */
    @JsonIgnore
    public Iterable<ApplicationSummary> getCompatibleApplications()
    {
        return ImmutableList.copyOf(compatibleApplications);
    }

    /**
     * Pricing information, if this is a Paid-via-Atlassian plugin.
     */
    @JsonIgnore
    public Option<Pricing> getPricing()
    {
        return option(pricing);
    }    
}
