package com.atlassian.marketplace.client.impl;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URLEncoder;

import com.atlassian.marketplace.client.HttpConfiguration;
import com.atlassian.marketplace.client.MarketplaceClient;
import com.atlassian.marketplace.client.MpacException;
import com.atlassian.marketplace.client.api.Applications;
import com.atlassian.marketplace.client.api.Categories;
import com.atlassian.marketplace.client.api.Plugins;
import com.atlassian.marketplace.client.impl.representations.RootRepresentation;
import com.atlassian.marketplace.client.model.Links;
import com.atlassian.upm.api.util.Option;

import com.google.common.collect.Multimap;

import static com.atlassian.upm.api.util.Option.none;
import static com.atlassian.upm.api.util.Option.some;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.io.IOUtils.closeQuietly;

/**
 * Default implementation of {@link MarketplaceClient}.
 */
public final class DefaultMarketplaceClient implements MarketplaceClient
{
    public static final URI DEFAULT_SERVER_URI = URI.create("https://marketplace.atlassian.com");
    public static final String API_VERSION = "1.0";

    private final URI baseUri;
    private final HttpHelper httpHelper;
    private final EntityEncoding encoding;
    
    /**
     * Constructs a {@link DefaultMarketplaceClient} using default settings for all properties.  This is
     * equivalent to <tt>DefaultMarketplaceClient(DEFAULT_SERVER_URI, HttpConfiguration.builder().build())</tt>.
     */
    public DefaultMarketplaceClient()
    {
        this(DEFAULT_SERVER_URI, HttpConfiguration.builder().build());
    }
    
    /**
     * Constructs a {@link DefaultMarketplaceClient} using the default HTTP implementation.
     * 
     * @param baseUri  The base URI of the MPAC server; the client will add "/rest" and the the API version number.
     * @param configuration  Client parameters such as timeouts, authentication, and proxy settings.
     */
    public DefaultMarketplaceClient(URI baseUri, HttpConfiguration configuration)
    {
        this(baseUri, new CommonsHttpHelper(configuration, baseUri), new JsonEntityEncoding());
    }
    
    /**
     * Constructs a {@link DefaultMarketplaceClient} using a specific HTTP implementation and
     * response parser.
     * 
     * @param baseUri  The base URI of the MPAC REST API, not including the version number.
     * @param httpHelper  An object that will execute all HTTP requests.
     * @param encoding  An {@link EntityEncoding} implementation.
     */
    public DefaultMarketplaceClient(URI baseUri, HttpHelper httpHelper, EntityEncoding encoding)
    {
        this.baseUri = normalizeBaseUri(checkNotNull(baseUri, "baseUri")).resolve("rest/" + API_VERSION + "/");
        this.httpHelper = checkNotNull(httpHelper, "httpHelper");
        this.encoding = encoding;
    }
    
    public boolean isReachable()
    {
        try
        {
            getRoot();
            return true;
        }
        catch (MpacException e)
        {
            return false;
        }
    }
    
    public Plugins plugins() throws MpacException
    {
        return new PluginsImpl(this, getRoot());
    }

    public Categories categories() throws MpacException
    {
        return new CategoriesImpl(this, getRoot());
    }

    public Applications applications() throws MpacException
    {
        return new ApplicationsImpl(this, getRoot());
    }

    private static URI normalizeBaseUri(URI baseUri)
    {
        URI norm = baseUri.normalize();
        if (norm.getPath().endsWith("/"))
        {
            return norm;
        }
        return URI.create(norm.toString() + "/");
    }

    RootRepresentation getRoot() throws MpacException
    {
        return getEntity(baseUri, RootRepresentation.class);
    }
    
    <T> T getEntity(URI uri, Class<T> type) throws MpacException
    {
        HttpHelper.Response response = httpHelper.get(uri);
        try
        {
            if ((response.getStatus() >= 400) || (response.getStatus() == 204))
            {
                throw new MpacException.ServerError(response.getStatus());
            }
            return decode(response.getContentStream(), type);
        }
        finally
        {
            response.close();
        }
    }
    
    <T> Option<T> getOptionalEntity(URI uri, Class<T> type) throws MpacException
    {
        HttpHelper.Response response = httpHelper.get(uri);
        try
        {
            if ((response.getStatus() == 204) || (response.getStatus() == 404))
            {
                return none();
            }
            if (response.getStatus() >= 400)
            {
                throw new MpacException.ServerError(response.getStatus());
            }
            if (response.isEmpty())
            {
                return none();
            }
            else
            {
                return some(decode(response.getContentStream(), type));
            }
        }
        finally
        {
            response.close();
        }
    }

    void postParams(URI uri, Multimap<String, String> params) throws MpacException
    {
        HttpHelper.Response response = httpHelper.postParams(uri, params);
        try
        {
            if (response.getStatus() >= 400)
            {
                throw new MpacException.ServerError(response.getStatus());
            }
        }
        finally
        {
            response.close();
        }
    }
    
    <T> void putEntity(URI uri, T entity) throws MpacException
    {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        encoding.encode(bos, entity);
        HttpHelper.Response response = httpHelper.put(uri, bos.toByteArray());
        if (response.getStatus() >= 400)
        {
            throw new MpacException.ServerError(response.getStatus());
        }
    }
    
    void deleteEntity(URI uri) throws MpacException
    {
        HttpHelper.Response response = httpHelper.delete(uri);
        if (response.getStatus() >= 400)
        {
            throw new MpacException.ServerError(response.getStatus());
        }
    }
    
    URI requireLinkUri(Links links, String rel, Class<?> entityClass) throws MpacException
    {
        for (URI href: links.get(rel))
        {
            return resolveLink(href);
        }
        throw new MpacException("Missing required API link \"" + rel + "\" from " + entityClass.getSimpleName());
    }

    URI resolveLink(URI href)
    {
        return href.isAbsolute() ? href : baseUri.resolve(href.toString());
    }
    
    String urlEncode(String s)
    {
        try
        {
            return URLEncoder.encode(s, "UTF-8");
        }
        catch (UnsupportedEncodingException e)
        {
            throw new IllegalStateException(e);
        }
    }
    
    private <T> T decode(InputStream is, Class<T> type) throws MpacException
    {
        try
        {
            return encoding.decode(is, type);
        }
        finally
        {
            closeQuietly(is);
        }
    }
}
