package com.atlassian.marketplace.client.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.SocketException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.atlassian.marketplace.client.HttpConfiguration;
import com.atlassian.marketplace.client.MarketplaceClient;
import com.atlassian.marketplace.client.MpacException;
import com.atlassian.marketplace.client.RequestDecorator;
import com.atlassian.upm.api.util.Option;

import com.google.common.collect.Multimap;

import org.apache.http.Header;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.NTCredentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.AuthCache;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.client.cache.CacheConfig;
import org.apache.http.impl.client.cache.CachingHttpClient;
import org.apache.http.impl.conn.PoolingClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.params.BasicHttpParams;
import org.apache.http.params.HttpParams;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.atlassian.upm.api.util.Option.some;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.http.client.params.ClientPNames.COOKIE_POLICY;
import static org.apache.http.client.params.CookiePolicy.IGNORE_COOKIES;
import static org.apache.http.client.protocol.ClientContext.AUTH_CACHE;
import static org.apache.http.conn.params.ConnRoutePNames.DEFAULT_PROXY;
import static org.apache.http.entity.ContentType.APPLICATION_JSON;
import static org.apache.http.params.CoreConnectionPNames.CONNECTION_TIMEOUT;
import static org.apache.http.params.CoreConnectionPNames.SO_TIMEOUT;
import static org.apache.http.util.EntityUtils.consumeQuietly;

/**
 * HTTP wrapper based on Apache HttpComponents.
 */
public final class CommonsHttpHelper implements HttpHelper
{
    private static final Logger logger = LoggerFactory.getLogger(MarketplaceClient.class);
    
    private final HttpClient client;
    private final HttpConfiguration config;
    private final RequestDecorator requestDecorator;
    
    public CommonsHttpHelper(HttpConfiguration configuration,
                             URI baseUri)
    {
        this.config = checkNotNull(configuration, "configuration");
        this.client = createCachingHttpClient(config, baseUri);
        this.requestDecorator = configuration.getRequestDecorator();
    }
    
    @Override
    public Response get(URI uri) throws MpacException
    {
        HttpGet method = new HttpGet(uri);
        configureMethod(method);
        return new ResponseImpl(executeMethod(method));
    }

    @Override
    public Response postParams(URI uri, Multimap<String, String> params) throws MpacException
    {
        HttpPost method = new HttpPost(uri);
        configureMethod(method);
        List<NameValuePair> formParams = new ArrayList<NameValuePair>();
        for (Map.Entry<String, String> param: params.entries())
        {
            formParams.add(new BasicNameValuePair(param.getKey(), param.getValue()));
        }
        try
        {
            method.setEntity(new UrlEncodedFormEntity(formParams));
        }
        catch (UnsupportedEncodingException e)
        {
            throw new MpacException(e);
        }
        return new ResponseImpl(executeMethod(method));
    }
    
    @Override
    public Response put(URI uri, byte[] content) throws MpacException
    {
        HttpPut method = new HttpPut(uri);
        configureMethod(method);
        method.setEntity(new ByteArrayEntity(content, APPLICATION_JSON));
        return new ResponseImpl(executeMethod(method));
    }
    
    @Override
    public Response delete(URI uri) throws MpacException
    {
        HttpDelete method = new HttpDelete(uri);
        configureMethod(method);
        return new ResponseImpl(executeMethod(method));
    }
    
    private void configureMethod(HttpUriRequest method)
    {
        if (requestDecorator != null)
        {
            for (Map.Entry<String, String> header: requestDecorator.getRequestHeaders().entrySet())
            {
                method.addHeader(header.getKey(), header.getValue());
            }
        }
    }
    
    private HttpResponse executeMethod(HttpUriRequest method) throws MpacException
    {
        logger.info(method.getMethod() + " " + method.getURI());
        try
        {
            return client.execute(method, createRequestContext(method));
        }
        catch (SocketException e)
        {
            throw new MpacException.ConnectionFailure(e);
        }
        catch (IOException e)
        {
            throw new MpacException(e);
        }
    }
    
    private HttpContext createRequestContext(HttpUriRequest method)
    {
        HttpContext ctx = new BasicHttpContext();
        if (config.hasCredentials())
        {
            // enable preemptive authentication for this request
            AuthCache authCache = new BasicAuthCache();
            authCache.put(new HttpHost(method.getURI().getHost(), method.getURI().getPort(), method.getURI().getScheme()),
                          new BasicScheme());
            ctx.setAttribute(AUTH_CACHE, authCache);
        }
        return ctx;
    }
    
    private static HttpClient createCachingHttpClient(HttpConfiguration config, URI baseUri)
    {
        HttpClient realClient = createHttpClient(config, some(baseUri));
        
        CacheConfig cacheConfig = new CacheConfig();
        cacheConfig.setSharedCache(false);
        cacheConfig.setMaxCacheEntries(config.getMaxCacheEntries());
        cacheConfig.setMaxObjectSize(config.getMaxCacheObjectSize());
        
        CachingHttpClient cachingClient = new CachingHttpClient(realClient, cacheConfig);
        
        return cachingClient;
    }
    
    /**
     * Helper method that configures an HttpClient with the specified proxy/timeout properties.
     * @param config  an {@link HttpConfiguration}
     * @param baseUri  base URI of the Marketplace server; this is only relevant if you're providing
     * basicauth credentials for the server itself (rather than for the proxy)
     * @return  a configured HttpClient
     */
    public static DefaultHttpClient createHttpClient(HttpConfiguration config, Option<URI> baseUri)
    {
        PoolingClientConnectionManager httpConnectionManager = new PoolingClientConnectionManager();
        httpConnectionManager.setDefaultMaxPerRoute(config.getMaxConnections());
        
        HttpParams httpParams = new BasicHttpParams();
        httpParams.setIntParameter(CONNECTION_TIMEOUT, config.getConnectTimeoutMillis());
        httpParams.setIntParameter(SO_TIMEOUT, config.getReadTimeoutMillis());
        httpParams.setParameter(COOKIE_POLICY, IGNORE_COOKIES);
        
        DefaultHttpClient httpClient = new DefaultHttpClient(httpConnectionManager, httpParams);
        
        if (config.hasCredentials())
        {
            for (URI uri: baseUri)
            {
                httpClient.getCredentialsProvider().setCredentials(new AuthScope(uri.getHost(), uri.getPort()),
                        new UsernamePasswordCredentials(config.getUsername(), config.getPassword()));
            }
        }
        
        if (config.hasProxy())
        {
            HttpConfiguration.ProxyConfiguration proxy = config.getProxy();
            httpClient.getParams().setParameter(DEFAULT_PROXY, new HttpHost(proxy.getHost(), proxy.getPort()));
            
            if (proxy.hasAuth())
            {
                AuthScope proxyAuthScope = new AuthScope(proxy.getHost(), proxy.getPort());
                Credentials proxyCredentials;
                switch (proxy.getAuthMethod())
                {
                    case NTLM:
                        proxyCredentials = new NTCredentials(proxy.getUsername(), proxy.getPassword(),
                                                             (proxy.getNtlmWorkstation() == null) ? "" : proxy.getNtlmWorkstation(),
                                                             (proxy.getNtlmDomain() == null) ? "" : proxy.getNtlmDomain());
                        break;
                        
                    default:
                        proxyCredentials = new UsernamePasswordCredentials(proxy.getUsername(), proxy.getPassword());
                        break;
                }
                httpClient.getCredentialsProvider().setCredentials(proxyAuthScope, proxyCredentials);
            }

        }
        
        return httpClient;
    }
    
    private static class ResponseImpl implements Response
    {
        private final HttpResponse response;
        
        ResponseImpl(HttpResponse response)
        {
            this.response = response;
        }
        
        public int getStatus()
        {
            return response.getStatusLine().getStatusCode();
        }
        
        public InputStream getContentStream() throws MpacException
        {
            try
            {
                return response.getEntity().getContent();
            }
            catch (IOException e)
            {
                throw new MpacException(e);
            }
        }
        
        public boolean isEmpty()
        {
            Header h = response.getFirstHeader("Content-Length");
            return (h != null) && (h.getValue().trim().equals("0"));
        }
        
        public void close()
        {
            if (response.getEntity() != null)
            {
                consumeQuietly(response.getEntity());
            }
        }
    }
}
