package com.atlassian.marketplace.client.api;

import java.util.HashSet;
import java.util.Set;

import com.atlassian.marketplace.client.model.MarketplaceType;
import com.atlassian.marketplace.client.model.PluginVersion;
import com.atlassian.upm.api.util.Option;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;

import static com.atlassian.upm.api.util.Option.none;
import static com.atlassian.upm.api.util.Option.some;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.isEmpty;
import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang.StringUtils.isNotBlank;

/**
 * Encapsulates plugin search parameters that can be passed to {@link Plugins#find(PluginQuery)}.
 */
public final class PluginQuery
{
    /**
     * Constants representing preset plugin list views, which may affect both the set of
     * plugins being queried and the sort order of the results.
     */
    public enum View
    {
        /**
         * Restricts the query to plugins made by Atlassian, with the most recently updated plugins first.
         */
        BY_ATLASSIAN ("by-atlassian"),
        /**
         * Restricts the query to a set of plugins picked by Atlassian staff, with the most recently
         * updated plugins first.
         */
        FEATURED ("featured"),
        /**
         * Queries all plugins, in descending order of total number of downloads.
         */
        POPULAR ("popular"),
        /**
         * Queries all plugins, in descending order of modification date.
         */
        RECENTLY_UPDATED ("recent"),
        /**
         * Queries all Paid-via-Atlassian plugins, in descending order of gross sales.
         */
        TOP_GROSSING ("top-grossing"),
        /**
         * Queries all plugins, in descending order of number of recent downloads.
         */
        TRENDING ("trending");
        
        private final String key;
        
        private View(String key)
        {
            this.key = key;
        }
        
        public String getKey()
        {
            return key;
        }
    }

    /**
     * Constants for querying plugins according to their paid/free attributes.
     */
    public enum Cost
    {
        /**
         * Restricts the query to plugins where {@link PluginVersion#getMarketplaceType()} is {@link MarketplaceType#FREE}.
         */
        FREE ("free"),
        /**
         * Restricts the query to plugins where {@link PluginVersion#getMarketplaceType()} is
         * either {@link MarketplaceType#PAID_VIA_VENDOR} or {@link MarketplaceType#PAID_VIA_ATLASSIAN}.
         */
        ALL_PAID ("paid"),
        /**
         * Restricts the query to plugins where {@link PluginVersion#getMarketplaceType()} is
         * {@link MarketplaceType#PAID_VIA_ATLASSIAN}.
         */
        PAID_VIA_ATLASSIAN ("marketplace");

        private final String key;
        
        private Cost(String key)
        {
            this.key = key;
        }
        
        public String getKey()
        {
            return key;
        }

        public static Option<Cost> fromKey(String key)
        {
            if (isBlank(key))
            {
                return none();
            }
            for (Cost type : values())
            {
                if (type.getKey().equals(key))
                {
                    return some(type);
                }
            }
            return none();
        }
    }
    
    private final Option<String> searchText;
    private final Option<ApplicationKey> application;
    private final Option<Long> appBuildNumber;
    private final Option<View> view;
    private final Option<Cost> cost;
    private final Iterable<String> categories;
    private final Option<PricingQuery> includePricing;
    private final int offset;
    private final Option<Integer> limit;
    
    /**
     * Returns a new {@link Builder} for constructing a PluginQuery.
     */
    public static Builder builder()
    {
        return new Builder();
    }

    /**
     * Returns a new {@link Builder} for constructing a PluginQuery based on an existing PluginQuery.
     * @param query the PluginQuery to copy the fields from
     * @return  a new {@link Builder} for constructing a PluginQuery based on an existing PluginQuery
     */
    public static Builder builder(PluginQuery query)
    {
        Builder builder = builder().searchText(query.getSearchText())
            .application(query.getApplication())
            .appBuildNumber(query.getAppBuildNumber())
            .view(query.getView())
            .cost(query.getCost())
            .includePricing(query.getIncludePricing())
            .offset(query.getOffset())
            .limit(query.getLimit());

        for (String category : query.getCategories())
        {
            builder.category(category);
        }

        return builder;
    }

    private PluginQuery(Builder builder)
    {
        searchText = builder.searchText;
        application = builder.application;
        appBuildNumber = builder.appBuildNumber;
        view = builder.view;
        cost = builder.cost;
        categories = ImmutableList.copyOf(builder.categories);
        includePricing = builder.includePricing;
        offset = builder.offset;
        limit = builder.limit;
    }
    
    public Option<String> getSearchText()
    {
        return searchText;
    }
    
    public Option<ApplicationKey> getApplication()
    {
        return application;
    }

    public Option<Long> getAppBuildNumber()
    {
        return appBuildNumber;
    }

    public Option<View> getView()
    {
        return view;
    }

    public Option<Cost> getCost()
    {
        return cost;
    }

    public Iterable<String> getCategories()
    {
        return categories;
    }
    
    public Option<PricingQuery> getIncludePricing()
    {
        return includePricing;
    }
    
    public int getOffset()
    {
        return offset;
    }
    
    public Option<Integer> getLimit()
    {
        return limit;
    }

    @Override
    public String toString()
    {
        ImmutableList.Builder<String> params = ImmutableList.builder();
        for (String s: searchText)
        {
            params.add("text(" + s + ")");
        }
        for (ApplicationKey a: application)
        {
            params.add("application(" + a.getKey() + ")");
        }
        for (Long ab: appBuildNumber)
        {
            params.add("appBuildNumber(" + ab + ")");
        }
        for (View v: view)
        {
            params.add("view(" + v.name() + ")");
        }
        for (Cost c: cost)
        {
            params.add("cost(" + c.name() + ")");
        }
        if (!isEmpty(categories))
        {
            params.add("categories(" + Joiner.on(", ").join(categories) + ")");
        }
        if (offset > 0)
        {
            params.add("offset(" + offset + ")");
        }
        for (Integer l: limit)
        {
            params.add("limit(" + l + ")");
        }
        return "PluginQuery(" + Joiner.on(", ").join(params.build()) + ")";
    }
    
    @Override
    public boolean equals(Object other)
    {
        return (other instanceof PluginQuery) ? toString().equals(other.toString()) : false;
    }

    @Override
    public int hashCode()
    {
        return toString().hashCode();
    }
    
    public static class Builder
    {
        private Option<String> searchText = none();
        private Option<ApplicationKey> application = none();
        private Option<Long> appBuildNumber = none();
        private Option<View> view = none();
        private Option<Cost> cost = none();
        private Set<String> categories = new HashSet<String>();
        private Option<PricingQuery> includePricing = none();
        private int offset = 0;
        private Option<Integer> limit = none();
        
        public PluginQuery build()
        {
            return new PluginQuery(this);
        }
        
        /**
         * Specifies text to search for in the plugin name and details.
         * @param searchText  the search string, or {@link Option#none} for no text search
         * @return  the same Builder
         */
        public Builder searchText(Option<String> searchText)
        {
            this.searchText = checkNotNull(searchText);
            return this;
        }
        
        /**
         * Restricts the query to plugins that are compatible with the specified application.
         * @param application  an {@link ApplicationKey}, or {@link Option#none} for no application filter
         * @return  the same Builder
         */
        public Builder application(Option<ApplicationKey> application)
        {
            this.application = checkNotNull(application);
            return this;
        }
        
        /**
         * Restricts the query to plugins that are compatible with the specified application version.
         * This is ignored if you have not specified {@link #application}.
         * @param appBuildNumber  the application build number, or {@link Option#none} for no application
         *   version filter
         * @return  the same Builder
         */
        public Builder appBuildNumber(Option<Long> appBuildNumber)
        {
            this.appBuildNumber = checkNotNull(appBuildNumber);
            return this;
        }
        
        /**
         * Changes the filter and sort order according to one of the options in {@link View}.
         * This is ignored if you have specified {@link #searchText}.
         * @param view  a {@link View} constant, or {@link Option#none} for the default plugins list
         * @return  the same Builder
         */
        public Builder view(Option<View> view)
        {
            this.view = checkNotNull(view);
            return this;
        }
        
        /**
         * Restricts the query to plugins that match one of the {@link Cost} options.
         * @param cost  a {@link Cost} constant, or {@link Option#none} for no cost filter
         * @return  the same Builder
         */
        public Builder cost(Option<Cost> cost)
        {
            this.cost = checkNotNull(cost);
            return this;
        }
        
        /**
         * Restricts the query to plugins that are in the specified category.  You can call this more
         * than once to include multiple categories.
         * @param category  a category name
         * @return  the same Builder
         */
        public Builder category(String category)
        {
            if (isNotBlank(category))
            {
                this.categories.add(category.trim());
            }
            return this;
        }
        
        /**
         * Specifies whether to include pricing information in the results for plugins that are
         * purchasable through the Atlassian Marketplace.  By default, none is included, meaning that
         * {@link com.atlassian.marketplace.client.model.PluginSummary#getPricing} will always
         * return {@link Option#none()}.
         * @param includePricing  a {@link PricingQuery} specifying which prices to include, or
         *   {@link Option#none}
         * @return  the same Builder
         */
        public Builder includePricing(Option<PricingQuery> includePricing)
        {
            this.includePricing = checkNotNull(includePricing);
            return this;
        }
        
        /**
         * Specifies the number of items to skip ahead in the result set.
         * @param offset  the starting item index (zero to start at the beginning)
         * @return  the same Builder
         */
        public Builder offset(int offset)
        {
            if (offset < 0)
            {
                throw new IllegalArgumentException("offset may not be negative");
            }
            this.offset = offset;
            return this;
        }
        
        /**
         * Specifies the maximum number of items to return at a time.  You may receive fewer items
         * than this if you exceed the maximum number allowed by the server, or if there aren't that
         * many items. 
         * @param limit  the maximum number of items, or {@link Option#none} to use the server's default
         * @return  the same Builder
         */
        public Builder limit(Option<Integer> limit)
        {
            for (int l: limit)
            {
                if (l < 0)
                {
                    throw new IllegalArgumentException("limit may not be negative");
                }
            }
            this.limit = limit;
            return this;
        }
    }
}
