package com.atlassian.streams.internal.feed;

import java.net.URI;

import com.atlassian.streams.api.common.Option;
import com.atlassian.streams.api.common.uri.Uri;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Multimap;

import org.joda.time.DateTime;

import static com.atlassian.streams.api.common.Option.none;
import static com.atlassian.streams.api.common.Option.some;
import static com.google.common.collect.Iterables.concat;

/**
 * Represents feed content that will be returned by the aggregator.  The properties
 * are roughly mapped to what is required by Atom, but this abstraction is meant to
 * be independent of the output format and implementation details of any {@link FeedRenderer}.
 * <p>
 * {@code FeedModel} differs from {@link com.atlassian.streams.api.StreamsFeed}
 * as follows:
 * <ul>
 * <li> It includes information that is not generated by the activity provider
 * and is not in the API, but is meaningful to the front end, such as {@link FeedHeader}s
 * and timeout/retry links. </li>
 * <li> The entries it contains do not have any lazily generated properties,
 * unlike entries in a {@code StreamsFeed}. </li>
 * <li> It can preserve format-specific raw content that was parsed by a
 * {@link FeedParser} so that it can be used unchanged if the feed is later
 * output in the same format.
 * </ul>
 */
public class FeedModel
{
    private final Uri uri;
    private final Option<String> title;
    private final Option<String> subtitle;
    private final Option<DateTime> updated;
    private final ImmutableMultimap<String, Uri> links;
    private final ImmutableList<FeedHeader> headers;
    private final ImmutableList<FeedEntry> entries;
    private final Option<Object> encodedContent;
    
    public static FeedModel.Builder builder(Uri uri)
    {
        return new Builder(uri);
    }
    
    public static FeedModel.Builder builder(URI uri)
    {
        return new Builder(Uri.fromJavaUri(uri));
    }
    
    private FeedModel(Builder builder)
    {
        this.uri = builder.uri;
        this.title = builder.title;
        this.subtitle = builder.subtitle;
        this.updated = builder.updated;
        this.links = ImmutableMultimap.copyOf(builder.links);
        this.headers = ImmutableList.copyOf(builder.headers);
        this.entries = ImmutableList.copyOf(builder.entries);
        this.encodedContent = builder.encodedContent;
    }
    
    /**
     * The base URI of the feed, which in Atom is represented as both an "id" element
     * and a link to "self".
     */
    public Uri getUri()
    {
        return uri;
    }
    
    /**
     * The feed title.
     */
    public Option<String> getTitle()
    {
        return title;
    }
    
    /**
     * The feed subtitle.
     */
    public Option<String> getSubtitle()
    {
        return subtitle;
    }
    
    /**
     * The last updated timestamp.
     */
    public Option<DateTime> getUpdated()
    {
        return updated;
    }

    /**
     * Links that are attached to the feed as a whole, rather than to an entry.
     */
    public ImmutableMultimap<String, Uri> getLinks()
    {
        return links;
    }
    
    /**
     * Extension elements that are attached to the feed as a whole, rather than to an entry.
     */
    public Iterable<FeedHeader> getHeaders()
    {
        return headers;
    }
    
    /**
     * All entries in the feed.
     */
    public Iterable<FeedEntry> getEntries()
    {
        return entries;
    }

    /**
     * Optional internal data stored by the {@link FeedParser} if this feed was
     * parsed from an external source.
     */
    public Option<Object> getEncodedContent()
    {
        return encodedContent;
    }

    /**
     * Constructs new immutable instances of {@link FeedModel}.  Not thread-safe.
     */
    public static class Builder
    {
        private Uri uri;
        private Option<String> title = none();
        private Option<String> subtitle = none();
        private Option<DateTime> updated = none();
        private Multimap<String, Uri> links = HashMultimap.create();
        private Iterable<FeedHeader> headers = ImmutableList.of();
        private Iterable<FeedEntry> entries = ImmutableList.of();
        private Option<Object> encodedContent = none();
        
        public Builder(Uri uri)
        {
            this.uri = uri;
        }
        
        /**
         * Constructs a Builder that copies the current properties of an existing
         * FeedRepresentation instance.  The original instance remains immutable.
         */
        public Builder(FeedModel from)
        {
            uri = from.getUri();
            title = from.getTitle();
            subtitle = from.getSubtitle();
            updated = from.getUpdated();
            links = HashMultimap.create(from.getLinks());
            headers = from.getHeaders();
            entries = from.getEntries();
            encodedContent = from.getEncodedContent();
        }
        
        public Builder title(Option<String> title)
        {
            this.title = title;
            return this;
        }

        public Builder subtitle(Option<String> subtitle)
        {
            this.subtitle = subtitle;
            return this;
        }
        
        public Builder updated(Option<DateTime> updated)
        {
            this.updated = updated;
            return this;
        }

        public Builder replaceLink(String rel, Uri href)
        {
            links.removeAll(rel);
            links.put(rel, href);
            return this;
        }

        public Builder addLink(String rel, Uri href)
        {
            links.put(rel, href);
            return this;
        }
        
        public Builder addLinkIfNotPresent(String rel, Uri href)
        {
            if (! links.containsKey(rel))
            {
                links.put(rel, href);
            }
            return this;
        }
        
        public Builder addHeaders(Iterable<FeedHeader> headers)
        {
            this.headers = concat(this.headers, headers);
            return this;
        }
        
        public Builder addEntries(Iterable<FeedEntry> entries)
        {
            this.entries = concat(this.entries, entries);
            return this;
        }

        public Builder encodedContent(Object encodedContent)
        {
            this.encodedContent = some(encodedContent);
            return this;
        }
        
        public FeedModel build()
        {
            return new FeedModel(this);
        }
    }
}
