define('model/file-change', [
    'backbone-brace',
    'util/deprecation',
    'model/commit-range',
    'model/conflict',
    'model/file-change-types',
    'model/path',
    'model/repository'
], function (
    Brace,
    deprecate,
    CommitRange,
    Conflict,
    ChangeType,
    Path,
    Repository
) {

    'use strict';

    var FileChange = Brace.Model.extend({
        namedAttributes : {
            "repository" : Repository,
            "commitRange" : CommitRange,
            "srcPath" : Path,
            "path" : Path,
            "line" : null,
            "search" : null,
            "type" : null,
            "nodeType" : null,
            "conflict" : Conflict,
            "diff" : null,
            "srcExecutable" : null,
            "executable" : null
        },
        initialize : function() {
            // BEGIN - remove in 4.0 when we stop deprecating Path in JSON output
            // This check avoids those nested deprecations sticking around in Brace-to-JSON-to-Brace conversion loops.
            if (this.getPath()) {
                this.setPath(new Path(this.getPath()));
            }
            if (this.getSrcPath()) {
                this.setSrcPath(new Path(this.getSrcPath()));
            }
            // END - remove in 4.0 when we stop deprecating path in JSON output

            this.setType(
                FileChange._mapChangeType(
                    this.getType(),
                    this.getSrcPath(),
                    this.getPath()
                )
            );
        }
    }, {
        fromDiff : function(diffJson, commitRange, repository) {

            // Let's work out the ChangeType for this file so we can consistently use it everywhere.
            var type;
            var firstHunk = diffJson.hunks[0];
            if (firstHunk) {
                // If we're looking at a Diff then the ChangeType will always be one of Add/Delete/Modify
                // if the file was only Moved/Renamed there would not be a diff.
                if (firstHunk.sourceLine === 0) {
                    type = ChangeType.ADD;
                } else if (firstHunk.destinationLine === 0) {
                    type = ChangeType.DELETE;
                } else {
                    type = ChangeType.MODIFY;
                }
            }

            return new FileChange({
                repository : repository,
                commitRange : commitRange,
                srcPath : diffJson.source,
                path : diffJson.destination,
                diff : diffJson,
                type: type
            });
        },
        _mapChangeType : function(modState, srcPath, path) {
            return (modState === ChangeType.MOVE && srcPath && srcPath.isSameDirectory(path)) ?
                    ChangeType.RENAME :
                    ChangeType.changeTypeFromId(modState);
        }
    });

    (function (oldToJSON) {
        FileChange.prototype.toJSON = function() {
            var json = oldToJSON.apply(this, arguments);
            if (this.getPath()) {
                json.path = deprecate.jsonAsBrace(this.getPath(), '3.1', '4.0');
            }
            if (this.getSrcPath()) {
                json.srcPath = deprecate.jsonAsBrace(this.getSrcPath(), '3.1', '4.0');
            }
            return json;
        };
    })(FileChange.prototype.toJSON);

    return FileChange;
});
