define('layout/repository', [
    'aui',
    'jquery',
    'stash/api/util/navbuilder',
    'util/events',
    'model/page-state',
    'model/repository',
    'widget/quick-copy-text',
    'widget/sidebar',
    'feature/repository/sticky-branches',
    'exports'
], function(
    AJS,
    $,
    navbuilder,
    events,
    pageState,
    Repository,
    quickCopyText,
    sidebar,
    stickyBranches,
    exports
) {

    function initRepositoryPageState(repositoryJson) {
        var repo = new Repository(repositoryJson);
        pageState.setRepository(repo);
        pageState.setProject(repo.getProject());
        // TODO: remove this. It is here for testing if people use the project avatar icon as a button.
        $('.aui-page-header-image > a').on('click', function() {
            events.trigger('stash.sidebar.clickAvatar.expanded.' + $('.aui-sidebar').attr('aria-expanded'), null);
        });
    }

    function getCloneUrlContainer() {
        return $('.clone-url');
    }

    function getCloneUrlProtocolTrigger() {
        return $('.repository-protocol');
    }

    /**
     * Input elements can't be sized to fit their contents, so we have to use a bit of javascript to do it for us.
     * This method creates a fake element to calculate the size of one monospace character and then sets the width
     * of the input element to be input.length * width. There are a few rounding issues on some browsers but should
     * be good enough for the most part.
     *
     * Finally, this method also binds events to focus/mouseup to automatically select the input text for the
     * convenience of the user who is going to want to copy the value.
     */
    function initCloneUrlInput() {
        var $container = getCloneUrlContainer(),
            $cloneInput = $container.find("input"),
            $cloneProtocolTrigger = getCloneUrlProtocolTrigger(),
            $cloneProtocolDropdown = $('#' + $cloneProtocolTrigger.attr("aria-owns")),
            $cloneProtocolDropdownItems = $cloneProtocolDropdown.find('li'),
            cloneUrl,
            moduleKey,
            oldModuleKey = '';

        if($cloneProtocolTrigger.is('button')) {
            updateCloneProtocolTrigger($cloneProtocolTrigger, $cloneProtocolTrigger.text());
            cloneUrl = $cloneProtocolTrigger.attr('data-clone-url');
            moduleKey = $cloneProtocolTrigger.attr('data-module-key');
        } else {
            updateCloneProtocolTrigger($cloneProtocolTrigger, $cloneProtocolDropdownItems.first().children('a').text());
            cloneUrl = $cloneProtocolDropdownItems.first().attr('data-clone-url');
            moduleKey = $cloneProtocolDropdownItems.first().attr('data-module-key');
        }
        $cloneInput.val(cloneUrl);
        $container.addClass(moduleKey);
        oldModuleKey = moduleKey;

        $cloneProtocolDropdownItems.on('click', function(event) {
            var $this = $(this);
            updateCloneProtocolTrigger($cloneProtocolTrigger, $this.text());
            $cloneInput.val($this.attr('data-clone-url')).select();
            moduleKey= $this.attr('data-module-key');
            $container.removeClass(oldModuleKey).addClass(moduleKey);
            oldModuleKey = moduleKey;
            events.trigger('stash.feature.repository.clone.protocol.changed', null, moduleKey, $this.attr('data-clone-url'));

            // the dropdown is outside of the inline dialog, so clicking on a dropdown item actually hides the inline
            // dialog, so we prevent that from happening...
            event.stopPropagation();
            if ($cloneProtocolDropdown.is(':visible')) { // but we still want to hide the dropdown onclick if it is visible
                $cloneProtocolTrigger.trigger("aui-button-invoke");
            }
            event.preventDefault();
        });

        events.trigger('stash.feature.repository.clone.protocol.initial', null, moduleKey, cloneUrl);

    }

    function updateCloneProtocolTrigger(trigger, newLabel) {
        var $cloneProtocolLabel = trigger.children('span').remove(); // pull the icon span element out and store it temporarily
        trigger.text(newLabel).append($cloneProtocolLabel); // replace the dropdown trigger text and add the icon span back in
    }

    function initCloneUrlDialog(cloneUrlDialogTrigger, cloneUrlDialogContent) {
        var $button = $(cloneUrlDialogTrigger);

        var $dialogContent = $(cloneUrlDialogContent);
        var dialogPos = null;
        var arrowPos = null;

        var dialog = AJS.InlineDialog($button, "clone-repo-dialog", function (content, trigger, showPopup) {
            content.append($dialogContent);
            showPopup();
            _.defer(function() {
                $dialogContent.find('.clone-url input').select();
            });
        }, {
            noBind: true,
            width: 360,
            gravity: 'w',
            hideCallback: function() { // hide the dropdown (if it is visible) when the inline dialog is hidden
                var $cloneProtocolTrigger = getCloneUrlProtocolTrigger(),
                    $cloneProtocolDropdown = $('#' + $cloneProtocolTrigger.attr("aria-owns"));

                if ($cloneProtocolDropdown.is(':visible')) {
                    $cloneProtocolTrigger.trigger("aui-button-invoke");
                }
            },
            calculatePositions: function () {
                if (dialogPos === null && arrowPos === null) {
                    // launching a standard dialog, fall back to the default implementation
                    return AJS.InlineDialog.opts.calculatePositions.apply(this, arguments);
                } else {
                    return {
                        gravity: 'w',
                        popupCss: dialogPos,
                        arrowCss: arrowPos
                    };
                }
            }
        });
        $(document).on('click' , cloneUrlDialogTrigger, function (e) {
            e.preventDefault();

            // check if the target was outside the sidebar (inside an inline dialog)
            if ($(e.target).closest('.aui-sidebar').length === 0) {
                var $dialog = $('#inline-dialog-sidebar-submenu');
                var $arrow = $dialog.find('#arrow-sidebar-submenu');
                // The absolute position of the dialog for the new dialog
                dialogPos = $dialog.offset();
                // The position of the arrow relative to the dialog for positioning in the new dialog.
                arrowPos = $arrow.position();
            } else {
                dialogPos = null;
                arrowPos = null;
            }

            dialog.show(e);
        });

        $(document).keyup(function(e) {
           if(e.keyCode === $.ui.keyCode.ESCAPE) {
               dialog.hide();
           }
        });

        $dialogContent
            .find("a.sourcetree-clone-button")
            .on('click', function () {
                dialog.hide();
          });
    }

    function bindCreatePullRequestButton() {
        var $createButton = $(".aui-page-header-actions .create-pull-request");

        events.on('stash.layout.branch.revisionRefChanged', function(revisionReference) {
            var createPullRequestBuilder = navbuilder.currentRepo().createPullRequest();
            if (!revisionReference.isDefault() && revisionReference.isBranch()) {
                createPullRequestBuilder = createPullRequestBuilder.sourceBranch(revisionReference.getId());
            }
            $createButton.attr('href', createPullRequestBuilder.build());
        });
    }

    function bindBadgesTipsy() {
        $('.repository-badge .badge').tooltip({
            gravity: 'n'
        });
    }

    // temporary @aui-override to add tipsy to the project avatar. Should be removed when this is implemented in the
    // AUI sidebar component
    function bindProjectAvatarTipsy() {
        var $trigger = $('.aui-sidebar[aria-expanded=false] .aui-page-header-image');
        $trigger.tooltip({
            gravity: 'w',
            delayIn: 0,
            live: true,
            html: true,
            aria: true,
            className: 'aui-sidebar-section-tooltip',
            title: function() {
                return $(this).find('.aui-avatar').attr('data-tooltip');
            }
       });
    }

    exports.onReady = function(repositoryJson, cloneUrlDialogTrigger, cloneUrlDialogContent) {
        initRepositoryPageState(repositoryJson);
        initCloneUrlInput();
        quickCopyText.onReady();
        sidebar.onReady();
        stickyBranches.onReady();
        initCloneUrlDialog(cloneUrlDialogTrigger, cloneUrlDialogContent);
        bindCreatePullRequestButton();
        bindBadgesTipsy();
        bindProjectAvatarTipsy();

        events.on('stash.widget.keyboard-shortcuts.register-contexts', function(keyboardShortcuts) {
            keyboardShortcuts.enableContext('repository');
        });
    };
});
