// WARNING: This module id replaces a removed older version of DiffView.
// It is a completely different module since 2.11.
// Plugin developers should not have been depending on the old one
// and should not depend on this one either.
;(function() {

// These _have_ to be registered once/globally :(
// NOTE: the sync require will fail in unit tests due to using RequireJS instead of Almond. We have to add a dep at the test level.
var baconUtil = require('util/bacon');
var requestNextHunk = baconUtil.keyboardEvents('requestSecondaryNext');
var requestPreviousHunk = baconUtil.keyboardEvents('requestSecondaryPrevious');

define('feature/file-content/diff-view', [
    'bacon',
    'codemirror',
    'jquery',
    'underscore',
    'util/array',
    'util/bacon',
    'util/events',
    'util/function',
    'util/navigator',
    'util/object',
    'util/performance',
    'util/region-scroll-forwarder',
    'util/request-page-scrolling',
    'util/scroll',
    'model/direction',
    'model/file-change',
    'feature/file-content/diff-view-options',
    'feature/file-content/diff-view-segment-types',
    'feature/file-content/ediff/ediff-markers'
],
/**
 * An abstract class used by Unified Diff View and Side-by-side Diff View for rendering code changes.
 *
 * Uses CodeMirror heavily.
 *
 * @exports feature/file-content/diff-view
 */
function(
    Bacon,
    CodeMirror,
    $,
    _,
    array,
    baconUtil,
    events,
    fn,
    navigator,
    obj,
    performance,
    RegionScrollForwarder,
    requestPageScrolling,
    scrollUtil,
    Direction,
    FileChange,
    diffViewOptions,
    diffViewSegmentTypes,
    ediffMarkers
) {
    'use strict';

    var ADDED = diffViewSegmentTypes.ADDED;
    var REMOVED = diffViewSegmentTypes.REMOVED;
    var CONTEXT = diffViewSegmentTypes.CONTEXT;

    /**
     * @typedef {object} Line
     * @property {number} destination - destination line number
     * @property {number} source - source line number
     * @property {string} lineType - CONTEXT, ADDED, REMOVED
     * @property {number} lineNumber - line number appropriate to the given line type
     * @property {string} line - The content of this line
     * @property {boolean} truncated - was this line truncated?
     */

    /**
     * Information about a line being displayed.
     *
     * @param {Line} line - the line itself
     * @param {Object} segment - the segment containing this line (matches REST output)
     * @param {Object} hunk - the hunk containing this line (matches REST output)
     * @param {Object} diff - the diff containing this line (matches REST output)
     * @param {Object} attributes - additional attributes related to this line.
     *
     * @property {Line} line - the line itself
     * @property {Object} segment - the segment containing this line (matches REST output)
     * @property {Object} hunk - the hunk containing this line (matches REST output)
     * @property {Object} diff - the diff containing this line (matches REST output)
     * @property {Object} attributes - additional attributes related to this line.
     * @property {{FROM : StashLineHandle, TO : StashLineHandle}} handles - a map of file type to line handle.
     *
     * @constructor
     */
    function LineInfo(line, segment, hunk, diff, attributes) {
        this.line = line;
        this.segment = segment;
        this.hunk = hunk;
        this.diff = diff;
        this.handles = { FROM : null, TO : null };
        this.attributes = attributes;
    }
    LineInfo.prototype._setHandle = function (fileType, handle) {
        this.handles[fileType] = handle;
    };

    /**
     * A public change object will contain some, but not all of the properties from a {@link ContentChange}
     *
     * @typedef {Object} PublicChange
     * @property {string} type - 'INITIAL' for the initial load, and 'INSERT' for expanded contexts. Other values may be added in the future for other types of change.
     * @property {Object} diff
     */

    // The property in a line that holds the line number we need, for each segment type. (e.g. ADDED segments care about the destination line number).
    var numPropForType = {
        'ADDED'   : 'destination',
        'REMOVED' : 'source',
        'CONTEXT' : 'source'
    };

    /**
     * Find the range of lines in a segment that are the "expanded" context.
     * i.e. the context lines that are not relevantContextLines
     *
     * @param {Array}  segments - array of segments to compare the
     * @param {Object} seg
     * @param {number} currentIndex
     * @param {number} relevantContextLines
     *
     * @returns {?{start: number, end: number}} the range of expanded context
     */
    function getExpandedRangeForSegment(segments, seg, currentIndex, relevantContextLines) {
        // If the current segment is a CONTEXT line, then we check the previous/next segment for being ADDED or REMOVED
        // Then we'll return an object indicating the range of lines that are "expanded" for this segment

        var prevSeg = segments[currentIndex - 1];
        var nextSeg = segments[currentIndex + 1];

        if (seg.type !== diffViewSegmentTypes.CONTEXT) {
            return null;
        }

        var start = 0;
        var end = seg.lines.length;

        // If the previous segment was an added/removed segment, set the start point
        if (prevSeg && (prevSeg.type === diffViewSegmentTypes.ADDED || prevSeg.type === diffViewSegmentTypes.REMOVED)) {
            start = relevantContextLines;
        }

        // If the next segment was an added/removed segment, set the end point
        if (nextSeg && (nextSeg.type === diffViewSegmentTypes.ADDED || nextSeg.type === diffViewSegmentTypes.REMOVED)) {
            end = end - relevantContextLines;
        }

        // don't return a range if there is overlap between the start and end points, this means that
        // all the context for this segment is relevant and should not be marked as "expanded"
        if (end > start) {
            // N.B. -1 to make it reference the array index
            return { start: start - 1, end: end };
        }

        return null;

    }

    /**
     * Will combine two arrays of segments into a single one, in file order.
     *
     * @param {Object[]} segments - The current segments list
     * @param {Object[]} newSegments - The newly added segments
     * @returns {Object[]} - The updated segments list
     */
    function mergeSegments(segments, newSegments) {
        return segments.concat(newSegments).sort(function(segA, segB) {
            var af = _.first(segA.lines);
            var bf = _.first(segB.lines);
            var al = _.last(segA.lines);
            var bl = _.last(segB.lines);
            return af.destination - bf.destination ||
                   af.source - bf.source ||
                   al.destination - bl.destination ||
                   al.source - bl.source;
        });
    }

    /**
     * Get all the segments from a diff object.
     *
     * @param {Object} diff
     * @returns {Object[]}
     * @private
     */
    function getSegmentsFromDiff(diff) {
        return _.chain(diff.hunks)
               .pluck('segments')
               .flatten(true)
               .value();
    }

    /**
     * Return whether a region contains modification lines.
     *
     * @param {Object} segment
     * @returns {boolean}
     * @private
     */
    function isModification(segment) {
        return segment && segment.type !== diffViewSegmentTypes.CONTEXT;
    }

    /**
     * Return whether a segment contains a conflict
     *
     * @param {Object} segment
     * @returns {boolean}
     * @private
     */
    function isConflicted(segment) {
        return Boolean(segment.lines[0].conflictMarker);
    }

    /**
     * Turn a diff object in to an array of line objects.
     *
     * @param {Object} diff
     * @returns {Array.LineInfo} line info array
     */
    function asLineInfos(diff, options) {
        return _.chain(diff.hunks)
            // Create an array of segments objects that contain the hunk and the segment
            .map(function(hunk) {
                return _.map(hunk.segments, function(segment, index) {
                    var expandedRange = getExpandedRangeForSegment(hunk.segments, segment, index, options.relevantContextLines);
                    return { hunk : hunk, segment : segment, expandedRange: expandedRange };
                });
            })
            .flatten()
            .map(function(hunkAndSegment) {
                var seg = hunkAndSegment.segment;
                return _.map(seg.lines, function(line, index) {

                    // Add some helper properties for add-ons to use.
                    line.lineType  = seg.type;
                    line.lineNumber = line[numPropForType[seg.type]];
                    // If this line is within the expanded range, make it as such.
                    var attributes = {
                        expanded: hunkAndSegment.expandedRange && index < hunkAndSegment.expandedRange.end && index > hunkAndSegment.expandedRange.start
                    };

                    return new LineInfo(line, seg, hunkAndSegment.hunk, diff, attributes);
                });
            })
            .flatten()
            .value();
    }


    var $lineNumber = $('<div class="line-number"></div>');
    var $lineNumberMarker = $('<div class="line-number-marker" data-file-type="" data-line-type="" data-line-number=""></div>');

    /**
     * Add line numbers to the editor's gutter
     *
     * @param {DiffView} diffView
     * @param {ContentChange} change
     */
    function addLineNumbers(diffView, change) {
        var gutterMarkerArgs = [];
        change.eachLine(function(data) {
            var line = data.line;
            var FROM = data.handles.FROM;
            var TO = data.handles.TO;

            var $fromClone = $lineNumber.clone();
            var $toClone = $lineNumber.clone();
            $fromClone.addClass('line-number-from');
            $toClone.addClass('line-number-to');
            $fromClone.html(line.lineType !== ADDED ? line.source : '&nbsp;');
            $toClone.html(line.lineType !== REMOVED ? line.destination : '&nbsp;');

            gutterMarkerArgs.push([ FROM || TO, 'line-number-from', $fromClone[0] ]);
            gutterMarkerArgs.push([ TO || FROM, 'line-number-to', $toClone[0] ]);

            _.chain([FROM, TO])
                .compact()
                .uniq()
                .forEach(function(h) {
                    var $marker = $lineNumberMarker.clone();
                    $marker.attr('data-file-type', h.fileType);
                    $marker.attr('data-line-type', line.lineType);
                    $marker.attr('data-line-number', line.lineNumber);
                    $marker.html(h.lineType === ADDED ? '+' : h.lineType === REMOVED ? '-' : '&nbsp;');

                    gutterMarkerArgs.push([ h, 'line-number-marker', $marker[0] ]);
                });
        }).done(function() {
            diffView.operation(function() {
                gutterMarkerArgs.forEach(function(args) {
                    diffView.setGutterMarker.apply(diffView, args);
                });
            });

            // fire the change event only once the lines are loaded.
            // This is necessary because we can't reliably address lines until the markers are rendered.
            var publicChange = getPublicChange(change);
            triggerPublicChange(publicChange);
            if (change.type === 'INITIAL') {
                triggerPublicLoad(publicChange);
            }
        });
    }


    var classes = {};
    classes[ADDED] = 'added';
    classes[REMOVED] = 'removed';
    classes[CONTEXT] = 'context';

    /**
     * Get CSS classes to apply to a line
     * @param {string} lineType - DiffViewSegmentType
     * @param {string} conflictMarker
     * @param {boolean} isInsert - non-standard line
     * @param {boolean} isExpanded - expanded context (e.g. non-relevant context in side-by-side)
     * @returns {string}
     */
    function getLineClasses(lineType, conflictMarker, isInsert, isExpanded) {
        var cssClass = 'line ' + classes[lineType];

        if (lineType !== CONTEXT) {
            cssClass += ' modified';
        }
        if (isExpanded === true || isInsert === true) {
            cssClass += ' expanded';
        }
        cssClass += (conflictMarker ? ' conflict conflict-' + conflictMarker.toLowerCase() : '');
        // Also add a 'new' class when we're expanding context
        cssClass += isInsert ? ' new' : '';
        return cssClass;
    }

    /**
     * Add the Diff classes to the editor based on the given diff information
     *
     * @param {DiffView} diffView
     * @param {ContentChange} change
     */
    function addDiffClasses(diffView, change) {
        var isInsert = change.type === 'INSERT';

        var affectedLines = [];
        change.eachLine(function(lineData) {
            var classes = getLineClasses(lineData.line.lineType, lineData.line.conflictMarker, isInsert, lineData.attributes.expanded);
            var FROM = lineData.handles.FROM;
            var TO = lineData.handles.TO;

            if (FROM) {
                diffView.addLineClass(FROM, 'wrap', classes);
                if (isInsert) {
                    affectedLines.push(FROM);
                }
            }
            if (TO && TO !== FROM) {
                diffView.addLineClass(TO, 'wrap', classes);
                if (isInsert) {
                    affectedLines.push(TO);
                }
            }

        }).done(function() {
            // We've been hanging on to the line handles so we can use them to remove the new class after a timeout.
            if (affectedLines.length) {
                removeNewClass(diffView, affectedLines);
            }
        });
    }

    /**
     * Remove the 'new' class from freshly inserted lines
     *
     * @param {DiffView} diffView
     * @param {Array.LineHandle} lines - an array of CodeMirror Line Handles
     */
    function removeNewClass(diffView, lines) {
        setTimeout(function() {
            if (!diffView._editor) { // we were destroyed in the meantime
                return;
            }
            diffView.operation(function() {
                _.each(lines, function(line, index){
                    diffView.removeLineClass(line, 'wrap', 'new');
                });
            });
        }, 1500);
    }

    /**
     * Add a class to the container signifying that the diff view is ready for programmatic access.
     * Used by func tests.
     * @param diffView
     */
    function addApiReadyClass(diffView) {
        diffView.$container.addClass('diff-api-ready');
    }

    /**
     * Get a public change object for editor change/load events
     *
     * @param {ChangeObject} change
     * @returns {PublicChange}
     */
    function getPublicChange(change) {
        var clone = $.extend({}, change);
        delete clone.pullRequest;
        delete clone.fileChange;
        return obj.freeze(clone);
    }

    /**
     * Trigger a public change event
     *
     * @param {PublicChange} change
     */
    function triggerPublicChange(change) {
        _.defer(_.bind(events.trigger, events, 'stash.feature.fileContent.diffViewContentChanged', null, change));
    }

    /**
     * Trigger a public load event
     *
     * @param {PublicChange} change
     */
    function triggerPublicLoad(change) {
        _.defer(_.bind(events.trigger, events, 'stash.feature.fileContent.diffViewContentLoaded', null, change));
    }

    /**
     * Create the CodeMirror instance for the given source container.
     *
     * @param {HTMLElement} containerEl
     * @param {Object} [options]
     * @returns {CodeMirror}
     */
    function createEditor(containerEl, options) {
        var editor = new CodeMirror(containerEl, $.extend({
            mode: 'text/plain',
            readOnly: true,
            lineNumbers: false, // we do this ourselves
            wholeLineUpdateBefore: false,
            cursorBlinkRate: 0,
            styleSelectedText: true
        }, options));
        blurEditorOnArrowKeys(editor);
        allowEditorKeysPassThrough(editor);
        clearSelectionOnEditorBlur(editor);
        return editor;
    }

    /**
     * Set up an event handler to handle keydown events on the editor that can then be passed through to the
     * document for regular handling. Our events don't fire because the CodeMirror events take place
     * in a textarea and the Stash keyboard shortcut handler ignores these events.
     *
     * @param {CodeMirror} editor
     */
    function allowEditorKeysPassThrough(editor) {
        editor.on('keydown', keyPassThroughHandler);
        editor.on('keypress', keyPassThroughHandler);
    }

    // Set up the keys that CodeMirror should ignore for us in ReadOnly mode.
    var disAllowedKeys = {};

    _.forEach([
        AJS.keyCode.TAB
    ], function(key) {
        disAllowedKeys[key] = true;
    });

    /**
     * Filters the key code on the event to see if it is in the disAllowed keys list.
     * @param {Event} e
     * @returns {boolean}
     */
    function disAllowedEditorKeys(e){
        var key = e.which || e.keyCode;
        return disAllowedKeys[key];
    }

    /**
     * Handle the editor keydown/keypress event and pass it through to the document
     * @param {CodeMirror} editor
     * @param {Event} e
     */
    function keyPassThroughHandler(editor, e) {

        var attributesToCopy = ['which', 'keyCode', 'shiftKey', 'ctrlKey', 'metaKey'];
        var passThroughEvent = jQuery.Event(e.type);

        _.forEach(attributesToCopy, function(attr) {
            passThroughEvent[attr] = e[attr];
        });

        //pass the event along to the document
        $(document).trigger(passThroughEvent);

        // Check if the key that was pressed is in the disAllowed list. If it is, then we will set the
        // codemirrorIgnore property on the event so that CodeMirror does not handle this key event
        // and perhaps more importantly, does not swallow the event.
        e.codemirrorIgnore = disAllowedEditorKeys(e);

    }

    /**
     * Blur the editor when an arrow key is pressed inside and it is not extending a selection.
     *
     * When CodeMirror has focus, you can use the arrow keys to navigate the diff. This is largely because we're
     * using a readonly mode for CodeMirror and there is no cursor (so it's not clear that your cursor is in fact
     * focused in the editor).
     *
     * This solution isn't bulletproof. Because we're monitoring the keydown event, we need at least 1 event to fire
     * to blur away from the textarea before the keydown event will scroll the page/diff again.
     *
     * @param {CodeMirror} editor
     */
    function blurEditorOnArrowKeys(editor) {

        editor.on('keydown', function(editor, e){
            // If this is an arrow key and the Shift key was NOT pressed down then we want to blur the editor.
            // 37 - 40 are keyCodes for arrow keys.
            // We also check that there isn't anything selected and that the shift key isn't pressed
            // as this would be used when creating/extending a selection.
            if (e.which >= 37 && e.which <= 40 && !e.shiftKey && !editor.somethingSelected()) {
                editor.getInputField().blur();
            }
        });
    }

    /**
     * When the editor is blurred we'll want to clear any selections that might be present.
     * @param {CodeMirror} editor
     */
    function clearSelectionOnEditorBlur(editor) {
        var isContextMenuBlur = false;

        // This is a debounced function so that the editor has a chance to fire the
        // contextmenu event which we'll want to exclude as a blurrer (so that users can right-click and copy text)
        var clearSelection = _.debounce(function() {
            var firstVisibleLine;
            if (!isContextMenuBlur) {
                firstVisibleLine = editor.lineAtHeight(editor.getScrollInfo().top + editor.heightAtLine(0));
                // We add +1 to the firstVisibleLine so CM won't scroll the diff up by a few px if we're in between lines.
                // i.e. you're scrolled between lines 12 and 13, it will unset the selection on line 14 so the diff
                // doesn't scroll by half a line
                editor.setSelection({line: firstVisibleLine + 1, ch:0});
            }
            isContextMenuBlur = false;
        }, 10);

        editor.on('contextmenu', function() {
            isContextMenuBlur = true;
        });

        editor.on('blur', function(editor) {
            if (editor.somethingSelected()) {
                clearSelection();
            }
            isContextMenuBlur = false;
        });

    }

    var apiMethods = ['getLineHandle', 'operation', 'getLine', 'addLineClass', 'removeLineClass', 'setGutterMarker', 'registerGutter', 'addLineWidget'];

    var defaults = {
        // focusPoint indicates the point - expressed as a fraction of the viewport - to which we will scroll things
        // when bringing them in to focus.
        focusPoint: 0.2
    };

    /**
     * Abstract class for viewing diffs.
     *
     * @param {Object} data
     * @param {Object} options
     * @constructor
     */
    function DiffView(data, options) {
        this._data = data;
        this.options = _.extend({}, defaults, options);
        this.$container = options.$container;

        this.gutters = [];

        this._internalLines = {
            CONTEXT : {},
            ADDED : {},
            REMOVED : {}
        };

        this.options.fileChange = new FileChange(this.options.fileChange);

        var apiArgs = [this].concat(apiMethods);
        _.bindAll.apply(_, apiArgs);
        this._api = _.pick.apply(_, apiArgs);
    }

    /**
     * Extend {DiffView} with event mixins
     */
    events.addLocalEventMixin(DiffView.prototype);

    /**
     * How content has been changed
     * @enum {string} ContentChangeType
     */
    DiffView.contentChangeType = {
        /** This is the initial load of the content. */
        INITIAL : 'INITIAL',
        /** New lines are being inserted into the content. */
        INSERT : 'INSERT'
    };

    /**
     * Initialize the DiffView
     */
    DiffView.prototype.init = function() {
        this._pr = this.options.fileChange.getCommitRange().getPullRequest();

        var dvOptions = this.options.diffViewOptions || diffViewOptions;

        this.$container.addClass('diff-type-' + this.options.fileChange.getType())
            .toggleClass('hide-ediff', dvOptions.get('hideEdiff'))
            .toggleClass('animated', !navigator.isIE() || navigator.majorVersion() >= 11);

        this._$fileToolbar = this.$container.siblings('.file-toolbar');
        this._$firstEditor = this.$container.children('.diff-editor').first();

        var diff = this._data;
        var self = this;

        events.trigger('stash.feature.fileContent.diffViewDataLoaded', null, this._data);

        // Set us up some event handlers for all the things.
        this.on('change', _.partial(addLineNumbers, this));

        this._destroyables = [];

        this._destroyables.push(ediffMarkers.init({ diffView: this }));
        this._destroyables.push(events.chainWith($(window)).on('scroll', function() {
            if (!self._expectFocusScroll) { // if we weren't expecting the window to scroll
                // blur our tracked focus
                self._invalidateFocus();
            }
            // always stop expecting a scroll
            self._expectFocusScroll = false;
        }));
        this._destroyables.push(events.chainWith(dvOptions).on('change', function(change) {
            if (change.key === 'hideEdiff') {
                self.$container.toggleClass('hide-ediff', change.value);
            }
        }));

        this._modifyDiff(diff, 'INITIAL').done(function() {
            // Only select the line after the scroll behaviour has been attached
            self._selectLine(self.options.fileChange.getLine());
            var search = self.options.fileChange.getSearch();
            // Don't do anything if no search specified (only on load though - otherwise we might want to clear)
            if (search) self._highlight(search);
        });

        this._destroyables.push(events.chain()
                .on('stash.feature.fileContent.diffViewContentLoaded', addApiReadyClass.bind(null, this))
                .on('internal.stash.feature.diffView.lineChange', this._selectLine.bind(this))
                .on('internal.stash.feature.diffView.highlightSearch', this._highlight.bind(this))
        );

        // Set up events that cause a resize of the diff area as bacon events and merge them in to a single Bacon.EventStream
        var resizeEvents = [
            'sidebar.expandEnd',
            'sidebar.collapseEnd',
            'changeset.difftree.collapseAnimationFinished'
        ];

        this.resizeEventStream = resizeEvents
            .reduce(function(events, event) {
                return events.merge(baconUtil.events('stash.feature.' + event));
            }, Bacon.fromArray([]));

        this.$container.addClass('fully-loaded');

        this._destroyables.push({destroy: requestNextHunk.onValue(this._scrollToChange.bind(this, Direction.DOWN))});
        this._destroyables.push({destroy: requestPreviousHunk.onValue(this._scrollToChange.bind(this, Direction.UP))});
    };

    /**
     * Destroy some things to help with GC
     */
    DiffView.prototype.destroy = function() {
        this.trigger('destroy');
        _.invoke(this._destroyables, 'destroy');
        this._destroyables = null;
        this._editor = null;
    };

    function abstractMethod() {
        throw new Error("DiffView implementation must define this.");
    }

    /**
     * Add a widget to the specified line
     *
     * @abstract
     * @function
     * @param {StashLineHandle} lineHandle - as returned from {@link getLineHandle}
     * @param {HTMLElement} el - the root element of the line widget
     * @param {Object} options - any options accepted by CodeMirror's equivalent method.
     * @returns {LineWidget} the return value of CodeMirror's equivalent method.
     */
    DiffView.prototype.addLineWidget = function (lineHandle, el, options) {
        var widget = this._editorForHandle(lineHandle).addLineWidget(lineHandle._handle, el, options);
        var self = this;
        self.trigger('widgetAdded');
        return {
            clear : function() {
                widget.clear();
                self.trigger('widgetCleared');
            },
            changed : function() {
                widget.changed();
                self.trigger('widgetChanged');
            }
        };
    };

    /**
     * Set gutter element for the specified gutter at the specified line.
     *
     * @abstract
     * @function
     * @param {StashLineHandle} lineHandle - as returned from {@link getLineHandle}
     * @param {string} gutterId - ID of the gutter for which to set a marker
     * @param {HTMLElement} el - element to set the gutter to.
     * @returns {StashLineHandle}
     */
    DiffView.prototype.setGutterMarker = abstractMethod;

    /**
     * Prepare the gutter marker element before it is added to the editors.
     *
     * @param {HTMLElement} el
     * @returns {HTMLElement}
     * @private
     */
    DiffView.prototype._prepareGutterMarkerElement = function(el) {
        // add a generic class
        var stashGutterClass = 'stash-gutter-marker';
        if (el.className.indexOf(stashGutterClass) === -1) {
            el.className += ' ' + stashGutterClass;
        }
        return el;
    };

    /**
     * Add a CSS class to a specified line
     *
     * @abstract
     * @function
     * @param {StashLineHandle} lineHandle - as returned from {@link getLineHandle}
     * @param {string} whichEl - 'wrap', 'background', or 'text' to specify which element to place the class on
     * @param {string} className - the class to add.
     * @returns {StashLineHandle}
     */
    DiffView.prototype.addLineClass = abstractMethod;

    /**
     * Remove a CSS class from a specified line
     *
     * @abstract
     * @function
     * @param {StashLineHandle} lineHandle - as returned from {@link getLineHandle}
     * @param {string} whichEl - 'wrap', 'background', or 'text' to specify which element to remove the class from
     * @param {string} className - the class to remove.
     * @returns {StashLineHandle}
     */
    DiffView.prototype.removeLineClass = abstractMethod;

    /**
     * Return the text on the line with the given line handle.
     *
     * @abstract
     * @function
     * @param {StashLineHandle} lineHandle - as returned from {@link getLineHandle}
     * @returns {string}
     */
    DiffView.prototype.getLine = abstractMethod;

    /**
     * Return the editor for a particular handle.
     *
     * @abstract
     * @function
     * @param {StashLineHandle} lineHandle - as returned from {@link getLineHandle}
     * @returns {CodeMirror} editor that is responsible for the given line
     * @protected
     */
    DiffView.prototype._editorForHandle = abstractMethod;

    /**
     * Return an array of all the editors for this view, mostly likely required to invoke a method on each instance.
     *
     * @abstract
     * @function
     * @returns {CodeMirror[]} all of the available editors
     * @protected
     */
    DiffView.prototype._getEditors = abstractMethod;

    /**
     * Register a gutter to be added tothe diff view
     * @abstract
     * @function
     * @param {string} name - The name of the gutter to register
     * @param {object} options
     * @param {number} [options.weight=0] - The weight of the gutter. This will determine where in the stack of gutters it will appear.
     */
    DiffView.prototype.registerGutter = abstractMethod;

    /**
     * Sort the gutters by weight
     *
     * @returns {Array<object>}
     */
    DiffView.prototype.sortGutters = function() {
        return this.gutters.sort(function(a, b) {
            return a.weight - b.weight;
        });
    };

    /**
     * Get unique gutters by given properties
     * @abstract
     * @returns {Array<object>}
     */
    DiffView.prototype.getGutters = function(props) {
        this.sortGutters();
        return obj.uniqueFromArray(this.gutters, props);
    };


    /**
     * Register a gutter
     * We explicitly register gutters to avoid adding duplicate gutters.
     *
     * @param {object} gutter
     * @param {string} gutter.name
     * @param {number} [gutter.weight=0]
     * @param {DiffFileType} [gutter.fileType]
     * @private
     *
     * @returns {Array<object>}
     */
    DiffView.prototype._registerGutter = function(gutter) {
        gutter.weight = gutter.weight || 0;
        // Add the new gutter
        this.gutters.push(gutter);
        this.gutters = this.getGutters();
        return this.gutters;
    };

    /**
     * Mark text on lines in the editor
     *
     * @abstract
     * @function
     * @param {LineInfo} line line to mark text on
     * @param {{lineOffset: number, ch: number}} from
     * @param {{lineOffset: number, ch: number}} to
     * @param {{className: string}} options
     * @returns {CodeMirror.TextMarker}
     */
    DiffView.prototype.markText = function(line, from, to, options) {
        var lineHandle = line.handles.FROM || line.handles.TO;
        var editor = this._editorForHandle(lineHandle);
        var lineIndex = editor.getLineNumber(lineHandle._handle);
        return editor.markText(
            {line: lineIndex + from.lineOffset, ch: from.ch},
            {line: lineIndex + to.lineOffset, ch: to.ch},
            {className: options.className}
        );
    };

    /**
     * Find the next segment in the list of segments for the current diff.
     *
     * @private
     *
     * @param {Object} focusSegment - the segment that currently has visual focus
     * @param {Object[]} segments - All the segments for the current diff
     * @param {Direction} direction - the direction in which we want to find the next segment
     * @returns {Object} new focus segment
     */
    DiffView.prototype._findNextChange = function(focusSegment, segments, direction) {
        var self = this;
        var foundCurrentFocus = false;

        var nextChangeIndex = -1;

        // if nothing has focus, we start from the top
        if (focusSegment == null) {
            direction = Direction.DOWN;
            foundCurrentFocus = true;
        }

        for (var n = 0, len = segments.length; n < len; n++) {
            var i = (direction === Direction.DOWN) ? n : len - n - 1;
            var segment = segments[i];
            if (foundCurrentFocus) {
                var prevSegment = segments[i - 1];
                       // only mods are focusable
                var isNextChange = isModification(segment) &&
                       // conflicted segments are combined, so we only care about the first.
                       !(isConflicted(segment) && prevSegment && isConflicted(prevSegment)) &&
                       // if we're combining mods, we only care about the first
                       !(self.options.combineLinkedSegments && prevSegment && isModification(prevSegment));

                if (isNextChange) {
                    nextChangeIndex = i;
                    break;
                }
            } else if(segment === focusSegment) {
                foundCurrentFocus = true;
            }
        }

        // should a segment be included in the list of focus segments (assuming the previous segment was included.
        function shouldAppendToFocus(segment) {
            // always include sibling conflicts
            // include sibling modifications only if we're combining segments
            return isConflicted(segment) || (self.options.combineLinkedSegments && isModification(segment));
        }

        if (nextChangeIndex > -1) {
            // we found it! Now we collect all the following changes that will be included in the focus
            var startInclusive = nextChangeIndex;
            var endExclusive = nextChangeIndex + 1;

            while(endExclusive < segments.length && shouldAppendToFocus(segments[endExclusive])) {
                endExclusive++;
            }
            return segments.slice(startInclusive, endExclusive);
        }

        return null;
    };

    /**
     * Focus a line based on a line number and its type.
     * @param {{no: number, type: string}=} line - type is either 'FROM' or 'TO'
     */
    DiffView.prototype._selectLine = function(line) {
        if (!line) return;
        var lineHandle = this.getLineHandleFromNumber(line.no, line.type);
        if (lineHandle) {
            this.scrollHandleIntoFocus(lineHandle);
            this._markLinesFocused([{editor: this._editorForHandle(lineHandle), handles: [lineHandle]}]);
        }
    };

    /**
     * Highlight all of the strings that match 'search'
     * @param {string} search - term to highlight
     * @private
     */
    DiffView.prototype._highlight = function(search) {
        this._getEditors().forEach(function(editor) {
            // Can't use editor.execCommand() because for some stupid reason you can't pass arguments...
            CodeMirror.commands.highlight(editor, search);
        });
    };

    /**
     * Scroll the diff to the next or previous change segment, given a direction.
     * @param {Direction} direction
     * @protected
     */
    DiffView.prototype._scrollToChange = function(direction) {
        var currFocus = this._focusedSegment;
        if (!currFocus) {
            var expectedDiffViewOffset = this._$fileToolbar.outerHeight(); // the toolbar stays on screen.
            var extraOffset = this.$container[0].getBoundingClientRect().top - expectedDiffViewOffset;
            currFocus = this._getFocusSegment(extraOffset);
        }
        var newFocusSegments = this._findNextChange(currFocus, this._allSegments, direction);

        if (newFocusSegments) {
            this._setFocusSegment(newFocusSegments);
            this._expectFocusScroll = true;
            this._focusedSegment = newFocusSegments[0];
        }
    };

    /**
     * Scroll to the next comment in a given direction from the current focus
     *
     * @param {Direction} direction
     * @private
     */
    DiffView.prototype._scrollToComment = function(direction) {
        // make sure that comments are not hidden when navigating them.
        diffViewOptions.set('hideComments', false);

        var anchors = _.chain(this.options.commentContext._containers)
            .pluck('anchor')
            .uniq(fn.invoke('getId'))
            .value();
        var nextAnchorInfo = this._findNextAnchor(direction, anchors, this._focusedComment);

        if (!nextAnchorInfo) {
            return;
        }

        this._expectFocusScroll = true;
        this._focusedComment = nextAnchorInfo;

        // scroll to the first file comment
        if (!nextAnchorInfo.handle) {
            this._scrollToSourcePosition(null, -this._editorInnerOffset());
            return;
        }

        this._highlightCommentForLine(nextAnchorInfo.handle);
        this.scrollHandleIntoFocus(nextAnchorInfo.handle);
    };

    DiffView.prototype._invalidateFocus = function() {
        this._focusedSegment = null;
        this._focusedComment = null;
    };

    var isFileAnchor = _.compose(fn.eq(undefined), fn.dot('_line'));
    function getAnchorInfo(editor, lineLookup, anchor) {
        var handle;
        if (!isFileAnchor(anchor)) {
            var handles = lineLookup[anchor._lineType][anchor._line].handles;
            handle = handles[anchor._fileType] || handles.FROM || handles.TO;
        }
        return {
            anchor : anchor,
            handle : handle,
            offset : handle ? editor.heightAtLine(handle._handle.lineNo(), 'local') : 0
        };
    }

    var dotLine = fn.dot('_line');
    var dotOffset = fn.dot('offset');

    /**
     * Used by the _findNextAnchor implementations
     *
     * @param {CodeMirror} editor - the editor containing anchors
     * @param {Array<Object>} anchors - a list of unique comment anchors within the given editor
     * @param {Direction} direction - which direction to search
     * @param {?{anchor : Object, handle ?Object, offset : number}} focusedAnchorInfo - info about the anchor that is currently focused
     * @returns {?{anchor : Object, handle ?Object, offset : number}}
     * @private
     */
    DiffView.prototype._findNextAnchorInEditor = function(editor, anchors, direction, focusedAnchorInfo) {
        if (anchors.length === 0) {
            return null;
        }

        anchors = _.sortBy(anchors, dotLine);

        // If we have a focus anchor in the list, we don't have to get pixel offsets for the anchors,
        // we only have to order them by relative index - perf optimization
        if (focusedAnchorInfo) {
            var focusIndex = array.findIndex(fn.propEqual(_.pick(focusedAnchorInfo.anchor, '_line', '_lineType', '_fileType')))(anchors);
            if (focusIndex !== -1) {
                // if moving up, we want the one before our current anchor.
                // if moving down, we want the one after our current anchor.
                var nextAnchor = anchors[focusIndex + (direction === Direction.UP ? -1 : 1)];
                return nextAnchor && getAnchorInfo(editor, this._internalLines, nextAnchor);
            }
        }

        // we don't have an anchor or the anchor isn't in our list, so we have to use offsets
        var currentFocusOffset;
        if (focusedAnchorInfo) {
            currentFocusOffset = focusedAnchorInfo.offset;
        } else {
            var expectedDiffViewOffset = this._$fileToolbar.outerHeight(); // the toolbar stays on screen.
            var extraOffset = this.$container[0].getBoundingClientRect().top - expectedDiffViewOffset;
            var scrollInfo = editor.getScrollInfo();
            currentFocusOffset = scrollInfo.top + (this.options.focusPoint * scrollInfo.clientHeight) - extraOffset;
        }

        var infos = anchors.map(getAnchorInfo.bind(null, editor, this._internalLines));

        if (direction === Direction.UP) {
            return _.find(infos.reverse(), function(info) {
                return info.offset < currentFocusOffset;
            });
        } else {
            return _.find(infos, function(info) {
                return info.offset > currentFocusOffset;
            });
        }
    };

    /**
     * Highlight a comment inside of a line for the given line handle
     * @param {StashLineHandle} stashLineHandle
     * @private
     */
    DiffView.prototype._highlightCommentForLine = function(stashLineHandle) {
        var commentNavAnimationDuration = 1000; // see @commentNavAnimationDuration in diff-view-animations.less
        this._addAnimationLineClass('line-comment-focused', commentNavAnimationDuration, [{editor:this._editorForHandle(stashLineHandle), handles: [stashLineHandle]}]);
    };

    /**
     * If something is done to affect the size of this diff view, refresh() can be called to force a re-rendering of it.
     *
     * @abstract
     * @function
     */
    DiffView.prototype.refresh = abstractMethod;

    /**
     * @typedef {Object} DiffLineLocator
     *
     * @property fileType
     * @property lineType
     * @property lineNumber
     */

    /**
     * Retrieve a handle for a given line identified by a DOM element element or {@link DiffLineLocator}.
     *
     * If you pass in a DOM element or jQuery object, the handle returned will be for
     * the line that element is contained within.
     *
     * If you pass in a string, we expect it to be a line type, and for the second parameter to be a line number. Together these
     * parameters identify a line in one of the files.
     *
     * @param {HTMLElement|jQuery|DiffLineLocator} locator - a DOM element inside one of the lines in this diff, or an object with locator properties
     * @returns {StashLineHandle} an object describing the line that can be used to interact with the diff.
     */
    DiffView.prototype.getLineHandle = function(locator) {

        if(locator && !locator.lineType) {
            var $lineNumbers = $(locator).closest('.line').find('.line-number-marker');
            locator = {
                fileType : $lineNumbers.attr('data-file-type'),
                lineType : $lineNumbers.attr('data-line-type'),
                lineNumber : $lineNumbers.attr('data-line-number')
            };
        }

        // This check might seem excessive, but in the event where a comment was made and the whitespace ignore option
        // changed, then the lineType may no longer be correct for this comment.
        // @TODO: Find a nicer way to solve comments + ignoreWhitespace
        var handles = locator &&
                      this._internalLines[locator.lineType][locator.lineNumber] &&
                      this._internalLines[locator.lineType][locator.lineNumber].handles;

        return handles && (handles[locator.fileType] || handles.FROM || handles.TO);
    };

    /**
     * Retrieve a handle for a given line number and its type.
     *
     * @param {number} lineNumber
     * @param {string} fileType - either 'FROM' or 'TO'
     * @returns {StashLineHandle} an object describing the line that can be used to interact with the diff.
     */
    DiffView.prototype.getLineHandleFromNumber = function(lineNumber, fileType) {
        // Unfortunately if the line is context then we have to find it manually - the index of _internalLines is based on the source
        function find(lines) {
            return _.find(lines, _.compose(fn.eq(lineNumber), fn.dot(fileType === 'TO' ? 'line.destination' : 'line.source')));
        }
        var handles = fileType === 'TO' ? this._internalLines[ADDED][lineNumber] : this._internalLines[REMOVED][lineNumber];
        // Search for context last, which could be on either side
        handles = handles || find(this._internalLines[CONTEXT]);
        handles = handles && handles.handles;
        return handles && (handles[fileType] || handles.FROM || handles.TO);
    };

    /**
     * Scroll the editor to a particular handle. Useful to scroll a line in to view when adding line widgets.
     *
     * We actually scroll the line below the targeted line in to view to ensure that the targeted
     * line is fully visible, including any widgets that may be part of the line.
     *
     * @param {StashLineHandle} handle - as returned from {@link getLineHandle}
     */
    DiffView.prototype.scrollHandleIntoView = function(handle) {
        var editor = this._editorForHandle(handle);

        // We check if the handle is the last line of the editor and ensure the entire line is
        // visible by requesting the scroll position for the bottom of the last line.
        if (editor.lastLine() === handle._handle.lineNo()) {
            editor.scrollTo(null, editor.heightAtLine(editor.lastLine()+1));
            return;
        }

        editor.scrollIntoView(handle._handle.lineNo()+1);
    };

    /**
     * Scroll a line handle in to the focus area
     * @param {StashLineHandle} handle
     */
    DiffView.prototype.scrollHandleIntoFocus = function(handle) {
        var editor = this._editorForHandle(handle);
        var editorScrollInfo = editor.getScrollInfo();
        var diffCanScroll = editorScrollInfo.height > editorScrollInfo.clientHeight;
        var scrollY;  // the editor-relative scroll position to scroll to.

        if (diffCanScroll) {
            var offset = editorScrollInfo.clientHeight * this.options.focusPoint;
            var linePos = editor.heightAtLine(handle._handle.lineNo(), 'local');
            scrollY = Math.max(0, linePos - offset);

            // Force the editor to blur to avoid scrolling issues.
            // If the editor has focus and the point where it has focus (i.e. the line that was clicked)
            // has moved beyond the top/bottom of the screen CodeMirror will try to keep it in focus on keypresses.
            // We only have to do this if we're going to actually scroll within the diff (hence the diffCanScroll check)
            editor.getInputField().blur();

            // use the editor for scrolling when possible.
            // but only if we're scrolling past 0 - otherwise CodeMirror will not trigger a scroll event.
            if (scrollY !== 0) {
                editor.scrollTo(null, scrollY);
                return;
            }
        } else {
            scrollY = 0;
        }

        this._scrollToSourcePosition(null, scrollY);
    };

    /**
     * Scrolls the window back to the top.
     */
    DiffView.prototype.scrollToTop = function () {
        this._scrollToSourcePosition(null, 0);
    };

    /**
     * Returns a public API for interacting with this diff view.
     *
     * @returns {DiffViewApi}
     */
    DiffView.prototype.api = function() {
        return this._api;
    };

    /**
     * Will be called when a request to modify the diff is received (e.g. during init() or when context is expanded).
     *
     * MUST inject/remove text in CodeMirror editor(s). Can assume you're being called within an operation().
     * MUST make successive calls to {@link LineInfo:_setHandle} for each new line. CONTEXT lines should call _setHandle twice, once for each fileType ('FROM' or 'TO')
     *
     * @abstract
     * @function
     * @param {Object} diff
     * @param {Object[]} lines
     * @protected
     */
    DiffView.prototype._acceptModification = abstractMethod;

    /**
     * Get the vertical offset between the root diff-view container and the CodeMirror editors inside.
     * This is useful for various scrolling calculations.
     * @returns {number} px between top of .diff-view and top of .diff-editor elements
     */
    DiffView.prototype._editorInnerOffset = function editorInnerOffset() {
        return this._$firstEditor[0].getBoundingClientRect().top - this.$container[0].getBoundingClientRect().top;
    };

    /**
     * Request scrolling from the page level be forwarded down to the diff view.
     * Scrolling of the file comments will be handled here. The subclass must handle any
     * forwarded scrolling through the editorScrolling object.
     * @param {{ onInternalScroll : function, resize : function, scroll : function, scrollSizing : function}} editorScrolling - an interface for accepting scroll events and propagating them within CodeMirror.
     * @private
     */
    DiffView.prototype._requestWindowScrolls = function(editorScrolling) {
        var self = this;
        return requestPageScrolling().done(function (scrollControl) {
            var scrollBus = new Bacon.Bus();

            var $container = self.$container.addClass('full-window-scrolling');
            var $fileContent = $container.closest('.file-content');

            // clientHeight as seen by layout - includes file header and editor height (but not file comment height)
            var clientHeight;

            // a function to translate the container to mimic scrolling when scroll events are forwarded.
            var scrollContainer = scrollUtil.fakeScroll($container[0]);


            // set up a combined size property we can watch that is a merged stream of the
            // window size property and a "resizeEventStream" that will receive values when we want to
            // trigger a resize of the diff based on current window size.
            // Notably this is triggered when the sidebar/difftree is expanded/collapsed as this may
            // affect vertical height of items above the diff view.
            var sizeProp = self.resizeEventStream.merge(baconUtil.getWindowSizeProperty().toEventStream()).toProperty();

            /**
             * Check if an object is a size object - we just check that the shape contains a width and height
             * @param {*} obj
             * @returns {boolean}
             */
            function isSizeObject(obj) {
                return obj && obj.hasOwnProperty('width') && obj.hasOwnProperty('height');
            }

            self._destroyables.push(scrollControl);
            self._destroyables.push({
                destroy : sizeProp
                    .scan(0, function(cachedSize, size) {
                        // make sure the size passed in to the stream is a size
                        // and has a value otherwise fall back to the cached size
                        return (isSizeObject(size) && size) || cachedSize;
                    })
                    .filter(_.identity)
                    .onValue(function(size) {
                        // we leave file comments out of the height - we'll translate it up ourselves
                        // and then the editor should be full screen once we reach the bottom of it.
                        clientHeight = size.height - self._$fileToolbar.outerHeight();
                        editorScrolling.resize(size.width, clientHeight);
                    })
            });

            // We debounce to avoid duplicate events causing double-firing. It also ensures
            // that we call forwardeeResized _after_ a 'change' event.
            var debouncedRefresh = _.debounce(function() {
                scrollControl.refresh();
            }, 10);
            self.on('widgetAdded', debouncedRefresh);
            self.on('widgetChanged', debouncedRefresh);
            self.on('widgetCleared', debouncedRefresh);
            self.on('change', debouncedRefresh);
            if (self.options.commentContext) {
                self.options.commentContext.on('fileCommentsResized', debouncedRefresh);
            }

            self._scrollToSourcePosition = function(x, y) {
                // translate from editor coords into layout-forwardee coords.
                scrollControl.scroll(x, y != null ? y + self._editorInnerOffset() : null);
            };

            // forward the editor's internal scrolls up to the page
            editorScrolling.onInternalScroll(self._scrollToSourcePosition);

            editorScrolling.onSizeChange(debouncedRefresh);

            // split any incoming scrolls from the window - scroll either the file comments or the editors.
            // first scroll the file comments, then pass it off to the SBS or unified editors.
            var scrollForwarder = new RegionScrollForwarder(scrollBus, [{
                id : 'file-comments-and-messages',
                getHeight : function() {
                    return self._editorInnerOffset() || 0;
                },
                setScrollTop : function(y) {
                    scrollContainer(0, y);
                }
            },
            {
                id : 'editors',
                getHeight : function() {
                    // the layout will only send relevant scrolls to us,
                    // so the last item can be Infinity with no consequences.
                    // In actuality, the height is editorScrollHeight - editorClientHeight
                    return Infinity;
                },
                setScrollTop : function(y) {
                    editorScrolling.scroll(null, y);
                }
            }]);

            self._destroyables.push(scrollForwarder);

            if (self.options.commentContext) {
                self.options.commentContext.on('fileCommentsResized', scrollForwarder.heightsChanged.bind(scrollForwarder));
            }

            scrollControl.setTarget({
                scrollSizing : function() {
                    var editorScrollInfo = editorScrolling.scrollSizing();
                    return {
                        height : editorScrollInfo.height + self._editorInnerOffset(),
                        clientHeight: editorScrollInfo.clientHeight
                    };
                },
                offset : function() {
                    return $fileContent.offset();
                },
                scroll : function(x, y) {
                    if (y != null) { // ignore horizontal changes
                        scrollBus.push({
                            top : y
                        });
                    }
                }
            });

            // This section of code is dealing with issues where diagonal scrolling by chrome and safari
            // cause codemirror to scroll itself without scrolling the page.
            // By intercepting diagonal scrolls during the capture phase we can prevent these from reaching
            // code mirror and manually handle the scroll action in the correct way.
            //
            // The values for wheelPixelsPerUnit (-0.7, -1/3) are taken from codemirror to match the behaviour
            // for when there is a single directional scroll.

            // These detections are taken from codemirror so they are in effect at the same time as the codemirror
            // detections for the code we want to avoid.
            var isChrome = /Chrome\//.test(window.navigator.userAgent);
            var isSafari = /Apple Computer/.test(window.navigator.vendor);

            var wheelPixelsPerUnit = 1;
            if (isChrome) {
                wheelPixelsPerUnit = -0.7;
            } else if (isSafari) {
                wheelPixelsPerUnit = -1/3;
            }

            var $window = $(window);
            (isChrome || isSafari) && self.$container[0].addEventListener('mousewheel', function(e) {
                if (e.wheelDeltaX && e.wheelDeltaY) {
                    // do the biggest action X or Y only not both.
                    if(Math.abs(e.wheelDeltaX) > Math.abs(e.wheelDeltaY)) {
                        // scroll the editor horizontal
                        var scrollInfo = editorScrolling.scrollSizing();
                        editorScrolling.scroll(scrollInfo.left - e.wheelDeltaX * wheelPixelsPerUnit, null);
                    } else {
                        // scroll the page vertical
                        $window.scrollTop($window.scrollTop() + e.wheelDeltaY * wheelPixelsPerUnit);
                    }
                    // prevent the events from reaching codemirror
                    e.stopImmediatePropagation();
                    e.preventDefault();
                }
            }, true);
        }).fail(function(err) {
            self._scrollToSourcePosition = function(x, y) {
                // scroll the window down to the editor first.
                var editorOffset = $(editor.getWrapperElement()).offset();
                window.scrollTo(window.scrollX, editorOffset.top);

                editorScrolling.scroll(x, y);
            };
        });
    };

    /**
     * Get all the lines from an array of internal line datas and create a newline separated string.
     *
     * @param {Array.LineInfo} lineInfos
     * @returns {string}
     * @protected
     */
    DiffView._combineTexts = function(lineInfos) {
        return _.chain(lineInfos).pluck('line').pluck('line').value().join('\n');
    };

    /**
     * Subclasses should call this to create CodeMirror editors.
     * @param options
     * @param $container jQuery element to attach the editor to
     * @returns {CodeMirror}
     * @protected
     */
    DiffView.prototype._createEditor = function(options, $container) {
        // add default gutters
        this._registerGutter({name: 'CodeMirror-linewidget', weight: 0});
        this._registerGutter({name: 'line-number-marker', weight: 1000});

        var gutters = (options.gutterProvider && options.gutterProvider()) || this.getGutters();
        options = $.extend({
            value : '' // initialize small so we can do everything at once in the operation.
        }, options, {
            gutters : _.pluck(gutters, 'name')
        });

        var editorContainer;
        if ($container && $container.length) {
            editorContainer = $container[0];
        } else {
            editorContainer = this.$container[0];
        }

        return createEditor(editorContainer, options);
    };

    DiffView.prototype._getLineClasses = getLineClasses;

    /**
     * Begin a request for content to be added to the view.
     *
     * @param {Object} diff - diff object shaped like our REST models
     * @param {string} changeType - INITIAL or INSERT
     * @param {*} ...args - additional arguments are passed into _acceptDiff and _populateHandles
     * @protected
     */
    DiffView.prototype._modifyDiff = function(diff, changeType/*, args*/) {
        var args = [].slice.call(arguments, 2);
        var self = this;

        // don't modify our inputs
        diff = $.extend(true, {}, diff);

        // keep track of segments for segment nav purposes.
        this._allSegments = mergeSegments(this._allSegments || [], getSegmentsFromDiff(diff));

        var lineInfos = asLineInfos(diff, { relevantContextLines: this.options.relevantContextLines });
        _.forEach(lineInfos, function(internalLineData) {
            self._internalLines[internalLineData.line.lineType][internalLineData.line.lineNumber] = internalLineData;
        });

        function editorOperation() {
            var modificationPromise = this._acceptModification.apply(this, [diff, lineInfos, changeType].concat(args));
            // acceptModification populates the handles on each lineInfo.
            // So now that that's done, we can freeze everything safely.
            // The CodeMirror line handle has access to DOM elements which we don't really want to freeze, and doing
            // so causes errors.
            // So we only shallow freeze the handles.
            _.chain(lineInfos).pluck('handles').values().flatten().each(obj.freeze);
            obj.deepFreeze(lineInfos, !'refreezeFrozen');

            /**
             * A content-changed event will be triggered when content in the editor is updated/changed
             * The following object will be passed along:
             *
             * @typedef {Object} ContentChange
             * @property {ContentChangeType} type
             * @property {Object} diff matches REST diff object
             * @property {PullRequest} [pullrequest] the pull request associated to this diff, if any.
             * @property {FileChange} fileChange a file change object describing the change at a file level.
             * @property {function(function(LineInfo))} eachLine executes a function for each line in the change, passing through a {@link LineInfo}
             */
            var change = obj.freeze({
                type: changeType,
                diff: diff,
                linesAdded: lineInfos.length,
                pullRequest: this._pr,
                fileChange: this.options.fileChange,
                view: this.api(),
                eachLine : function(fn) {
                    var map = performance.frameBatchedMap(fn, {
                        min : 500,
                        initial : 200 // just enough to render the first screen.
                    }, self.operation.bind(self));

                    var deferred = map(lineInfos);
                    self.on('destroy', deferred.reject.bind(deferred));
                    return deferred.promise();
                }
            });

            // hack so classes are added synchronously.
            // We need classes on our elements upfront because they affect sizing of the lines and can get things out of
            // whack if they aren't immediately present.
            // They are also much faster to add than the line numbers are.
            addDiffClasses(this, {
                type : changeType,
                eachLine : function(fn) {
                    _.forEach(lineInfos, fn);
                    return $.Deferred().resolve();
                }
            });

            return modificationPromise.done(function() {
                self.trigger('change', change);

                if (changeType === 'INITIAL') {
                    self.trigger('load', change);
                }
            });
        }

        return this.operation(_.bind(editorOperation, this));
    };

    /**
     * Handles the animation applied to a number of lines across potentially multiple editors/handles (eg. side-by-side).
     *
     * @param {string} className
     * @param {number} ms
     * @param {{editor: {CodeMirror}, handles: StashLineHandle[]}} editorsAndHandles
     */
    function addAnimationLineClass(className, ms, editorsAndHandles) {
        // Cancel previous animations (if any)
        if (!this._currFocusPromises) this._currFocusPromises = {};
        this._currFocusPromises[className] && _.invoke(this._currFocusPromises[className], 'abort');

        this._currFocusPromises[className] = _.map(editorsAndHandles, function(args) {
            var handles = args.handles, editor = args.editor;
            if (!handles || !handles.length) {
                return $.Deferred().reject().promise({
                    abort: $.noop
                });
            }

            function toggleClasses(add) {
                editor.operation(function () {
                    _.forEach(handles, function (handle) {
                        editor[add ? 'addLineClass' : 'removeLineClass'](handle._handle, 'wrap', className);
                    });
                });
            }

            var aborted = false;
            var deferred = $.Deferred();

            var addHighlight = _.once(function () {
                editor.off('scroll', addHighlight);

                if (aborted) {
                    return;
                }
                toggleClasses(true);
                deferred.always(toggleClasses.bind(null, false));

                setTimeout(deferred.resolve.bind(deferred), ms);
            });

            // ensure the scroll happens first.
            // But at the edge of the document, no scroll will occur, so force it 100ms
            // later if it hasn't happened.
            editor.on('scroll', addHighlight);
            setTimeout(addHighlight, 100);

            return deferred.promise({
                abort: function () {
                    aborted = true;
                    deferred.resolve();
                }
            });
        });
    }

    // diff-view-animation.less has an animation lasting for @segmentNavAnimationDuration.
    // Make sure to update that variable if you change the time here.
    var segmentNavAnimationDuration = 1000;

    DiffView.prototype._markLinesFocused = _.partial(addAnimationLineClass, 'line-focused', segmentNavAnimationDuration);
    DiffView.prototype._addAnimationLineClass = addAnimationLineClass;

    return DiffView;
}); // closing diff-view define() call
}()); // closing outer closure
